/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common-apiext
 * 项目描述：工具
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.apiext;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.FieldPosition;
import java.text.Format;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.ScriptException;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import com.google.common.base.CaseFormat;

import net.wicp.tams.common.Conf;
import net.wicp.tams.common.callback.IValueEncoder;
import net.wicp.tams.common.callback.impl.valueencoder.ValueEncoderMap;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectException;

/***
 * 字符串辅助类
 * 
 * @author andy.zhou
 *
 */
public abstract class StringUtil {

	public static String convertStr(CaseFormat ori, CaseFormat dest, String oristr) {
		return ori.to(dest, oristr);
	}

	public static String convertStr(String oristr) {
		return convertStr(CaseFormat.valueOf(Conf.get("common.apiext.string.format.ori")),
				CaseFormat.valueOf(Conf.get("common.apiext.string.format.dest")), oristr);
	}

	public static String buildRule(String patter) {
		String patterTrue = patter.replace("^", "").replace("$", "").replace(",", "|");
//		if("*".equals(patter)) {
//			patterTrue=StrPattern.no.getCode();
//		}
		if (patterTrue.endsWith("_")) {
			return String.format("^%s[0-9]*$", patterTrue);
		} else {
			return String.format("^%s$", patterTrue);
		}
	}

	/***
	 * 把值散列到partitions个分区中。
	 * 
	 * @param value      要散列的值，可以是int/long/string
	 * @param partitions 要散列到几个分区
	 * @return
	 */
	public static int partition(String value, int partitions) {
		long valueL = 0;
		if (StringUtil.isNull(value)) {// 防止第1列或分库分表键的值为空值的情况
			return 0;
		}
		try {
			valueL = Long.parseLong(value);
		} catch (Exception e) {
			valueL = MurmurHash3.murmurhash3_x86_32(value, 0, value.length(), 25342);// 种子25342
																						// //有可能产生负数如"f9ce4495-2b42-4839-8d9a-3ba03c7c1ce8"
		}
		return Math.abs((int) valueL % partitions);
	}

	/***
	 * string转为InputStream
	 * 
	 * @param oristring
	 * @return
	 */
	public static InputStream convertInputStream(String oristring) {
		InputStream in_withcode;
		try {
			in_withcode = convertInputStream(oristring.getBytes("UTF-8"));
			return in_withcode;
		} catch (UnsupportedEncodingException e) {
			return null;
		}
	}

	public static InputStream convertInputStream(byte[] oriByteAry) {
		InputStream in_withcode = new ByteArrayInputStream(oriByteAry);
		return in_withcode;
	}

	/****
	 * <p>
	 * 判断字符串是否为空值
	 * </p>
	 *
	 * <pre>
	 * StringUtil.isNull(true,"null")      = true
	 * StringUtil.isNull(true,"Null")      = true
	 * StringUtil.isNull(true,"   ")       = true
	 * StringUtil.isNull(true,"bob")      = false
	 * </pre>
	 * 
	 * @param NullIsTrue 出现"null"值是否判断为true
	 * @param inputObj   要判断的字符串
	 * @return true是空值，false不是空值
	 */
	public static boolean isNull(boolean NullIsTrue, Object inputObj) {
		if (inputObj == null) {
			return true;
		}
		String inputstr = trimSpace(String.valueOf(inputObj));
		if (NullIsTrue) {
			return inputstr.equalsIgnoreCase("null") || StringUtils.isBlank(inputstr);
		} else {
			return StringUtils.isBlank(inputstr);
		}
	}

	/****
	 * 判断对象是否为空"null"字符串也为空
	 * 
	 * @param inputObj 要判断的对象
	 * @return true:是空值 false：不是
	 */
	public static boolean isNull(Object inputObj) {
		return isNull(true, inputObj);
	}

	/****
	 * 判断对象是否不为空，"null"字符串也为空
	 * 
	 * @param inputObj 要判断的对象
	 * @return true:不是空 false：是
	 */
	public static boolean isNotNull(Object inputObj) {
		return !isNull(true, inputObj);
	}

	/***
	 * 判断是否为true字符串
	 * 
	 * @param truestr
	 * @return
	 */
	public static boolean isTrueStr(String truestr) {
		if (StringUtil.isNull(truestr)) {
			return false;
		}
		if ("true".equalsIgnoreCase(StringUtil.trimSpace(truestr))) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * 把为空的字符按指定字符返回，<br>
	 * 如果inputStr[0]为null或"" 则取inputStr[1]值<br>
	 * 
	 * @param isDelNullStr 是否要排除为"null"的字符串，因为String.value(null)=="null"
	 * @param inputStr     输入要转换的数组
	 * @return String 处理后字符串
	 */
	public static String hasNull(boolean isDelNullStr, String... inputStr) {
		if (inputStr == null)
			return "";
		String returnStr;
		switch (inputStr.length) {
		case 0:
			returnStr = "";
			break;
		case 1:
			returnStr = trimSpace(inputStr[0]);
			break;
		default:
			if (inputStr[0] == null || inputStr[0].trim().length() <= 0)
				if (inputStr[1] == null) {
					return null;
				} else {
					returnStr = trimSpace(inputStr[1]);
				}
			else
				returnStr = trimSpace(inputStr[0]);
			break;
		}
		if (isDelNullStr) {
			returnStr = "null".equalsIgnoreCase(returnStr) ? "" : returnStr;
		}
		return returnStr;
	}

	public static String hasNull(String... inputStr) {
		return hasNull(true, inputStr);
	}

	/**
	 * 去掉字符串前后的空格(半角,全角空格)
	 * 
	 * @param str 要处理的字符串
	 * @return String 处理后字符串
	 */
	public static String trimSpace(String str) {
		if (str == null || str.trim().length() == 0) {
			return "";
		}

		int len = str.length();
		int first = 0;
		int last = str.length() - 1;
		char c;

		for (c = str.charAt(first); (first < len) && (c == '\u3000' || c == ' '); first++) {
			c = str.charAt(first);
		}
		if (first > 0) {
			first--;
		}
		if (len > 0) {
			c = str.charAt(last);
			while ((last > 0) && (c == '\u3000' || c == ' ')) {
				last--;
				c = str.charAt(last);
			}
			last = last + 1;
		}
		if (first >= last) {
			return "";
		}
		return ((first > 0) || (last < len)) ? str.substring(first, last) : str;
	}

	/**
	 * 把分转为元
	 * 
	 * @param fen 要转换的分数值
	 * @return String 返回元
	 * 
	 */
	public static String convertFenToYuan(String fen) {
		return convertFenToYuan(Double.parseDouble(fen));
	}

	/**
	 * 把分转换为元
	 * 
	 * @param fen 要转换的分数值
	 * @return String 返回元
	 */
	public static String convertFenToYuan(double fen) {
		String result;
		try {
			double yuan = fen / 100;
			result = NumberUtil.handleScale(yuan, 2).toString();
		} catch (Exception ex) {
			result = "0.00";
		}
		return result;
	}

	/***
	 * 获取字符串中含数字和字母的个数
	 * 
	 * @param src 输计算的字符串
	 * @return 计算的个数
	 */
	public static int sumOfNumLet(String src) {
		String figures = "0123456789";
		String letters = "abcdefghijklmnopqrstuvwxyz";
		int sum = 0;
		for (int i = 0; (src != null) && (i < src.length()); i++) {
			char ch = src.charAt(i);
			if ((figures.indexOf(ch) != -1) || (letters.indexOf(ch) != -1))
				sum++;
		}
		return sum;
	}

	/**
	 * tapestry输出变量 时要填此格式化对象
	 */
	public final static Format formatCommon = new Format() {
		private static final long serialVersionUID = -8271124584977967310L;

		@Override
		public Object parseObject(String source, ParsePosition pos) {
			return source;
		}

		@Override
		public StringBuffer format(Object obj, StringBuffer toAppendTo, FieldPosition pos) {
			return toAppendTo.append(obj);
		}
	};

	/**
	 * 首字母转成大写
	 * 
	 * @param s 要转换字符串
	 * @return 转换后字符串
	 */
	public static String toUpperCaseFirstOne(String s) {
		if (StringUtils.isBlank(s)) {
			return "";
		} else if (Character.isUpperCase(s.charAt(0))) {
			return s;
		} else {
			return (new StringBuilder()).append(Character.toUpperCase(s.charAt(0))).append(s.substring(1)).toString();
		}
	}

	/***
	 * String转为对象，handler为空就是基本数据类型
	 *
	 * @param type    对象类型
	 * @param v       对象值
	 * @param handler 转换规则
	 * 
	 * @param <T>     要转的类型
	 * @return 对象
	 */
	@SuppressWarnings("unchecked")
	public static final <T> T str2Object(Class<T> type, String v, IValueEncoder<T> handler) {
		Object param = null;
		if (handler != null)
			return handler.toValue(v);

		if (type != String.class && StringUtils.isEmpty(v)) {
			return null;
		}
		if (type == String.class)
			param = v;
		else if (type == int.class || type == Integer.class)
			param = Integer.parseInt(v);
		else if (type == long.class || type == Long.class)
			param = Long.parseLong(v);
		else if (type == byte.class || type == Byte.class)
			param = Byte.parseByte(v);
		else if (type == char.class || type == Character.class)
			param = v.charAt(0);
		else if (type == float.class || type == Float.class)
			param = Float.parseFloat(v);
		else if (type == double.class || type == Double.class)
			param = Double.parseDouble(v);
		else if (type == short.class || type == Short.class)
			param = Short.parseShort(v);
		else if (type == boolean.class || type == Boolean.class)
			param = Boolean.parseBoolean(v);
		else if (Date.class.isAssignableFrom(type))
			param = DateUtil.objToDate(v);
		else if (Enum.class.isAssignableFrom(type)) {
			try {
				param = type.getField(v).get(null);
			} catch (Exception e) {
			}
		} else if (type == BigDecimal.class) {
			try {
				param = new BigDecimal(v);
			} catch (Exception e) {
			}
		} else
			throw new IllegalArgumentException(String.format("object type '%s' not valid", type));
		return (T) param;
	}

	/***
	 * String转为基本数据类型
	 * 
	 * @param type 对象类型
	 * @param v    简单值
	 * @return 简单对象
	 */
	public static final <T> T str2Object(Class<T> type, String v) {
		return str2Object(type, v, null);
	}

	/***
	 * 产生指定个数的字符串
	 * 
	 * @param str
	 * @param num
	 * @return
	 */
	public static final String proStr(String str, int num) {
		StringBuffer buf = new StringBuffer();
		for (int i = 0; i < num; i++) {
			buf.append(str);
		}
		return buf.toString();
	}

	/****
	 * 用值设置对象的域，支持级联
	 *
	 * @param retobj    要设置的对象
	 * @param fieldName 要设置的对象的域名, a.b 表示子对象
	 * @param value     要设置的值
	 * @throws Exception 设置时异常
	 */
	public static void packObj(Object retobj, String fieldName, String value) throws Exception {
		int indexdot = fieldName.indexOf(".");
		if (indexdot <= 0) {
			Field field = null;
			try {
				field = retobj.getClass().getDeclaredField(fieldName);
			} catch (Exception e) {
				return;
			}
			Class<?> fildeClass = field.getType();
			Object valueObj = null;
			if (ReflectAssist.isInterface(fildeClass, "java.util.Map")) {
				valueObj = StringUtil.str2Object(Map.class, value, new ValueEncoderMap());
			} else if ("java.util.Date".equals(fildeClass.getName())) {
				if (StringUtil.isNotNull(value)) {
					try {
						SimpleDateFormat sdf = new SimpleDateFormat("EEE MMM dd HH:mm:ss z yyyy", java.util.Locale.US);
						valueObj = sdf.parse(value);
					} catch (Exception e) {// 偿试
						valueObj = DateUtil.objToDate(value);
					}
				}
			} else {
				valueObj = StringUtil.str2Object(fildeClass, value);
			}
			if (valueObj != null) {
				BeanUtils.setProperty(retobj, fieldName, valueObj);
			}
		} else {
			String fieldNameTrue = fieldName.substring(0, indexdot);
			Field field = null;
			try {
				field = retobj.getClass().getDeclaredField(fieldNameTrue);
			} catch (Exception e) {
				return;
			}
			Class<?> filedClass = field.getType();
			Object fieldvalue = filedClass.newInstance();
			packObj(fieldvalue, fieldName.substring(indexdot + 1), value);
			BeanUtils.setProperty(retobj, fieldNameTrue, fieldvalue);
		}
	}

	/**
	 * Convert byte[] to hex
	 * string.这里我们可以将byte转换成int，然后利用Integer.toHexString(int)来转换成16进制字符串。
	 * 
	 * @param src byte[] data
	 * @return hex string
	 */
	public static String bytesToHexString(byte[] src) {
		StringBuilder stringBuilder = new StringBuilder("");
		if (src == null || src.length <= 0) {
			return null;
		}
		for (int i = 0; i < src.length; i++) {
			int v = src[i] & 0xFF;
			String hv = Integer.toHexString(v);
			if (hv.length() < 2) {
				stringBuilder.append(0);
			}
			stringBuilder.append(hv);
		}
		return stringBuilder.toString();
	}

	/**
	 * Convert hex string to byte[]
	 * 
	 * @param hexString the hex string
	 * @return byte[]
	 */
	public static byte[] hexStringToBytes(String hexString) {
		if (hexString == null || hexString.equals("")) {
			return null;
		}
		hexString = hexString.toUpperCase();
		int length = hexString.length() / 2;
		char[] hexChars = hexString.toCharArray();
		byte[] d = new byte[length];
		for (int i = 0; i < length; i++) {
			int pos = i * 2;
			d[i] = (byte) (charToByte(hexChars[pos]) << 4 | charToByte(hexChars[pos + 1]));
		}
		return d;
	}

	/**
	 * Convert char to byte
	 * 
	 * @param c char
	 * @return byte
	 */
	public static byte charToByte(char c) {
		return (byte) "0123456789ABCDEF".indexOf(c);
	}

	public static String getFileName(String url) {
		try {
			String fileName = new URL(url).getFile();
			int endIndex = fileName.indexOf("?");
			int tempIndex = fileName.lastIndexOf("/");
			fileName = tempIndex == fileName.length() - 1 ? fileName.substring(0, tempIndex) : fileName;
			if (endIndex > 0) {
				int beginIndex = fileName.substring(0, endIndex).lastIndexOf("/");
				return fileName.substring(beginIndex + 1, endIndex);
			} else {
				int beginIndex = fileName.lastIndexOf("/");
				return fileName.substring(beginIndex + 1);
			}
		} catch (MalformedURLException e) {
			return null;
		}
	}

	public static Object expr(String expr, Map<String, Object> params) {
		// String expr = "x+y*10";
		ScriptEngineManager manager = new ScriptEngineManager();
		ScriptEngine engine = manager.getEngineByName("js");
		for (String key : params.keySet()) {
			engine.put(key, params.get(key));
		}
		Object result = null;
		try {
			result = engine.eval(expr);
		} catch (ScriptException e) {
		}
		return result;
	}

	@SuppressWarnings("unchecked")
	public static Object expr(String expr, String... params) {
		Map<String, Object> input = CollectionUtil.newMap(params);
		return expr(expr, input);
	}

	/**
	 * 产生49152-65535的端口号，只要str相同，产生的端口号不会变
	 * 
	 * @param str 原始字符串
	 * @return 生成的int
	 */
	public static int buildPort(String str) {
		int hashvalue = hashInt(str);
		return (int) (49152 + hashvalue % 16383);
	}

	/***
	 * 产生hash值
	 * 
	 * @param str 原始str
	 * @return 对应的hash值
	 */
	public static int hashInt(String str) {
		int hashvalue = MurmurHash3.murmurhash3_x86_32(str, 0, str.length(),
				Integer.parseInt(Conf.get("common.apiext.murmurhash.seed")));
		return hashvalue;
	}

	/***
	 * 16进制字符转为整形
	 * 
	 * @param ox 16进制数
	 * @return 转换后的整形
	 * @throws ProjectException 转换失败异常
	 */
	public static int OxStringtoInt(String ox) throws ProjectException {
		ox = ox.toLowerCase();
		if (ox.startsWith("0x")) {
			ox = ox.substring(2, ox.length());
		}
		int ri = 0;
		int oxlen = ox.length();
		if (oxlen > 8)
			throw new ProjectException(ExceptAll.param_error, "too lang");
		for (int i = 0; i < oxlen; i++) {
			char c = ox.charAt(i);
			int h;
			if (('0' <= c && c <= '9')) {
				h = c - 48;
			} else if (('a' <= c && c <= 'f')) {
				h = c - 87;

			} else if ('A' <= c && c <= 'F') {
				h = c - 55;
			} else {
				throw new ProjectException(ExceptAll.param_error, "not a integer ");
			}
			byte left = (byte) ((oxlen - i - 1) * 4);
			ri |= (h << left);
		}
		return ri;

	}

	/***
	 * 是不是bigdecimal
	 * 
	 * @param str
	 * @return
	 */
	public static boolean isBigDecimal(String str) {
		if (str == null || str.trim().length() == 0) {
			return false;
		}
		char[] chars = str.toCharArray();
		int sz = chars.length;
		int i = (chars[0] == '-') ? 1 : 0;
		if (i == sz)
			return false;

		if (chars[i] == '.')
			return false;// 除了负号，第一位不能为'小数点'

		boolean radixPoint = false;
		for (; i < sz; i++) {
			if (chars[i] == '.') {
				if (radixPoint)
					return false;
				radixPoint = true;
			} else if (!(chars[i] >= '0' && chars[i] <= '9')) {
				return false;
			}
		}
		return true;
	}

	/***
	 * k8s需要的规则：https://datatracker.ietf.org/doc/html/rfc1123
	 * 
	 * must be 63 characters or less (can be empty),
	 * 
	 * unless empty, must begin and end with an alphanumeric character
	 * ([a-z0-9A-Z]),
	 * 
	 * could contain dashes (-), underscores (_), dots (.), and alphanumerics
	 * between.
	 * 
	 * @return
	 */
	public static String formatRfc1123(String oriName) {
		String idstr = StringUtil.trimSpace(oriName).replace("_", "-").replace(".", "-");
		return idstr.length() >= 62 ? idstr.substring(0, 62) : idstr;
	}

	/**
	 * 比较版本大小
	 *
	 * 说明：支n位基础版本号+1位子版本号 示例：1.0.2大于1.0.1 , 1.0.1.1大于1.0.1
	 *
	 * @param version1 版本1
	 * @param version2 版本2
	 * @return 0:相同 1:version1大于version2 -1:version1小于version2
	 */
	public static int compareVersion(String version1, String version2) {
		if (version1.equals(version2)) {
			return 0; // 版本相同
		}
		String[] v1Array = version1.split("\\.");
		String[] v2Array = version2.split("\\.");
		int v1Len = v1Array.length;
		int v2Len = v2Array.length;
		int baseLen = 0; // 基础版本号位数（取长度小的）
		if (v1Len > v2Len) {
			baseLen = v2Len;
		} else {
			baseLen = v1Len;
		}

		for (int i = 0; i < baseLen; i++) { // 基础版本号比较
			if (v1Array[i].equals(v2Array[i])) { // 同位版本号相同
				continue; // 比较下一位
			} else {
				return Integer.parseInt(v1Array[i]) > Integer.parseInt(v2Array[i]) ? 1 : -1;
			}
		}
		// 基础版本相同，再比较子版本号
		if (v1Len != v2Len) {
			return v1Len > v2Len ? 1 : -1;
		} else {
			// 基础版本相同，无子版本号
			return 0;
		}
	}

	/***
	 * 把字符串的空格全部替换为指定的字符串
	 * 
	 * @param ori      原字符
	 * @param newStr   要替换的字符
	 * @param distinct 是否去重
	 * @return
	 */
	public static String blankToChart(String ori, String newStr, boolean distinct) {
		String oriTrue = trimSpace(ori);
		if (!distinct) {
			return oriTrue.replace(" ", newStr);
		}
		String[] ary = oriTrue.split(" ");
		List<String> res = new ArrayList<String>();
		for (int i = 0; i < ary.length; i++) {
			String temp = trimSpace(ary[i]);
			if (StringUtils.isNotBlank(temp)) {
				res.add(temp);
			}
		}
		String arrayJoin = CollectionUtil.listJoin(res, newStr);
		return arrayJoin;
	}

	/***
	 * 组装url参数
	 * 
	 * @param httpurl
	 * @param params
	 * @return
	 */
	public static String packageUrlParams(String httpurl, String... params) {
		if (ArrayUtils.isEmpty(params)) {
			return httpurl;
		}
		StringBuffer buf = new StringBuffer();
		Map<String, String> paramsAnd = CollectionUtil.newMapStr(params);
		paramsAnd.forEach((k, v) -> buf.append(String.format("&%s=%s", k, v)));

		String httpurlTrue = httpurl.endsWith("/") ? httpurl.substring(0, httpurl.length() - 1) : httpurl;
		if (!httpurlTrue.contains("?")) {// 不含有?
			httpurlTrue = buf.length() > 0 ? httpurlTrue + "?" + buf.substring(1) : httpurlTrue;// 跟据参数来判断
		} else {
			if (buf.length() > 0) {
				httpurlTrue = httpurlTrue.endsWith("?") ? httpurlTrue + buf.substring(1) : httpurlTrue + buf.toString();
			}
		}
		return httpurlTrue;
	}

	/**
	 * 是否为表情
	 * 
	 * @param codePoint
	 * @return
	 */
	private static boolean isEmojiCharacter(char codePoint) {
		return (codePoint == 0x0) || (codePoint == 0x9) || (codePoint == 0xA) || (codePoint == 0xD)
				|| ((codePoint >= 0x20) && (codePoint <= 0xD7FF)) || ((codePoint >= 0xE000) && (codePoint <= 0xFFFD))
				|| ((codePoint >= 0x10000) && (codePoint <= 0x10FFFF));
	}

	/***
	 * 过滤表情符
	 * 
	 * @param source
	 * @return
	 */
	public static String filterEmoji(String source) {
		if (StringUtils.isBlank(source)) {
			return source;
		}
		StringBuilder buf = null;
		int len = source.length();
		for (int i = 0; i < len; i++) {
			char codePoint = source.charAt(i);
			if (isEmojiCharacter(codePoint)) {
				if (buf == null) {
					buf = new StringBuilder(source.length());
				}
				buf.append(codePoint);
			}
		}
		if (buf == null) {
			return source;
		} else {
			if (buf.length() == len) {
				buf = null;
				return source;
			} else {
				return buf.toString();
			}
		}
	}

}