package net.wicp.tams.common.constant;

import java.io.File;
import java.io.InputStream;
import java.net.URL;

import org.apache.commons.lang3.ArrayUtils;

import net.wicp.tams.common.apiext.IOUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.apiext.TarUtil;
import net.wicp.tams.common.callback.IRemote;

/***
 * 文件路径寻找规则
 * 
 * @author andy.zhou
 *
 */
public enum PathType {
	web("web根目录", true), clp("classpath根目录", true), rel("相对于class的路径", true), abs("绝对路径", true), env("环境变量的地址", true),home("用户目录",true), // env:JAVA_HOME:/bin

	s3("s3网络存储", false, "net.wicp.tams.common.aws.TamsRemoteImpl"), oss("oss存储", false);

	private final org.slf4j.Logger log = org.slf4j.LoggerFactory.getLogger(PathType.class);

	private final String desc;
	// 是本机存储
	private final boolean isLocal;

	private final IRemote remote;

	public IRemote getRemote() {
		return remote;
	}

	private PathType(String desc, boolean isLocal, String remoteClass) {
		this.desc = desc;
		this.isLocal = isLocal;
		if (StringUtil.isNotNull(remoteClass)) {
			IRemote remote3 = null;
			try {
				remote3 = (IRemote) Class.forName(remoteClass).newInstance();
			} catch (Exception e) {
				log.warn("the class [{}] no find", remoteClass);
				// throw new RuntimeException(e);
			}
			this.remote = remote3;
		} else {
			this.remote = null;
		}
	}

	private PathType(String desc, boolean isLocal) {
		this(desc, isLocal, null);

	}

	/***
	 * 解析PathType类型的路径得到实际路径，如 web:/aaaa/bbb
	 * 
	 * @param inputpath PathType类型的路径
	 * @param splitchar 类型分隔符
	 * @return 真实的地址
	 */
	private static String getPath(String inputpath, String splitchar, boolean isSaveJar) {
		if (StringUtil.isNull(inputpath)) {
			return "";
		}
		splitchar = StringUtil.hasNull(splitchar, ":");
		String[] inputpathary = inputpath.split(splitchar);
		if (ArrayUtils.isEmpty(inputpathary)) {
			return "";
		}
		String PathTypeStr = inputpathary.length > 1 ? inputpathary[0] : "";
		PathType pathType = getByName(PathTypeStr);
		String inputpathStr = inputpathary.length > 1 ? inputpathary[1] : inputpathary[0];
		if (pathType == PathType.abs) {// 绝对值不拆
			inputpathStr = inputpath.replace("abs:", "");
		}
		inputpathStr = inputpathStr.replace("//", "/");// 支持"//"模式

		if (!pathType.isLocal) {// 远程存储
			if (pathType.getRemote() == null) {
				throw new RuntimeException("没有指定的实现类，s3引入common-aws包");
			}
			String path = IOUtil.mergeFolderAndFilePath(System.getProperty("user.home"), ".tams");
			String dirName = inputpathStr.substring(inputpathStr.lastIndexOf("/") + 1).replace(".tar", "");
			String lastDir = IOUtil.mergeFolderAndFilePath(path, dirName);
			File lastFile = new File(lastDir);
			if (lastFile.exists()) {
				return lastDir;
			}
			try {
				// 保存到本机
				if (isSaveJar) {
					pathType.getRemote().findStream(inputpathStr, lastDir + ".tar");
					TarUtil.decompress(lastDir + ".tar", lastDir);
				} else {
					InputStream findStream = pathType.getRemote().findStream(inputpathStr);
					TarUtil.decompress(findStream, lastDir);
				}

				return lastDir;
			} catch (Exception e) {
				throw new RuntimeException("download the remote file error", e);
			}
		}
		String retstr = inputpathStr;
		switch (pathType) {
		case web:
			URL classpathurl = Thread.currentThread().getContextClassLoader().getResource("");// "/"
			File file = new File(classpathurl.getPath());
			String webrootstr = file.getParentFile().getParentFile().getPath();
			retstr = IOUtil.mergeFolderAndFilePath(webrootstr, inputpathStr);
			break;
		case clp:
			URL classpath = Thread.currentThread().getContextClassLoader().getResource("");
			retstr = IOUtil.mergeFolderAndFilePath(classpath.getPath(), inputpathStr);
			break;
		case rel:
			URL classpathrel = Thread.currentThread().getContextClassLoader().getResource(".");
			retstr = IOUtil.mergeFolderAndFilePath(classpathrel.getPath(), inputpathStr);
			break;
		case abs:
			retstr = inputpathStr;
			break;
		case  home:
			retstr = IOUtil.mergeFolderAndFilePath(System.getProperty("user.home"), inputpathStr);
			break;
		case env:
			String envpath = System.getenv(inputpathStr);
			if (inputpathary.length == 2) {
				retstr = envpath;
			} else {
				retstr = IOUtil.mergeFolderAndFilePath(envpath, inputpathary[2]);
			}
			break;
		default:
			break;
		}
		return retstr;
	}

	public static String getPath(String inputpath, boolean isSaveJar) {
		return getPath(inputpath, ":",isSaveJar);
	}

	/***
	 * 通过名字得到路径寻找类型
	 * 
	 * @param inputpath 路径寻找类型字符串
	 * @return 路径类型
	 */
	public static PathType getByName(String inputpath) {
		if (StringUtil.isNull(inputpath)) {
			return clp;
		}
		String[] inputpathary = inputpath.split(":");
		if (ArrayUtils.isEmpty(inputpathary)) {
			return clp;
		}
		String PathTypeStr = inputpathary[0];
		for (PathType ele : PathType.values()) {
			if (PathTypeStr.equalsIgnoreCase(ele.name())) {
				return ele;
			}
		}
		return clp;
	}

	public String getDesc() {
		return desc;
	}

	public boolean isLocal() {
		return isLocal;
	}
}
