package net.wicp.tams.common.apiext;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;

import org.apache.commons.codec.binary.Base64;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.constant.DES;

/**
 * 
 * @author 偏锋书生
 *
 */
@Slf4j
public abstract class PwdUtil {

	private static Cipher cipher;
	static {
		// 生成Cipher对象,指定其支持的AES/CBC/PKCS7Padding算法
		try {
			// Security.addProvider(new BouncyCastleProvider());
			cipher = Cipher.getInstance("AES/CBC/PKCS5Padding");
		} catch (NoSuchAlgorithmException e) {
			log.error("No Such Algorithm", e);
		} catch (NoSuchPaddingException e) {
			log.error("No Such Padding", e);
		}
	}

	/***
	 * Base64压缩
	 * 
	 * @param key
	 *            要压缩的文字
	 * @return 压缩后文字
	 */
	public final static String StringToBase64(String key) {
		return new String(Base64.encodeBase64(StringUtil.hexStringToBytes(key)));
	}

	/***
	 * 把二进制转为string
	 * 
	 * @param src
	 *            要转换的源数组
	 * @return 转换后的字符串
	 */
	public final static String base64FromBin(byte[] src) {
		Base64 base64 = new Base64();
		try {
			return new String(base64.encode(src), "UTF-8");
		} catch (UnsupportedEncodingException e) {
			log.error("", e);
			return null;
		}
	}

	/***
	 * Base64解压
	 * 
	 * @param key
	 *            要解压的base码
	 * @return 解压后原文
	 */
	public final static String Base64ToString(String key) {
		return StringUtil.bytesToHexString(Base64.decodeBase64(key.getBytes()));
	}

	/***
	 * 产生MD5编码
	 * 
	 * @param s
	 *            要编码的字符串
	 * @return MD5编码
	 */
	public final static String MD5(String s) {
		char hexDigits[] = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F' };
		try {
			byte[] strTemp = s.getBytes();
			MessageDigest mdTemp = MessageDigest.getInstance("MD5");
			mdTemp.update(strTemp);
			byte[] md = mdTemp.digest();
			int j = md.length;
			char str[] = new char[j * 2];
			int k = 0;
			for (int i = 0; i < j; i++) {
				byte byte0 = md[i];
				str[k++] = hexDigits[byte0 >>> 4 & 0xf];
				str[k++] = hexDigits[byte0 & 0xf];
			}
			return new String(str);
		} catch (Exception e) {
			log.error("MD5加密失败");
			return null;
		}
	}

	/***
	 * 计算24位长的密码byte值,首先对原始密钥做MD5算hash值，再用前8位数据对应补全后8位
	 * 
	 * @param strKey
	 * @return 加密字节码
	 * @throws Exception
	 *             异常
	 */
	private static byte[] GetKeyBytes(String strKey) throws Exception {
		if (null == strKey || strKey.length() < 1)
			throw new Exception("key is null or empty!");
		java.security.MessageDigest alg = java.security.MessageDigest.getInstance("MD5");
		alg.update(strKey.getBytes(Conf.get("common.apiext.encode")));
		byte[] bkey = alg.digest();
		int start = bkey.length;
		byte[] bkey24 = new byte[24];
		for (int i = 0; i < start; i++) {
			bkey24[i] = bkey[i];
		}
		for (int i = start; i < 24; i++) {// 为了与.net16位key兼容
			bkey24[i] = bkey[i - start];
		}
		return bkey24;
	}

	/****
	 * 加密数据
	 * 
	 * @param keybyte
	 *            24个字节的加密密钥
	 * @param src
	 *            被加密的数据缓冲区（源）
	 * @param des
	 *            加密算法
	 * @return 加密后的字节码
	 */
	private static byte[] encryptMode(byte[] keybyte, byte[] src, DES des) {

		try {
			// 生成密钥
			SecretKey deskey = new SecretKeySpec(keybyte, des.name()); // 加密
			Cipher c1 = Cipher.getInstance(des.name());
			c1.init(Cipher.ENCRYPT_MODE, deskey);
			return c1.doFinal(src);
		} catch (Exception e1) {
			log.error("加密数据失败", e1);
		}
		return null;
	}

	private static byte[] decryptMode(byte[] keybyte, byte[] src, DES des) {
		try {
			SecretKey deskey = new SecretKeySpec(keybyte, des.name());// 生成密钥
			Cipher c1 = Cipher.getInstance(des.name());// 解密
			c1.init(Cipher.DECRYPT_MODE, deskey);
			return c1.doFinal(src);
		} catch (Exception e) {
			log.error("解密数据失败", e);
		}
		return null;
	}

	/***
	 * 通过key 用3DES加密
	 * 
	 * @param origValue
	 *            要加密的数数据
	 * @param key
	 *            加密用的key
	 * @return 加密后的密文
	 */
	public static String Encrypt3DES(String origValue, String key) {
		String str = null;
		try {
			str = Base64.encodeBase64String(
					encryptMode(GetKeyBytes(key), origValue.getBytes(Conf.get("common.apiext.encode")), DES.DESede));
		} catch (Exception e) {
			log.error("加密失败", e);
		}
		return str;
	}

	/***
	 * 用配置文件的配置做为key加密
	 * 
	 * @param origValue
	 *            要加密的数数据
	 * @return 加密后的数据
	 */
	public static String Encrypt3DES(String origValue) {
		return Encrypt3DES(origValue, Conf.get("common.apiext.threedes.key"));
	}

	/***
	 * 通过key 用3DES加密,是Encrypt3DES的逆过程
	 * 
	 * @param value
	 *            要解密的密文
	 * @param key
	 *            要解密的key值
	 * @return 解密后的原值
	 */
	public static String Decrypt3DES(String value, String key) {
		try {
			byte[] b = decryptMode(GetKeyBytes(key), new Base64().decode(value), DES.DESede);
			return new String(b);
		} catch (Exception e) {
			log.error("解密失败", e);
			return null;
		}
	}

	/***
	 * 用配置文件的配置做为key解密
	 * 
	 * @param value
	 *            要解密的密文
	 * @return 解密后的源文
	 */
	public static String Decrypt3DES(String value) {
		return Decrypt3DES(value, Conf.get("common.apiext.threedes.key"));
	}

	/***
	 * 把base64字符转原文的字节码
	 * 
	 * @param p_Str
	 *            base码
	 * @return 原文的二进制形式
	 * @throws IOException
	 *             转码异常
	 */
	public static byte[] fromBase64(String p_Str) throws IOException {

		return Base64.decodeBase64(p_Str);
	}

	/***
	 * 把原文件转为base64编码
	 * 
	 * @param bPic
	 *            文件的byte字节
	 * @return Base编码
	 */
	public static String toBase64(byte[] bPic) {
		return Base64.encodeBase64String(bPic);
	}

	// ///////////////////////////////////////////////////////////////////////////////////////////////////////////
	public static byte[] md5(byte data[]) {
		return getMd5Digest().digest(data);
	}

	public static byte[] md5(InputStream is) throws IOException {
		return digest(getMd5Digest(), is);
	}

	public static byte[] sha(byte data[]) {
		return getShaDigest().digest(data);
	}

	public static byte[] sha(InputStream is) throws IOException {
		return digest(getShaDigest(), is);
	}

	/***
	 * 用Aes加密
	 * 
	 * @param data
	 *            要加密的原始数据
	 * @return 加密后的IV和密文,分别String[0] 密文 String[1] IV
	 */
	public static String[] encryptByAes(String data) {
		String[] result = new String[2];
		Key sKeySpec = null;
		try {
			sKeySpec = new SecretKeySpec(Base64.decodeBase64(Conf.get("common.apiext.aes.key").getBytes("UTF-8")),
					"AES");
			// 根据密钥，对Cipher对象进行初始化，ENCRYPT_MODE表示加密模式
			cipher.init(Cipher.ENCRYPT_MODE, sKeySpec);
			result[1] = Base64.encodeBase64String(cipher.getIV());
			result[0] = Base64.encodeBase64String(cipher.doFinal(data.getBytes("UTF-8")));
		} catch (IllegalBlockSizeException e) {
			log.error("IllegalBlockSize", e);
		} catch (BadPaddingException e) {
			log.error("BadPadding", e);
		} catch (UnsupportedEncodingException e) {
			log.error("UnsupportedEncoding", e);
		} catch (InvalidKeyException e) {
			log.error("InvalidKey", e);
		}
		return result;
	}

	/**
	 * 用Aes解密
	 * 
	 * @param data
	 *            密文数据
	 * @param iv
	 *            解密IV
	 * @return 解密后的原始数据
	 */
	public static String decryptByAes(String data, String iv) {
		String result = "";
		Key sKeySpec = null;
		try {
			sKeySpec = new SecretKeySpec(Base64.decodeBase64(Conf.get("common.apiext.aes.key").getBytes("UTF-8")),
					"AES");
			// 根据密钥，对Cipher对象进行初始化，DECRYPT_MODE表示加密模式
			cipher.init(Cipher.DECRYPT_MODE, sKeySpec, new IvParameterSpec(Base64.decodeBase64(iv.getBytes("UTF-8"))));
			result = new String(cipher.doFinal(Base64.decodeBase64(data.getBytes("UTF-8"))));
		} catch (InvalidKeyException e) {
			log.error("InvalidKey", e);
		} catch (InvalidAlgorithmParameterException e) {
			log.error("InvalidAlgorithmParameter", e);
		} catch (IllegalBlockSizeException e) {
			log.error("IllegalBlockSize", e);
		} catch (BadPaddingException e) {
			log.error("BadPadding", e);
		} catch (UnsupportedEncodingException e) {
			log.error("UnsupportedEncoding", e);
		}
		return result;
	}

	private static MessageDigest getMd5Digest() {
		return getDigest("MD5");
	}

	private static MessageDigest getShaDigest() {
		return getDigest("SHA");
	}

	private static MessageDigest getDigest(String algorithm) {
		try {
			return MessageDigest.getInstance(algorithm);
		} catch (NoSuchAlgorithmException e) {
			throw new RuntimeException(e);
		}
	}

	private static byte[] digest(MessageDigest md, InputStream is) throws IOException {
		final byte buffer[] = new byte[4096];
		for (int count = is.read(buffer); count > 0; count = is.read(buffer)) {
			md.update(buffer, 0, count);
		}
		return md.digest();
	}
}
