package net.wicp.tams.common;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import com.alibaba.fastjson.JSONObject;

import net.wicp.tams.common.I18N.MessageUtils;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.IExcept;
import net.wicp.tams.common.exception.ProjectException;
import net.wicp.tams.common.exception.bean.ParamInfoBean;

/**
 * 操作返回的对象，exceptAll是必须要有的对象
 * 
 * @author 偏锋书生
 */
public class Result implements java.io.Serializable {
	private static final long serialVersionUID = 1L;
	private boolean result;// true:正确,false:业务出错
	private String message;
	private IExcept except;// 异常编码
	private Object[] retObjs;// 操作成功后，如果想带一些返回值在此设置

	/**
	 * 由异常来构建返回结果
	 * 
	 * @param opeExcept
	 *            操作异常
	 */
	public Result(ProjectException opeExcept) {
		this.result = false;
		if (opeExcept == null) {
			throw new IllegalArgumentException();
		}
		this.except = opeExcept.getExcept();
		this.message=opeExcept.getMessage();
	}

	public Result(IExcept except) {
		if (except == ExceptAll.no)
			this.result = true;
		else
			this.result = false;
		this.except = except;
	}

	private Result(boolean result) {
		this.result = result;
	}

	private Result(String message) {
		this.result = false;
		this.except = ExceptAll.project_undefined;
		this.message = message;
	}

	/****
	 * 得到成功的返回结果，单例
	 * 
	 * @return 成功
	 */
	public static Result getSuc() {
		return getSuc(null);
	}

	/**
	 * {"message":"操作成功","except":"no","suc":true}
	 * 
	 * @param jsonStr
	 *            原始json串
	 * @return 从json得到结果
	 */
	public static Result getFromJson(String jsonStr) {
		JSONObject jsonobj = JSONObject.parseObject(jsonStr);
		boolean result = jsonobj.getBoolean("suc");
		Result retobj = new Result(result);
		retobj.setMessage(jsonobj.getString("message"));
		return retobj;
	}

	public static Result getSuc(String sucInfo) {
		sucInfo = StringUtil.isNull(sucInfo) ? MessageUtils.getInstance().getString("common.hint.success") : sucInfo;
		Result suc = new Result(true);
		suc.setMessage(sucInfo);
		suc.except = ExceptAll.no;
		return suc;
	}

	/***
	 * 得到错误的结果
	 * 
	 * @param errmsg
	 *            错误信息
	 * @return 错误
	 */
	public static Result getError(String errmsg) {
		return new Result(errmsg);
	}

	/**
	 * 得到错误编码，如果正确则为“no”
	 * 
	 * @return 结果的异常
	 */
	public IExcept getExcept() {
		return this.except;
	}

	public boolean isSuc() {
		return result;
	}

	/***
	 * 返回的对象，注意读与取的先后顺序
	 * 
	 * @param index
	 *            对象的索引
	 * @return 返回的对象
	 */
	public Object retObjs(int index) {
		if (ArrayUtils.isEmpty(retObjs))
			return null;
		return retObjs[index];
	}

	public Object[] retObjs() {
		return this.retObjs;
	}

	/***
	 * 设置返回值
	 * 
	 * @param retObjs
	 *            需要返回的多个对象
	 * @return 返回的结果
	 */
	@SuppressWarnings("unchecked")
	public <T extends Object> Result setRetObjs(T... retObjs) {
		this.retObjs = retObjs;
		return this;
	}

	public void setMessage(String message) {
		this.message = message;
	}

	// ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	public String getMessage(ParamInfoBean errBean) {
		if (StringUtils.isNotBlank(this.message)) {
			return this.message;
		}
		return this.except.getErrMsg(errBean);
	}

	public String getMessage() {
		if (StringUtils.isNotBlank(this.message)) {
			return this.message;
		}
		return this.except.getErrMsg();
	}

	// ///////////////////////////////////////////////////////////////////////////////////////////////////////////////
	private String jsonstrformate = "{\"result\":%s,\"value\":\"%s\",\"code\":\"%s\",\"msg\":\"%s\"}";

	public JSONObject retJsonObj(ParamInfoBean errBean) {
		String message = getMessage(errBean);
		return JSONObject.parseObject(String.format(jsonstrformate, result ? 1 : 0, this.except.getErrorValue(),
				this.except.getErrorCode(), message));
	}

	public JSONObject retJsonObj() {
		String message = getMessage();
		JSONObject retjson = JSONObject.parseObject(String.format(jsonstrformate, result ? 1 : 0,
				this.except.getErrorValue(), this.except.getErrorCode(), message.replace("\"", "\'").replace("\\", "\\\\")));  //把"转为'  //eg: 正则表达式会出现"\." 变为 “\\.”
		if (ArrayUtils.isNotEmpty(retObjs)) {
			retjson.put("retObjs", retObjs);
		}
		return retjson;
	}

}
