/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.ali.oss;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.io.FileUtils;

import com.aliyun.oss.OSS;
import com.aliyun.oss.OSSClientBuilder;
import com.aliyun.oss.model.ListObjectsV2Request;
import com.aliyun.oss.model.ListObjectsV2Result;
import com.aliyun.oss.model.OSSObject;
import com.aliyun.oss.model.OSSObjectSummary;
import com.aliyun.oss.model.PutObjectResult;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.Result;
import net.wicp.tams.common.apiext.IOUtil;
import net.wicp.tams.common.apiext.StringUtil;

@Slf4j
public class OssClientTams {
	private static volatile OssClientTams inst;

	private final OSS client;

	public OSS getClient() {
		return client;
	}

	private OssClientTams() {
		OSS client = new OSSClientBuilder().build(Conf.get("common.ali.profile.endpoint"),
				Conf.get("common.ali.profile.accessKey"), Conf.get("common.ali.profile.secretKey"));
		this.client = client;
	}

	public static OssClientTams getInst() {
		if (inst == null) {
			synchronized (OssClientTams.class) {
				if (inst == null) {
					inst = new OssClientTams();
				}
			}
		}
		return inst;
	}

	public OSSObject getObject(String key) {
		OSSObject ossObject = this.client.getObject(Conf.get("common.ali.oss.default.bucketName"), key);
		return ossObject;
	}

	/***
	 * 保存key到指定目录中
	 * 
	 * @param key           oss的key
	 * @param parentDirPath 要保存的目录
	 * @return
	 */
	public Result saveFileByKey(String key, String parentDirPath) {
		OSSObject ossObject = this.getObject(key);
		try {
			String filePath = IOUtil.mergeFolderAndFilePath(parentDirPath, key.substring(key.lastIndexOf("/") + 1));
			FileUtils.copyInputStreamToFile(ossObject.getObjectContent(), new File(filePath));
		} catch (Exception e) {
			log.error("保存文件失败", e);
			return Result.getError(e.getMessage());
		}
		return Result.getSuc();
	}

	/***
	 * 存文件
	 * 
	 * @param key  路径
	 * @param file 要存的文件
	 * @return 版本号
	 */
	public String putObject(String key, File file) {
		PutObjectResult putObject = this.client.putObject(Conf.get("common.ali.oss.default.bucketName"), key, file);
		return putObject.getVersionId();
	}

	/***
	 * 把bucketName和key组成一个key 如：lc-bigdata/doris/check_system.sh
	 * 表示：lc-bigdata的bucketName的doris/check_system.sh路径
	 * 
	 * @param bucketNameAndKey 带有bucketname的key
	 * @return
	 */
	public OSSObject getObjectByBucketNameKey(String bucketNameAndKey) {
		String pathtrue = bucketNameAndKey.charAt(0) == '/' ? bucketNameAndKey.substring(1) : bucketNameAndKey;
		int firstIndexOf = pathtrue.indexOf("/");
		OSSObject object = this.client.getObject(pathtrue.substring(0, firstIndexOf),
				pathtrue.substring(firstIndexOf + 1));
		return object;
	}

	public List<String> getFileKeyByDir(String dirPath) {
		List<String> retlist = new ArrayList<>();
		getFileKeyByDir(dirPath, null, retlist);
		return retlist;
	}

	private void getFileKeyByDir(String dirPath, String continuationToken, List<String> retlist) {
		ListObjectsV2Request listObjectsRequest = new ListObjectsV2Request(
				Conf.get("common.ali.oss.default.bucketName"));
		listObjectsRequest.setDelimiter("/");
		if (StringUtil.isNotNull(continuationToken)) {
			listObjectsRequest.setContinuationToken(continuationToken);
		}
		// 列出主目录下的所有文件夹。
		listObjectsRequest.setPrefix(dirPath);
		ListObjectsV2Result list = this.client.listObjectsV2(listObjectsRequest);
//		List<String> commonPrefixes = list.getCommonPrefixes();
		List<OSSObjectSummary> objectSummaries = list.getObjectSummaries();
		for (OSSObjectSummary ossObjectSummary : objectSummaries) {
			retlist.add(ossObjectSummary.getKey());
		}
		if (StringUtil.isNotNull(list.getNextContinuationToken())) {
			getFileKeyByDir(dirPath, list.getNextContinuationToken(), retlist);
		}
	}

	// 删除这个目录
	public void delKeyByDir(String dirPath) {
		List<String> allKey = getFileKeyByDir(dirPath);
		delKeys(allKey.toArray(new String[allKey.size()]));
	}

	// 删除指定的key
	public void delKeys(String... keys) {
		for (String key : keys) {
			this.client.deleteObject(Conf.get("common.ali.oss.default.bucketName"), key);
		}
	}

}
