package net.wicp.tams.plugin.cache;

import java.io.File;
import java.util.List;

import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtMethod;
import javassist.Modifier;
import javassist.bytecode.CodeAttribute;
import javassist.bytecode.LocalVariableAttribute;
import net.wicp.tams.commons.apiext.XmlUtil;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.configuration.XMLConfiguration;
import org.apache.commons.configuration.tree.ConfigurationNode;
import org.apache.commons.configuration.tree.xpath.XPathExpressionEngine;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;

/**
 * 把缓存放到二进制编码中
 *
 * @goal addcache
 * @phase prepare-package *
 */
public class Addcache extends AbstractMojo {
	public static final String CACHES = "caches";
	public static final String CACHE = "cache";
	public static final String METHOD = "method";
	public static final String GET = "get";
	public static final String PUT = "put";

	/**
	 * 需要加入缓存的配置文件 project.build.resources
	 *
	 * @parameter expression="${project.build.outputDirectory}/cache.xml"
	 * @required
	 */
	private File config;

	/**
	 * class产生的路径
	 *
	 * @parameter expression="${project.build.outputDirectory}"
	 * @required
	 */
	private String classroot;

	private String[] excludeType = { "String", "Integer" };

	@Override
	public void execute() throws MojoExecutionException, MojoFailureException {
		getLog().info("------------------------加强缓存---begin---------------------");
		XMLConfiguration xml = null;
		try {
			xml = new XMLConfiguration(config);
			xml.setEncoding("UTF-8");
			xml.setExpressionEngine(new XPathExpressionEngine());// 设置Xpath解析器
		} catch (Exception e) {
			throw new MojoFailureException("读缓存配置错误,默认文件为cache.xml.");
		}
		ConfigurationNode node = xml.getRoot();
		if (StringUtils.isBlank(node.getName()) || "null".equals(node.getName())) {
			getLog().info("------------------------加强缓存---end------没有找到合适的配置文件-----------");
			return;
		}

		if (!CACHES.equalsIgnoreCase(node.getName())) {
			throw new MojoFailureException(String.format("配置文件错误:根必须为%s", CACHES));
		}
		List<ConfigurationNode> cachelist = node.getChildren(CACHE);
		if (CollectionUtils.isEmpty(cachelist)) {
			return;
		}
		for (int i = 0; i < cachelist.size(); i++) {
			ConfigurationNode cacheNode = cachelist.get(i);
			boolean checkresult = checkCacheMethed(cacheNode, i);
			if (!checkresult) {
				continue;
			}
			List<ConfigurationNode> methodlist = cacheNode.getChildren(METHOD);
			if (CollectionUtils.isEmpty(methodlist)) {
				continue;
			}
			for (int j = 0; j < methodlist.size(); j++) {
				ConfigurationNode methodNode = methodlist.get(j);
				String className = findValueByAttrName(methodNode, ColProperty.classname);
				String name = findValueByAttrName(methodNode, ColProperty.name);
				String key = findValueByAttrName(methodNode, ColProperty.key);
				if (StringUtils.isBlank(className) || StringUtils.isBlank(name) || StringUtils.isBlank(key)) {
					getLog().error(String.format("第%s个Cache区第%s个方法缺少必要的参数", i, j));
					continue;
				}
				alterMethod(cacheNode, methodNode);
			}
		}
		getLog().info("------------------------加强缓存---end---------------------");
	}

	private void alterMethod(ConfigurationNode cacheNode, ConfigurationNode methodNode) {
		ConfigurationNode getNode = XmlUtil.getFirstNodeByNodeName(cacheNode, GET);
		ConfigurationNode putNode = XmlUtil.getFirstNodeByNodeName(cacheNode, PUT);
		String staticnameGet = findValueByAttrName(getNode, ColProperty.staticname);
		String classnameGet = findValueByAttrName(getNode, ColProperty.classname);

		String staticnamePut = findValueByAttrName(putNode, ColProperty.staticname);
		String classnamePut = findValueByAttrName(putNode, ColProperty.classname);

		String className = findValueByAttrName(methodNode, ColProperty.classname);
		String name = findValueByAttrName(methodNode, ColProperty.name);
		String key = findValueByAttrName(methodNode, ColProperty.key);
		String expire = findValueByAttrName(methodNode, ColProperty.expire);
		String idcol = findValueByAttrName(methodNode, ColProperty.idcol);
		String param = findValueByAttrName(methodNode, ColProperty.param);

		idcol = StringUtils.isEmpty(idcol) ? "id" : idcol;
		try {
			ClassPool cp = ClassPool.getDefault();
			cp.appendClassPath(classroot);
			CtClass cc = cp.get(className);
			CtMethod m = cc.getDeclaredMethod(name);
			String retType = m.getReturnType().getName();// 方法返回类型
			CtClass[] paramTypes = m.getParameterTypes();
			if (paramTypes.length == 0) {// 没有传入参数的方法跳过
				return;
			}

			if (StringUtils.isBlank(param)) {// param为空表示查询
				String serializableTypeName = null;// 找到参数中Serializable类型参数
				for (CtClass ctClass : paramTypes) {
					if (ArrayUtils.contains(excludeType, ctClass.getSimpleName())) {// 排除的Serializable类型
						continue;
					}
					CtClass[] inteName = ctClass.getInterfaces();
					boolean finded = false;
					if (inteName != null && inteName.length > 0) {
						for (int i = 0; i < inteName.length; i++) {
							CtClass interclass = inteName[i];
							if ("java.io.Serializable".equals(interclass.getName())) {
								serializableTypeName = ctClass.getName();
								finded = true;
								break;
							}
						}
					}
					if (finded) {
						break;
					}
				}
				if (StringUtils.isBlank(serializableTypeName)) {
					serializableTypeName = retType;
				}
				// 找到参数名对应的参数
				int idPos = findParamIndex(idcol, m, paramTypes);
				if (idPos == -1) {
					getLog().info(String.format("%s.%s处理错误:没有找到参数%s", className, name, idcol));
					return;
				}
				String getKey = "String.format(\"" + key + "\",new Object[]{String.valueOf($" + idPos + ")})";
				StringBuffer buff = new StringBuffer("{");
				buff.append(String.format("Object retobj = %s.%s(%s.class,%s);", classnameGet, staticnameGet,
						serializableTypeName, getKey));
				// 设置参数1为null,在后面put缓存时会跟据它不是null来决定设置缓存
				buff.append("if(retobj!=null){$1=null; return (" + retType + ")retobj;}");
				buff.append("}");
				m.insertBefore(buff.toString());
				// 插入缓存代码
				String getIdStr = "(($r)$_).get" + idcol.substring(0, 1).toUpperCase() + idcol.substring(1) + "()";
				String proKey = "String.format(\"" + key + "\",new Object[]{ " + getIdStr + "})";

				String tempStr = "{if($_!=null && $1!=null) %s.%s($_,%s);}";
				if (StringUtils.isNotBlank(expire)) {// 带有效期
					tempStr = "{Integer expire = ($w)" + expire + "; if($_!=null && $1!=null) %s.%s($_,%s,expire);}";
				}
				m.insertAfter(String.format(tempStr, classnamePut, staticnamePut, proKey));
			} else {// 表示保存
				int paramPos = findParamIndex(param, m, paramTypes);
				if (paramPos <= 0) {
					getLog().info(String.format("%s.%s处理错误:没有找到参数%s", className, name, param));
					return;
				}
				// 插入缓存代码
				String getIdStr = "$" + paramPos + ".get" + idcol.substring(0, 1).toUpperCase() + idcol.substring(1)
						+ "()";
				String proKey = "String.format(\"" + key + "\",new Object[]{ " + getIdStr + "})";
				// String paramType=paramTypes[paramPos-1].getName();

				String tempStr = "{if($" + paramPos + "!=null) %s.%s($" + paramPos + ",%s);}";
				if (StringUtils.isNotBlank(expire)) {// 带有效期
					tempStr = "{Integer expire = ($w)" + expire + "; if($" + paramPos + "!=null) %s.%s($" + paramPos
							+ ",%s,expire);}";
				}
				m.insertAfter(String.format(tempStr, classnamePut, staticnamePut, proKey));

			}

			cc.writeFile(classroot);
			getLog().info(String.format("处理完毕:%s.%s", className, name));
		} catch (Exception e) {
			getLog().error(String.format("在加强%s的%s方法时出错:%s", className, name, String.valueOf(e)));
		}

	}

	/***
	 * 通过参数名在参数的位置
	 *
	 * @param findParam
	 * @param m
	 * @param paramTypes
	 * @return
	 */
	private int findParamIndex(String findParam, CtMethod m, CtClass[] paramTypes) {
		CodeAttribute codeAttribute = m.getMethodInfo().getCodeAttribute();
		LocalVariableAttribute attr = (LocalVariableAttribute) codeAttribute.getAttribute(LocalVariableAttribute.tag);
		int pos = Modifier.isStatic(m.getModifiers()) ? 0 : 1;
		int idPos = -1;// id所在第几个参数
		for (int j = 0; j < paramTypes.length; j++) {
			String paramName = attr.variableName(j + pos);
			if (paramName.equalsIgnoreCase(findParam)) {
				idPos = j + 1;
				break;
			}
		}
		return idPos;
	}

	private boolean checkCacheMethed(ConfigurationNode cacheNode, int i) {
		ConfigurationNode getNode = XmlUtil.getFirstNodeByNodeName(cacheNode, GET);
		ConfigurationNode putNode = XmlUtil.getFirstNodeByNodeName(cacheNode, PUT);
		if (getNode == null || putNode == null) {
			getLog().error(String.format("第%s个Cache区必须要配置get方法和put方法", i));
			return false;
		}

		String staticname1 = findValueByAttrName(getNode, ColProperty.staticname);
		String classname1 = findValueByAttrName(getNode, ColProperty.classname);

		String staticname2 = findValueByAttrName(putNode, ColProperty.staticname);
		String classname2 = findValueByAttrName(putNode, ColProperty.classname);

		if (StringUtils.isBlank(staticname1) || StringUtils.isBlank(classname1) || StringUtils.isBlank(staticname2)
				|| StringUtils.isBlank(staticname1) || StringUtils.isBlank(classname2)) {
			getLog().error(String.format("第%s个Cache区缺少get和put方法要配置好staticname/classname", i));
			return false;
		}
		return true;
	}

	private static final String findValueByAttrName(ConfigurationNode node, ColProperty colProperty) {
		return XmlUtil.findValueByAttrName(node, colProperty.name());
	}

}
