package net.troja.eve.esi;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.file.Files;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.Invocation;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.GenericType;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;

import org.glassfish.jersey.client.ClientConfig;
import org.glassfish.jersey.client.ClientProperties;
import org.glassfish.jersey.filter.LoggingFilter;
import org.glassfish.jersey.jackson.JacksonFeature;
import org.glassfish.jersey.media.multipart.FormDataBodyPart;
import org.glassfish.jersey.media.multipart.FormDataContentDisposition;
import org.glassfish.jersey.media.multipart.MultiPart;
import org.glassfish.jersey.media.multipart.MultiPartFeature;

import net.troja.eve.esi.auth.ApiKeyAuth;
import net.troja.eve.esi.auth.Authentication;
import net.troja.eve.esi.auth.HttpBasicAuth;
import net.troja.eve.esi.auth.OAuth;

public class ApiClient {
    private final Map<String, String> defaultHeaderMap = new HashMap<>();
    private String basePath = "https://esi.tech.ccp.is";
    private boolean debugging = false;
    private int connectionTimeout = 0;

    private Client httpClient;
    private final JSON json;
    private String tempFolderPath = null;

    private Map<String, Authentication> authentications;

    private int statusCode;
    private Map<String, List<String>> responseHeaders;

    private DateFormat dateFormat;

    public ApiClient() {
        json = new JSON();
        httpClient = buildHttpClient(debugging);

        dateFormat = new RFC3339DateFormat();

        // Set default User-Agent.
        setUserAgent("Swagger-Codegen/1.0.0/java");

        // Setup authentications (key: authentication name, value:
        // authentication).
        authentications = new HashMap<>();
        authentications.put("evesso", new OAuth());
        // Prevent the authentications from being modified.
        authentications = Collections.unmodifiableMap(authentications);
    }

    /**
     * Gets the JSON instance to do JSON serialization and deserialization.
     */
    public JSON getJSON() {
        return json;
    }

    public Client getHttpClient() {
        return httpClient;
    }

    public ApiClient setHttpClient(final Client httpClient) {
        this.httpClient = httpClient;
        return this;
    }

    public String getBasePath() {
        return basePath;
    }

    public ApiClient setBasePath(final String basePath) {
        this.basePath = basePath;
        return this;
    }

    /**
     * Gets the status code of the previous request
     */
    public int getStatusCode() {
        return statusCode;
    }

    /**
     * Gets the response headers of the previous request
     */
    public Map<String, List<String>> getResponseHeaders() {
        return responseHeaders;
    }

    /**
     * Get authentications (key: authentication name, value: authentication).
     */
    public Map<String, Authentication> getAuthentications() {
        return authentications;
    }

    /**
     * Get authentication for the given name.
     *
     * @param authName
     *            The authentication name
     * @return The authentication, null if not found
     */
    public Authentication getAuthentication(final String authName) {
        return authentications.get(authName);
    }

    /**
     * Helper method to set username for the first HTTP basic authentication.
     */
    public void setUsername(final String username) {
        for (final Authentication auth : authentications.values()) {
            if (auth instanceof HttpBasicAuth) {
                ((HttpBasicAuth) auth).setUsername(username);
                return;
            }
        }
        throw new RuntimeException("No HTTP basic authentication configured!");
    }

    /**
     * Helper method to set password for the first HTTP basic authentication.
     */
    public void setPassword(final String password) {
        for (final Authentication auth : authentications.values()) {
            if (auth instanceof HttpBasicAuth) {
                ((HttpBasicAuth) auth).setPassword(password);
                return;
            }
        }
        throw new RuntimeException("No HTTP basic authentication configured!");
    }

    /**
     * Helper method to set API key value for the first API key authentication.
     */
    public void setApiKey(final String apiKey) {
        for (final Authentication auth : authentications.values()) {
            if (auth instanceof ApiKeyAuth) {
                ((ApiKeyAuth) auth).setApiKey(apiKey);
                return;
            }
        }
        throw new RuntimeException("No API key authentication configured!");
    }

    /**
     * Helper method to set API key prefix for the first API key authentication.
     */
    public void setApiKeyPrefix(final String apiKeyPrefix) {
        for (final Authentication auth : authentications.values()) {
            if (auth instanceof ApiKeyAuth) {
                ((ApiKeyAuth) auth).setApiKeyPrefix(apiKeyPrefix);
                return;
            }
        }
        throw new RuntimeException("No API key authentication configured!");
    }

    /**
     * Helper method to set access token for the first OAuth2 authentication.
     */
    public void setAccessToken(final String accessToken) {
        for (final Authentication auth : authentications.values()) {
            if (auth instanceof OAuth) {
                ((OAuth) auth).setAccessToken(accessToken);
                return;
            }
        }
        throw new RuntimeException("No OAuth2 authentication configured!");
    }

    /**
     * Set the User-Agent header's value (by adding to the default header map).
     */
    public ApiClient setUserAgent(final String userAgent) {
        addDefaultHeader("User-Agent", userAgent);
        return this;
    }

    /**
     * Add a default header.
     *
     * @param key
     *            The header's key
     * @param value
     *            The header's value
     */
    public ApiClient addDefaultHeader(final String key, final String value) {
        defaultHeaderMap.put(key, value);
        return this;
    }

    /**
     * Check that whether debugging is enabled for this API client.
     */
    public boolean isDebugging() {
        return debugging;
    }

    /**
     * Enable/disable debugging for this API client.
     *
     * @param debugging
     *            To enable (true) or disable (false) debugging
     */
    public ApiClient setDebugging(final boolean debugging) {
        this.debugging = debugging;
        // Rebuild HTTP Client according to the new "debugging" value.
        httpClient = buildHttpClient(debugging);
        return this;
    }

    /**
     * The path of temporary folder used to store downloaded files from
     * endpoints with file response. The default value is <code>null</code>,
     * i.e. using the system's default tempopary folder.
     *
     * @see https
     *      ://docs.oracle.com/javase/7/docs/api/java/io/File.html#createTempFile
     *      (java.lang.String,%20java.lang.String,%20java.io.File)
     */
    public String getTempFolderPath() {
        return tempFolderPath;
    }

    public ApiClient setTempFolderPath(final String tempFolderPath) {
        this.tempFolderPath = tempFolderPath;
        return this;
    }

    /**
     * Connect timeout (in milliseconds).
     */
    public int getConnectTimeout() {
        return connectionTimeout;
    }

    /**
     * Set the connect timeout (in milliseconds). A value of 0 means no timeout,
     * otherwise values must be between 1 and {@link Integer#MAX_VALUE}.
     */
    public ApiClient setConnectTimeout(final int connectionTimeout) {
        this.connectionTimeout = connectionTimeout;
        httpClient.property(ClientProperties.CONNECT_TIMEOUT, connectionTimeout);
        return this;
    }

    /**
     * Get the date format used to parse/format date parameters.
     */
    public DateFormat getDateFormat() {
        return dateFormat;
    }

    /**
     * Set the date format used to parse/format date parameters.
     */
    public ApiClient setDateFormat(final DateFormat dateFormat) {
        this.dateFormat = dateFormat;
        // also set the date format for model (de)serialization with Date
        // properties
        json.setDateFormat((DateFormat) dateFormat.clone());
        return this;
    }

    /**
     * Parse the given string into Date object.
     */
    public Date parseDate(final String str) {
        try {
            return dateFormat.parse(str);
        } catch (final java.text.ParseException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Format the given Date object into string.
     */
    public String formatDate(final Date date) {
        return dateFormat.format(date);
    }

    /**
     * Format the given parameter object into string.
     */
    public String parameterToString(final Object param) {
        if (param == null) {
            return "";
        } else if (param instanceof Date) {
            return formatDate((Date) param);
        } else if (param instanceof Collection) {
            final StringBuilder b = new StringBuilder();
            for (final Object o : (Collection) param) {
                if (b.length() > 0) {
                    b.append(',');
                }
                b.append(String.valueOf(o));
            }
            return b.toString();
        } else {
            return String.valueOf(param);
        }
    }

    /*
     * Format to {@code Pair} objects.
     */
    public List<Pair> parameterToPairs(final String collectionFormat, final String name, final Object value) {
        final List<Pair> params = new ArrayList<>();

        // preconditions
        if (name == null || name.isEmpty() || value == null) {
            return params;
        }

        Collection valueCollection;
        if (value instanceof Collection) {
            valueCollection = (Collection) value;
        } else {
            params.add(new Pair(name, parameterToString(value)));
            return params;
        }

        if (valueCollection.isEmpty()) {
            return params;
        }

        // get the collection format
        final String format = collectionFormat == null || collectionFormat.isEmpty() ? "csv" : collectionFormat; // default:
        // csv

        // create the params based on the collection format
        if ("multi".equals(format)) {
            for (final Object item : valueCollection) {
                params.add(new Pair(name, parameterToString(item)));
            }

            return params;
        }

        String delimiter = ",";

        if ("csv".equals(format)) {
            delimiter = ",";
        } else if ("ssv".equals(format)) {
            delimiter = " ";
        } else if ("tsv".equals(format)) {
            delimiter = "\t";
        } else if ("pipes".equals(format)) {
            delimiter = "|";
        }

        final StringBuilder sb = new StringBuilder();
        for (final Object item : valueCollection) {
            sb.append(delimiter);
            sb.append(parameterToString(item));
        }

        params.add(new Pair(name, sb.substring(1)));

        return params;
    }

    /**
     * Check if the given MIME is a JSON MIME. JSON MIME examples:
     * application/json application/json; charset=UTF8 APPLICATION/JSON
     */
    public boolean isJsonMime(final String mime) {
        return mime != null && mime.matches("(?i)application\\/json(;.*)?");
    }

    /**
     * Select the Accept header's value from the given accepts array: if JSON
     * exists in the given array, use it; otherwise use all of them (joining
     * into a string)
     *
     * @param accepts
     *            The accepts array to select from
     * @return The Accept header to use. If the given array is empty, null will
     *         be returned (not to set the Accept header explicitly).
     */
    public String selectHeaderAccept(final String[] accepts) {
        if (accepts.length == 0) {
            return null;
        }
        for (final String accept : accepts) {
            if (isJsonMime(accept)) {
                return accept;
            }
        }
        return StringUtil.join(accepts, ",");
    }

    /**
     * Select the Content-Type header's value from the given array: if JSON
     * exists in the given array, use it; otherwise use the first one of the
     * array.
     *
     * @param contentTypes
     *            The Content-Type array to select from
     * @return The Content-Type header to use. If the given array is empty, JSON
     *         will be used.
     */
    public String selectHeaderContentType(final String[] contentTypes) {
        if (contentTypes.length == 0) {
            return "application/json";
        }
        for (final String contentType : contentTypes) {
            if (isJsonMime(contentType)) {
                return contentType;
            }
        }
        return contentTypes[0];
    }

    /**
     * Escape the given string to be used as URL query value.
     */
    public String escapeString(final String str) {
        try {
            return URLEncoder.encode(str, "utf8").replaceAll("\\+", "%20");
        } catch (final UnsupportedEncodingException e) {
            return str;
        }
    }

    /**
     * Serialize the given Java object into string entity according the given
     * Content-Type (only JSON is supported for now).
     */
    public Entity<?> serialize(final Object obj, final Map<String, Object> formParams, final String contentType)
            throws ApiException {
        Entity<?> entity;
        if (contentType.startsWith("multipart/form-data")) {
            final MultiPart multiPart = new MultiPart();
            for (final Entry<String, Object> param : formParams.entrySet()) {
                if (param.getValue() instanceof File) {
                    final File file = (File) param.getValue();
                    final FormDataContentDisposition contentDisp = FormDataContentDisposition.name(param.getKey())
                            .fileName(file.getName()).size(file.length()).build();
                    multiPart
                            .bodyPart(new FormDataBodyPart(contentDisp, file, MediaType.APPLICATION_OCTET_STREAM_TYPE));
                } else {
                    final FormDataContentDisposition contentDisp = FormDataContentDisposition.name(param.getKey())
                            .build();
                    multiPart.bodyPart(new FormDataBodyPart(contentDisp, parameterToString(param.getValue())));
                }
            }
            entity = Entity.entity(multiPart, MediaType.MULTIPART_FORM_DATA_TYPE);
        } else if (contentType.startsWith("application/x-www-form-urlencoded")) {
            final Form form = new Form();
            for (final Entry<String, Object> param : formParams.entrySet()) {
                form.param(param.getKey(), parameterToString(param.getValue()));
            }
            entity = Entity.entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE);
        } else {
            // We let jersey handle the serialization
            entity = Entity.entity(obj, contentType);
        }
        return entity;
    }

    /**
     * Deserialize response body to Java object according to the Content-Type.
     */
    @SuppressWarnings("unchecked")
    public <T> T deserialize(final Response response, final GenericType<T> returnType) throws ApiException {
        if (response == null || returnType == null) {
            return null;
        }

        if ("byte[]".equals(returnType.toString())) {
            // Handle binary response (byte array).
            return (T) response.readEntity(byte[].class);
        } else if (returnType.getRawType() == File.class) {
            // Handle file downloading.
            final T file = (T) downloadFileFromResponse(response);
            return file;
        }

        String contentType = null;
        final List<Object> contentTypes = response.getHeaders().get("Content-Type");
        if (contentTypes != null && !contentTypes.isEmpty()) {
            contentType = String.valueOf(contentTypes.get(0));
        }
        if (contentType == null) {
            throw new ApiException(500, "missing Content-Type in response");
        }

        return response.readEntity(returnType);
    }

    /**
     * Download file from the given response.
     *
     * @throws ApiException
     *             If fail to read file content from response and write to disk
     */
    public File downloadFileFromResponse(final Response response) throws ApiException {
        try {
            final File file = prepareDownloadFile(response);
            Files.copy(response.readEntity(InputStream.class), file.toPath());
            return file;
        } catch (final IOException e) {
            throw new ApiException(e);
        }
    }

    public File prepareDownloadFile(final Response response) throws IOException {
        String filename = null;
        final String contentDisposition = (String) response.getHeaders().getFirst("Content-Disposition");
        if (contentDisposition != null && !"".equals(contentDisposition)) {
            // Get filename from the Content-Disposition header.
            final Pattern pattern = Pattern.compile("filename=['\"]?([^'\"\\s]+)['\"]?");
            final Matcher matcher = pattern.matcher(contentDisposition);
            if (matcher.find()) {
                filename = matcher.group(1);
            }
        }

        String prefix;
        String suffix = null;
        if (filename == null) {
            prefix = "download-";
            suffix = "";
        } else {
            final int pos = filename.lastIndexOf('.');
            if (pos == -1) {
                prefix = filename + "-";
            } else {
                prefix = filename.substring(0, pos) + "-";
                suffix = filename.substring(pos);
            }
            // File.createTempFile requires the prefix to be at least three
            // characters long
            if (prefix.length() < 3) {
                prefix = "download-";
            }
        }

        if (tempFolderPath == null) {
            return File.createTempFile(prefix, suffix);
        } else {
            return File.createTempFile(prefix, suffix, new File(tempFolderPath));
        }
    }

    /**
     * Invoke API by sending HTTP request with the given options.
     *
     * @param path
     *            The sub-path of the HTTP URL
     * @param method
     *            The request method, one of "GET", "POST", "PUT", and "DELETE"
     * @param queryParams
     *            The query parameters
     * @param body
     *            The request body object
     * @param headerParams
     *            The header parameters
     * @param formParams
     *            The form parameters
     * @param accept
     *            The request's Accept header
     * @param contentType
     *            The request's Content-Type header
     * @param authNames
     *            The authentications to apply
     * @param returnType
     *            The return type into which to deserialize the response
     * @return The response body in type of string
     */
    public <T> T invokeAPI(final String path, final String method, final List<Pair> queryParams, final Object body,
            final Map<String, String> headerParams, final Map<String, Object> formParams, final String accept,
            final String contentType, final String[] authNames, final GenericType<T> returnType) throws ApiException {
        updateParamsForAuth(authNames, queryParams, headerParams);

        // Not using `.target(this.basePath).path(path)` below,
        // to support (constant) query string in `path`, e.g. "/posts?draft=1"
        WebTarget target = httpClient.target(basePath + path);

        if (queryParams != null) {
            for (final Pair queryParam : queryParams) {
                if (queryParam.getValue() != null) {
                    target = target.queryParam(queryParam.getName(), queryParam.getValue());
                }
            }
        }

        Invocation.Builder invocationBuilder = target.request().accept(accept);

        for (final Entry<String, String> entry : headerParams.entrySet()) {
            final String value = entry.getValue();
            if (value != null) {
                invocationBuilder = invocationBuilder.header(entry.getKey(), value);
            }
        }

        for (final Entry<String, String> entry : defaultHeaderMap.entrySet()) {
            final String key = entry.getKey();
            if (!headerParams.containsKey(key)) {
                final String value = entry.getValue();
                if (value != null) {
                    invocationBuilder = invocationBuilder.header(key, value);
                }
            }
        }

        final Entity<?> entity = serialize(body, formParams, contentType);

        Response response;

        if ("GET".equals(method)) {
            response = invocationBuilder.get();
        } else if ("POST".equals(method)) {
            response = invocationBuilder.post(entity);
        } else if ("PUT".equals(method)) {
            response = invocationBuilder.put(entity);
        } else if ("DELETE".equals(method)) {
            response = invocationBuilder.delete();
        } else if ("PATCH".equals(method)) {
            response = invocationBuilder.header("X-HTTP-Method-Override", "PATCH").post(entity);
        } else {
            throw new ApiException(500, "unknown method type " + method);
        }

        statusCode = response.getStatusInfo().getStatusCode();
        responseHeaders = buildResponseHeaders(response);

        if (response.getStatus() == Status.NO_CONTENT.getStatusCode()) {
            return null;
        } else if (response.getStatusInfo().getFamily() == Status.Family.SUCCESSFUL) {
            if (returnType == null) {
                return null;
            } else {
                return deserialize(response, returnType);
            }
        } else {
            String message = "error";
            String respBody = null;
            if (response.hasEntity()) {
                try {
                    respBody = String.valueOf(response.readEntity(String.class));
                    message = respBody;
                } catch (final RuntimeException e) {
                    // e.printStackTrace();
                }
            }
            throw new ApiException(response.getStatus(), message, buildResponseHeaders(response), respBody);
        }
    }

    /**
     * Build the Client used to make HTTP requests.
     */
    private Client buildHttpClient(final boolean debugging) {
        final ClientConfig clientConfig = new ClientConfig();
        clientConfig.register(MultiPartFeature.class);
        clientConfig.register(json);
        clientConfig.register(JacksonFeature.class);
        if (debugging) {
            clientConfig.register(LoggingFilter.class);
        }
        return ClientBuilder.newClient(clientConfig);
    }

    private Map<String, List<String>> buildResponseHeaders(final Response response) {
        final Map<String, List<String>> responseHeaders = new HashMap<>();
        for (final Entry<String, List<Object>> entry : response.getHeaders().entrySet()) {
            final List<Object> values = entry.getValue();
            final List<String> headers = new ArrayList<>();
            for (final Object o : values) {
                headers.add(String.valueOf(o));
            }
            responseHeaders.put(entry.getKey(), headers);
        }
        return responseHeaders;
    }

    /**
     * Update query and header parameters based on authentication settings.
     *
     * @param authNames
     *            The authentications to apply
     */
    private void updateParamsForAuth(final String[] authNames, final List<Pair> queryParams,
            final Map<String, String> headerParams) {
        for (final String authName : authNames) {
            final Authentication auth = authentications.get(authName);
            if (auth == null) {
                throw new RuntimeException("Authentication undefined: " + authName);
            }
            auth.applyToParams(queryParams, headerParams);
        }
    }
}
