package net.tomred.liquibase.validator;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;

import com.google.inject.Guice;
import com.google.inject.Injector;
import com.google.inject.Module;
import com.google.inject.ProvisionException;

import net.tomred.liquibase.validator.add.column.module.AddColumnToAuditSchemaWithRemarksModule;
import net.tomred.liquibase.validator.exists.column.module.ColumnExistsModule;
import net.tomred.liquibase.validator.exists.table.module.TableExistsModule;
import net.tomred.liquibase.validator.global.properties.module.PreventGlobalPropertiesModule;
import net.tomred.liquibase.validator.module.ValidatorModule;
import net.tomred.liquibase.validator.stax.LiquibaseHandler;

/**
 * An extensible Maven plugin to validate liquibase files.
 * 
 * @author liquibase@tomred.net
 */

@Mojo(name = "validate", defaultPhase = LifecyclePhase.TEST, threadSafe = true)
public class ValidateMojo extends AbstractMojo {

	private final Log logger;

	/**
	 * The base directory from which to begin locating files for validation.
	 */
	@Parameter(required = false)
	private String dir;

	/**
	 * The files to be ignored, can be relative paths or just file names.
	 */
	@Parameter
	private List<String> ignore;

	/**
	 * Validator Enabled Map
	 */
	@Parameter
	private Map<String, Boolean> validators;

	public ValidateMojo() {
		LogFactory.setLogger(getLog());
		logger = LogFactory.getInstance();
	}

	public void execute() throws MojoExecutionException {
		long start = System.currentTimeMillis();

		if (validators == null) {
			validators = new HashMap<String, Boolean>();
		}
		// -- Configure Modules
		Injector injector = createInjector();
		// -- Validate all files
		try {
			Map<File, List<ValidationResponse>> responses = injector.getInstance(LiquibaseHandler.class).validate();
			long end = System.currentTimeMillis();
			// -- Present Responses
			int errorCount = MavenOutput.printValidationErrors(responses);
			ValidatorConfiguration config = injector.getInstance(ValidatorConfiguration.class);
			MavenOutput.printStatistics(config.getFilesToProcess().size(), responses.size(), errorCount, (end - start));
			if (responses.size() > 0) {
				throw new MojoExecutionException("Liquibase Validation Failed.");
			}
		} catch (ProvisionException pe) {
			throw new MojoExecutionException("At least one validator must be enabled");
		}
	}

	/**
	 * Not the cleanest way but a quick means of adding configurable validators
	 * 
	 * @param directory
	 * @return
	 */
	private Injector createInjector() {
		List<Module> modules = new ArrayList<Module>();
		modules.add(new ValidatorModule(getDirectory(), ignore));
		if (!validators.containsKey(ColumnExistsModule.module) || validators.get(ColumnExistsModule.module)) {
			modules.add(new ColumnExistsModule());
		}
		if (!validators.containsKey(TableExistsModule.module) || validators.get(TableExistsModule.module)) {
			modules.add(new TableExistsModule());
		}
		if (!validators.containsKey(AddColumnToAuditSchemaWithRemarksModule.module) || validators.get(AddColumnToAuditSchemaWithRemarksModule.module)) {
			modules.add(new AddColumnToAuditSchemaWithRemarksModule());
		}
		if (!validators.containsKey(PreventGlobalPropertiesModule.module) || validators.get(PreventGlobalPropertiesModule.module)) {
			modules.add(new PreventGlobalPropertiesModule());
		}
		if (modules.size() == 1) {
			throw new IllegalStateException("Atleast one Validator must be enabled");
		}
		return Guice.createInjector(modules);
	}

	private File getDirectory() {
		if (dir == null) {
			this.dir = ".";
		}

		File directory = new File(dir);
		logger.info("-------------------------------------------------------------------------------");
		logger.info("Validating " + directory.getAbsolutePath());
		logger.info("-------------------------------------------------------------------------------");

		return directory;
	}
}
