package net.tomred.liquibase.validator.utils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import net.tomred.liquibase.validator.LogFactory;

import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.maven.plugin.logging.Log;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

public class XmlFilenameFilter implements IOFileFilter {
	private final Log log = LogFactory.getInstance();
	private List<String> ignores = new ArrayList<String>();

	public XmlFilenameFilter(List<String> ignores) {
		if (ignores != null) {
			this.ignores.addAll(ignores);
		}
	}

	@Override
	public boolean accept(File file) {
		try {
			return isNotOnIngoreList(file) && isTypeXML(file) && file.exists() && isNotEmpty(file) && isAccessible(file) && containsLiquibaseNamespace(file);
		} catch (IOException e) {
			return false;
		}
	}

	private boolean isNotEmpty(File file) {
		return file.length() != 0L;
	}

	private boolean isAccessible(File file) {
		return file.canRead();
	}

	private boolean isTypeXML(File file) {
		return file.getName().endsWith(".xml");
	}

	private boolean isNotOnIngoreList(File file) throws IOException {
		for (String ignore : ignores) {
			if (file.getCanonicalPath().trim().contains(ignore.trim())) {
				return false;
			}
		}
		return true;
	}

	private boolean containsLiquibaseNamespace(File file) {
		FileInputStream fis = null;
		try {
			fis = new FileInputStream(file);
			DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
			dbf.setNamespaceAware(true);
			DocumentBuilder db = dbf.newDocumentBuilder();
			Document document = db.parse(fis);
			Node node = document.getDocumentElement();
			String namespaceURI = node.getNamespaceURI();
			return namespaceURI != null && namespaceURI.contains("http://www.liquibase.org");
		} catch (FileNotFoundException e) {
			logException(file, e);
			return false;
		} catch (ParserConfigurationException e) {
			logException(file, e);
			return false;
		} catch (SAXException e) {
			logException(file, e);
			return false;
		} catch (IOException e) {
			logException(file, e);
			return false;
		} finally {
			try {
				if (fis != null) {
					fis.close();
				}
			} catch (IOException e) {
				System.out.println("Failed to close FileInputStream : " + e.toString());
			}
		}
	}

	private void logException(File file, Exception e) {
		log.error(file.getPath());
		log.error(" --- " + e.toString());
		log.error("");
	}

	@Override
	public boolean accept(File arg0, String arg1) {
		return true;
	}

}
