package net.thucydides.easyb

import net.thucydides.core.steps.StepFailure
import org.easyb.BehaviorStep
import org.easyb.domain.Behavior
import org.easyb.listener.ExecutionListenerAdaptor
import org.easyb.result.Result
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import static net.thucydides.core.steps.ExecutedStepDescription.withTitle
import static org.easyb.result.Result.FAILED
import static org.easyb.result.Result.IGNORED
import static org.easyb.result.Result.PENDING
import static org.easyb.util.BehaviorStepType.AND
import static org.easyb.util.BehaviorStepType.GIVEN
import static org.easyb.util.BehaviorStepType.SCENARIO
import static org.easyb.util.BehaviorStepType.STORY
import static org.easyb.util.BehaviorStepType.THEN
import static org.easyb.util.BehaviorStepType.WHEN
import net.thucydides.core.steps.StepEventBus
import net.thucydides.core.steps.ExecutedStepDescription
import org.easyb.util.BehaviorStepType

class ThucydidesExecutionListener extends ExecutionListenerAdaptor {

    private static final Logger LOGGER = LoggerFactory.getLogger(ThucydidesExecutionListener.class);

    ThucydidesExecutionListener() {
    }

    void startBehavior(Behavior behavior) {
        StepEventBus.eventBus.clear()
    }

    private Stack<BehaviorStepType> stepTypeStack = new Stack<BehaviorStepType>();

    private boolean stepFailed

    @Override
    void startStep(BehaviorStep step) {
        LOGGER.debug("THUCYDIDES STARTING STEP $step")
        switch (step.stepType) {

            case SCENARIO :
                StepEventBus.eventBus.testStarted(groupNameFrom(step))
                stepFailed = false;
                break;

            case [GIVEN, WHEN, THEN, AND] :
                String groupName = groupNameFrom(step);
                StepEventBus.eventBus.stepStarted(ExecutedStepDescription.withTitle(groupName))
                break;
        }
    }

    String groupNameFrom(BehaviorStep step) {

        switch (step.stepType) {
            case GIVEN :
                return "Given $step.name"

            case WHEN :
                return "When $step.name"

            case THEN :
                return "Then $step.name"

            case AND :
                return "And $step.name"

            default :
                return step.name
        }
    }

    @Override
    void stopStep() {}


    void gotResult(Result result) {
        if (StepEventBus.eventBus.areStepsRunning()) {
            notifyStepListenerUsingEasybResult(result)
        }
    }

    private def notifyStepListenerUsingEasybResult(Result result) {
        switch (result.status()) {
            case FAILED:
                System.out.println("STEP FAILED")
                StepEventBus.eventBus.stepFailed(new StepFailure(withTitle(result.cause.message), result.cause));
                break;

            case IGNORED:
                System.out.println("STEP IGNORED")
                StepEventBus.eventBus.stepIgnored()
                break;

            case PENDING:
                System.out.println("STEP PENDING")
                StepEventBus.eventBus.stepPending()
                break;

            default:
                if (aPreviousStepHasFailed()) {
                    System.out.println("STEP SKIPPED")
                    StepEventBus.eventBus.stepIgnored();
                }  else {
                    System.out.println("STEP FINISHED")
                    StepEventBus.eventBus.stepFinished()
                }
                break;
        }
    }


    private boolean aPreviousStepHasFailed() {
        if (stepFailed) {
            System.out.println("PREVIOUS STEP FAILED")
            return stepFailed
        } else {
            if (StepEventBus.eventBus.aStepInTheCurrentTestHasFailed()) {
                System.out.println("A STEP HAS FAILED")
                stepFailed = true;
                return false;
            }
        }
    }

    void completeTesting() {
    }
}
