/**
 * ====================================================================
 *            Nuts : Network Updatable Things Service
 *                  (universal package manager)
 * <br>
 * is a new Open Source Package Manager to help install packages
 * and libraries for runtime execution. Nuts is the ultimate companion for
 * maven (and other build managers) as it helps installing all package
 * dependencies at runtime. Nuts is not tied to java and is a good choice
 * to share shell scripts and other 'things' . Its based on an extensible
 * architecture to help supporting a large range of sub managers / repositories.
 *
 * <br>
 *
 * Copyright [2020] [thevpc]
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain a
 * copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 * <br>
 * ====================================================================
 */
package net.thevpc.nuts.runtime.core.model;

import net.thevpc.nuts.*;
import net.thevpc.nuts.runtime.bundles.common.MapToFunction;
import net.thevpc.nuts.runtime.core.util.CoreNutsUtils;

import java.util.*;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.UnaryOperator;
import java.util.stream.Collectors;

import net.thevpc.nuts.runtime.core.util.CoreArrayUtils;

public class DefaultNutsDescriptorBuilder implements NutsDescriptorBuilder {

    private static final long serialVersionUID = 1L;

    private NutsId id;
    //    private String alternative;
    private NutsId[] parents=new NutsId[0]; //defaults to empty;
    private String packaging;
    //    private String ext;
    private boolean executable;
    private boolean application;
    private NutsArtifactCall executor;
    private NutsArtifactCall installer;
    /**
     * short description
     */
    private String name;
    private List<String> icons=new ArrayList<>();
    private List<String> categories;
    private String genericName;
    /**
     * some longer (but not too long) description
     */
    private String description;
    private List<String> arch=new ArrayList<>(); //defaults to empty
    private List<String> os=new ArrayList<>(); //defaults to empty;
    private List<String> osdist=new ArrayList<>(); //defaults to empty;
    private List<String> platform=new ArrayList<>(); //defaults to empty;
    private List<String> desktopEnvironment=new ArrayList<>(); //defaults to empty;
    private List<NutsIdLocation> locations=new ArrayList<>(); //defaults to empty;
    private List<NutsClassifierMapping> classifierMappings=new ArrayList<>(); //defaults to empty;
    private List<NutsDependency> dependencies=new ArrayList<>(); //defaults to empty;
    private List<NutsDependency> standardDependencies=new ArrayList<>(); //defaults to empty;
    private Map<String, String> properties=new HashMap<>(); //defaults to empty;
    private transient NutsSession session;

    public DefaultNutsDescriptorBuilder() {
    }
    
    public DefaultNutsDescriptorBuilder(NutsSession session) {
        this.session=session;
    }

    public DefaultNutsDescriptorBuilder(NutsDescriptor other,NutsSession session) {
        this.session=session;
        set(other);
    }

    @Override
    public NutsDescriptorBuilder clear() {
        setId((NutsId)null);
//            setAlternative(null);
        setPackaging(null);
        setParents(null);
        setExecutable(false);
        setApplication(false);
        setDescription(null);
        setName(null);
        setExecutor(null);
        setInstaller(null);
        setClassifierMappings(null);
        setArch(null);
        setOs(null);
        setOsdist(null);
        setPlatform(null);
        setDesktopEnvironment(null);
        setLocations(null);
        setDependencies((NutsDependency[])null);
        setStandardDependencies((NutsDependency[])null);
        setProperties(null);
        setIcons(new ArrayList<>());
        setCategories(null);
        setGenericName(null);
        return this;
    }

    @Override
    public NutsDescriptorBuilder set(NutsDescriptorBuilder other) {
        if (other != null) {
            setId(other.getId());
//            setAlternative(other.getAlternative());
            setPackaging(other.getPackaging());
            setParents(other.getParents());
            setExecutable(other.isExecutable());
            setApplication(other.isApplication());
            setDescription(other.getDescription());
            setName(other.getName());
            setExecutor(other.getExecutor());
            setInstaller(other.getInstaller());
//            setExt(other.getExt());
            setArch(other.getArch());
            setOs(other.getOs());
            setOsdist(other.getOsdist());
            setPlatform(other.getPlatform());
            setDesktopEnvironment(other.getDesktopEnvironment());
            setLocations(other.getLocations());
            setDependencies(other.getDependencies());
            setStandardDependencies(other.getStandardDependencies());
            setProperties(other.getProperties());
            setIcons(new ArrayList<>(other.getIcons()));
            setCategories(other.getCategories());
            setGenericName(other.getGenericName());
        }else{
            clear();
        }
        return this;
    }

    @Override
    public NutsDescriptorBuilder set(NutsDescriptor other) {
        if (other != null) {
            setId(other.getId());
//            setAlternative(other.getAlternative());
            setPackaging(other.getPackaging());
            setParents(other.getParents());
            setExecutable(other.isExecutable());
            setApplication(other.isApplication());
            setDescription(other.getDescription());
            setName(other.getName());
            setExecutor(other.getExecutor());
            setInstaller(other.getInstaller());
            setClassifierMappings(other.getClassifierMappings());
            setArch(other.getArch());
            setOs(other.getOs());
            setOsdist(other.getOsdist());
            setPlatform(other.getPlatform());
            setDesktopEnvironment(other.getDesktopEnvironment());
            setLocations(other.getLocations());
            setDependencies(other.getDependencies());
            setStandardDependencies(other.getStandardDependencies());
            setProperties(other.getProperties());
            setIcons(new ArrayList<>(Arrays.asList(other.getIcons())));
            setGenericName(other.getGenericName());
            setCategories(new ArrayList<>(Arrays.asList(other.getCategories())));
        }else{
            clear();
        }
        return this;
    }

    @Override
    public NutsDescriptorBuilder setId(String id) {
        this.id = session.getWorkspace().id().parser().setLenient(false).parse(id);
        return this;
    }

    @Override
    public NutsDescriptorBuilder setId(NutsId id) {
        this.id = id;
        return this;
    }

    @Override
    public NutsDescriptorBuilder setName(String name) {
        this.name = NutsUtilStrings.trim(name);
        return this;
    }

    @Override
    public NutsDescriptorBuilder setExecutor(NutsArtifactCall executor) {
        this.executor = executor;
        return this;
    }

    @Override
    public NutsDescriptorBuilder setInstaller(NutsArtifactCall installer) {
        this.installer = installer;
        return this;
    }

//    @Override
//    public NutsDescriptorBuilder setAlternative(String alternative) {
//        this.alternative = alternative;
//        return this;
//    }

    @Override
    public NutsDescriptorBuilder setDescription(String description) {
        this.description = NutsUtilStrings.trim(description);
        return this;
    }

    @Override
    public NutsDescriptorBuilder setExecutable(boolean executable) {
        this.executable = executable;
        return this;
    }

    @Override
    public NutsDescriptorBuilder setApplication(boolean nutsApp) {
        this.application = nutsApp;
        return this;
    }

    //    @Override
//    public NutsDescriptorBuilder setExt(String ext) {
//        this.ext = NutsUtilStrings.trim(ext);
//        return this;
//    }
    public NutsDescriptorBuilder addPlatform(String platform) {
        if (platform != null) {
            if (this.platform == null) {
                this.platform = new ArrayList<>();
            }
            this.platform.add(platform);
        }
        return this;
    }

    public NutsDescriptorBuilder setPlatform(String[] platform) {
        this.platform = new ArrayList<>(Arrays.asList(CoreArrayUtils.toArraySet(platform)));
        return this;
    }

    public NutsDescriptorBuilder addDesktopEnvironment(String desktopEnvironment) {
        if (desktopEnvironment != null) {
            if (this.desktopEnvironment == null) {
                this.desktopEnvironment = new ArrayList<>();
            }
            this.desktopEnvironment.add(desktopEnvironment);
        }
        return this;
    }

    public NutsDescriptorBuilder setDesktopEnvironment(String[] desktopEnvironment) {
        this.desktopEnvironment = new ArrayList<>(Arrays.asList(CoreArrayUtils.toArraySet(desktopEnvironment)));
        return this;
    }

    public NutsDescriptorBuilder setOs(String[] os) {
        this.os = new ArrayList<>(Arrays.asList(CoreArrayUtils.toArraySet(os)));
        return this;
    }

    public NutsDescriptorBuilder setOsdist(String[] osdist) {
        this.osdist = new ArrayList<>(Arrays.asList(CoreArrayUtils.toArraySet(osdist)));
        return this;
    }

    public NutsDescriptorBuilder setArch(String[] arch) {
        this.arch = new ArrayList<>(Arrays.asList(CoreArrayUtils.toArraySet(arch)));
        return this;
    }

    @Override
    public NutsDescriptorBuilder setProperties(Map<String, String> properties) {
        if (properties == null || properties.isEmpty()) {
            this.properties = null;
        } else {
            this.properties = new HashMap<>(properties);
        }
        return this;
    }

    @Override
    public NutsDescriptorBuilder addProperties(Map<String, String> properties) {
        if (properties == null || properties.isEmpty()) {
            //do nothing
        } else {
            HashMap<String, String> p = new HashMap<>();
            if(this.properties!=null){
                p.putAll(this.properties);
            }
            p.putAll(properties);
            this.properties = p;
        }
        return this;
    }

    @Override
    public NutsDescriptorBuilder addLocation(NutsIdLocation location) {
        if (this.locations == null) {
            this.locations = new ArrayList<>();
        }
        this.locations.add(location);
        return this;
    }

    @Override
    public NutsDescriptorBuilder setLocations(NutsIdLocation[] locations) {
        this.locations = (locations == null)?new ArrayList<>() : new ArrayList<>(Arrays.asList(locations));
        return this;
    }


    @Override
    public NutsDescriptorBuilder addClassifierMapping(NutsClassifierMapping mapping) {
        if (this.classifierMappings == null) {
            this.classifierMappings = new ArrayList<>();
        }
        this.classifierMappings.add(mapping);
        return this;
    }

    @Override
    public NutsDescriptorBuilder setClassifierMappings(NutsClassifierMapping[] value) {
        this.classifierMappings = value==null?new ArrayList<>() : new ArrayList<>(Arrays.asList(value));
        return this;
    }

    @Override
    public NutsDescriptorBuilder setPackaging(String packaging) {
        this.packaging = NutsUtilStrings.trim(packaging);
        return this;
    }

    public NutsDescriptorBuilder setParents(NutsId[] parents) {
        this.parents = parents == null ? new NutsId[0] : new NutsId[parents.length];
        if (parents != null) {
            System.arraycopy(parents, 0, this.parents, 0, this.parents.length);
        }
        return this;
    }

//    public String getAlternative() {
//        return alternative;
//    }

    @Override
    public NutsArtifactCall getInstaller() {
        return installer;
    }

    @Override
    public Map<String, String> getProperties() {
        if (properties == null) {
            properties = new HashMap<>();
        }
        return properties;
    }

    @Override
    public NutsId[] getParents() {
        return parents;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public boolean isExecutable() {
        return executable;
    }

    @Override
    public boolean isApplication() {
        return application;
    }

    @Override
    public NutsArtifactCall getExecutor() {
        return executor;
    }

    //    @Override
//    public String getExt() {
//        return ext;
//    }
    @Override
    public String getPackaging() {
        return packaging;
    }

    @Override
    public NutsId getId() {
        return id;
    }

    @Override
    public NutsDependency[] getDependencies() {
        return dependencies == null ? new NutsDependency[0] : dependencies.toArray(new NutsDependency[0]);
    }

    @Override
    public NutsDependency[] getStandardDependencies() {
        return standardDependencies == null ? new NutsDependency[0] : standardDependencies.toArray(new NutsDependency[0]);
    }

    @Override
    public String[] getArch() {
        return arch == null ? new String[0]
                : arch.toArray(new String[0]);
    }

    public String[] getOs() {
        return os == null ? new String[0] : os.toArray(new String[0]);
    }

    public String[] getOsdist() {
        return osdist == null ? new String[0] : osdist.toArray(new String[0]);
    }

    public String[] getPlatform() {
        return platform == null ? new String[0] : platform.toArray(new String[0]);
    }

    public String[] getDesktopEnvironment() {
        return desktopEnvironment == null ? new String[0] : desktopEnvironment.toArray(new String[0]);
    }

    @Override
    public NutsDescriptorBuilder setDependencies(NutsDependency[] dependencies) {
        this.dependencies = new ArrayList<>();
        if(dependencies!=null) {
            for (NutsDependency dependency : dependencies) {
                if (dependency == null) {
                    throw new NullPointerException();
                }
                this.dependencies.add(dependency);
            }
        }
        return this;
    }

    @Override
    public NutsDescriptorBuilder setStandardDependencies(NutsDependency[] dependencies) {
        this.standardDependencies = new ArrayList<>();
        if(dependencies!=null) {
            for (NutsDependency dependency : dependencies) {
                if (dependency == null) {
                    throw new NullPointerException();
                }
                this.standardDependencies.add(dependency);
            }
        }
        return this;
    }

    @Override
    public NutsDescriptor build() {
        return new DefaultNutsDescriptor(
                getId(), /*getAlternative(),*/ getParents(), getPackaging(), isExecutable(), isApplication(),
                //                getExt(),
                getExecutor(), getInstaller(),
                getName(), getDescription(), getArch(), getOs(), getOsdist(), getPlatform(),
                getDesktopEnvironment(),
                getDependencies(), getStandardDependencies(),
                getLocations(), getProperties(), getClassifierMappings(),
                genericName,
                categories==null?new String[0] :categories.toArray(new String[0]) ,
                icons==null?new String[0] :icons.toArray(new String[0]) ,
                session
        );
    }

    @Override
    public NutsIdLocation[] getLocations() {
        return locations == null ? new NutsIdLocation[0] : locations.toArray(new NutsIdLocation[0]);
    }

    @Override
    public NutsClassifierMapping[] getClassifierMappings() {
        return classifierMappings == null ? new NutsClassifierMapping[0] : classifierMappings.toArray(new NutsClassifierMapping[0]);
    }

    @Override
    public NutsDescriptorBuilder setProperty(String name, String value) {
        if(value==null){
            if(properties!=null) {
                properties.remove(name);
            }
        }else {
            if(properties==null){
                properties=new HashMap<>();
            }
            properties.put(name, value);
        }
        return this;
    }

    @Override
    public NutsDescriptorBuilder addOs(String os) {
        if (this.os == null) {
            this.os = new ArrayList<>();
        }
        this.os.add(os);
        return this;
    }

    @Override
    public NutsDescriptorBuilder addOsdist(String osdist) {
        if (this.osdist == null) {
            this.osdist = new ArrayList<>();
        }
        this.osdist.add(osdist);
        return this;
    }

    @Override
    public NutsDescriptorBuilder addArch(String arch) {
        if (this.arch == null) {
            this.arch = new ArrayList<>();
        }
        this.arch.add(arch);
        return this;
    }

    @Override
    public NutsDescriptorBuilder removeOs(String os) {
        if (this.os != null) {
            this.os.remove(os);
        }
        return this;
    }

    @Override
    public NutsDescriptorBuilder removeOsdist(String osdist) {
        if (this.osdist != null) {
            this.osdist.remove(osdist);
        }
        return this;
    }

    @Override
    public NutsDescriptorBuilder removeArch(String arch) {
        if (this.arch != null) {
            this.arch.remove(arch);
        }
        return this;
    }

    @Override
    public NutsDescriptorBuilder removePlatform(String platform) {
        if (this.platform != null) {
            this.platform.remove(platform);
        }
        return this;
    }

    @Override
    public NutsDescriptorBuilder removeDesktopEnvironment(String desktopEnvironment) {
        if (this.desktopEnvironment != null) {
            this.desktopEnvironment.remove(desktopEnvironment);
        }
        return this;
    }

    @Override
    public NutsDescriptorBuilder removeDependency(NutsDependency dependency) {
        if (this.dependencies != null) {
            this.dependencies.remove(dependency);
        }
        return this;
    }

    @Override
    public NutsDescriptorBuilder removeStandardDependency(NutsDependency dependency) {
        if (this.standardDependencies != null) {
            this.standardDependencies.remove(dependency);
        }
        return this;
    }

    @Override
    public NutsDescriptorBuilder addDependency(NutsDependency dependency) {
        if (dependency == null) {
            throw new NullPointerException();
        }
        if (this.dependencies == null) {
            this.dependencies = new ArrayList<>();
        }
        this.dependencies.add(dependency);
        return this;
    }

    @Override
    public NutsDescriptorBuilder addStandardDependency(NutsDependency dependency) {
        if (this.standardDependencies == null) {
            this.standardDependencies = new ArrayList<>();
        }
        this.standardDependencies.add(dependency);
        return this;
    }

    @Override
    public NutsDescriptorBuilder addDependencies(NutsDependency[] dependencies) {
        if (this.dependencies == null) {
            this.dependencies = new ArrayList<>();
        }
        this.dependencies.addAll(Arrays.asList(dependencies));
        return this;
    }

    @Override
    public NutsDescriptorBuilder addStandardDependencies(NutsDependency[] dependencies) {
        if (this.standardDependencies == null) {
            this.standardDependencies = new ArrayList<>();
        }
        this.standardDependencies.addAll(Arrays.asList(dependencies));
        return this;
    }

    @Override
    public NutsDescriptorBuilder applyProperties() {
        return applyProperties(getProperties());
    }

    @Override
    public NutsDescriptorBuilder replaceProperty(Predicate<Map.Entry<String, String>> filter, Function<Map.Entry<String, String>, String> converter) {
        if (converter == null) {
            return this;
        }
        Map<String, String> p = new LinkedHashMap<>();
        boolean someUpdate = false;
        for (Iterator<Map.Entry<String, String>> it = getProperties().entrySet().iterator(); it.hasNext(); ) {
            Map.Entry<String, String> entry = it.next();
            if (filter == null || filter.test(entry)) {
                String v = converter.apply(entry);
                if (v != null) {
                    p.put(entry.getKey(), entry.getValue());
                    if (!Objects.equals(v, entry.getValue())) {
                        someUpdate = true;
                    }
                } else {
                    it.remove();
                }
            }
        }
        if (someUpdate) {
            for (Map.Entry<String, String> entry : p.entrySet()) {
                getProperties().replace(entry.getKey(), entry.getValue());
            }
        }
        return this;
    }

    @Override
    public NutsDescriptorBuilder applyParents(NutsDescriptor[] parentDescriptors) {
        NutsId n_id = getId();
//        String n_alt = getAlternative();
        String n_packaging = getPackaging();
//        String n_ext = getExt();
        boolean n_executable = isExecutable();
        String n_name = getName();
        List<String> n_categories = getCategories();
        if(n_categories==null){
            n_categories=new ArrayList<>();
        }else{
            n_categories=new ArrayList<>(n_categories);
        }
        List<String> n_icons = getIcons();
        if(n_icons==null){
            n_icons=new ArrayList<>();
        }else{
            n_icons=new ArrayList<>(n_icons);
        }
        String n_genericName = getGenericName();
        String n_desc = getDescription();
        NutsArtifactCall n_executor = getExecutor();
        NutsArtifactCall n_installer = getInstaller();
        Map<String, String> n_props = new HashMap<>();
        for (NutsDescriptor parentDescriptor : parentDescriptors) {
            n_props.putAll(parentDescriptor.getProperties());
        }
        Map<String, String> properties = getProperties();
        if (properties != null) {
            n_props.putAll(properties);
        }
        LinkedHashSet<NutsDependency> n_deps = new LinkedHashSet<>();
        LinkedHashSet<NutsDependency> n_sdeps = new LinkedHashSet<>();
        LinkedHashSet<String> n_archs = new LinkedHashSet<>();
        LinkedHashSet<String> n_os = new LinkedHashSet<>();
        LinkedHashSet<String> n_osdist = new LinkedHashSet<>();
        LinkedHashSet<String> n_platform = new LinkedHashSet<>();
        LinkedHashSet<String> n_desktopEnvironment = new LinkedHashSet<>();
        for (NutsDescriptor parentDescriptor : parentDescriptors) {
            n_id = CoreNutsUtils.applyNutsIdInheritance(n_id, parentDescriptor.getId(),session.getWorkspace());
            if (!n_executable && parentDescriptor.isExecutable()) {
                n_executable = true;
            }
            if (n_executor == null) {
                n_executor = parentDescriptor.getExecutor();
            }
            if (n_executor == null) {
                n_installer = parentDescriptor.getInstaller();
            }

            //packaging is not inherited!!
            //n_packaging = applyStringInheritance(n_packaging, parentDescriptor.getPackaging());
//            n_ext = CoreNutsUtils.applyStringInheritance(n_ext, parentDescriptor.getExt());
            n_name = CoreNutsUtils.applyStringInheritance(n_name, parentDescriptor.getName());
            n_genericName = CoreNutsUtils.applyStringInheritance(n_genericName, parentDescriptor.getGenericName());
            n_desc = CoreNutsUtils.applyStringInheritance(n_desc, parentDescriptor.getDescription());
            n_deps.addAll(Arrays.asList(parentDescriptor.getDependencies()));
            n_sdeps.addAll(Arrays.asList(parentDescriptor.getStandardDependencies()));
            n_archs.addAll(Arrays.asList(parentDescriptor.getArch()));
            n_icons.addAll(Arrays.asList(parentDescriptor.getIcons()));
            n_categories.addAll(Arrays.asList(parentDescriptor.getCategories()));
            n_os.addAll(Arrays.asList(parentDescriptor.getOs()));
            n_osdist.addAll(Arrays.asList(parentDescriptor.getOsdist()));
            n_platform.addAll(Arrays.asList(parentDescriptor.getPlatform()));
            n_desktopEnvironment.addAll(Arrays.asList(parentDescriptor.getDesktopEnvironment()));
        }
        n_deps.addAll(Arrays.asList(getDependencies()));
        n_sdeps.addAll(Arrays.asList(getStandardDependencies()));
        n_archs.addAll(Arrays.asList(getArch()));
        n_os.addAll(Arrays.asList(getOs()));
        n_osdist.addAll(Arrays.asList(getOsdist()));
        n_platform.addAll(Arrays.asList(getPlatform()));
        n_desktopEnvironment.addAll(Arrays.asList(getDesktopEnvironment()));
        NutsId[] n_parents = new NutsId[0];

        setId(n_id);
//        setAlternative(n_alt);
        setParents(n_parents);
        setPackaging(n_packaging);
        setExecutable(n_executable);
        setExecutor(n_executor);
        setInstaller(n_installer);
        setName(n_name);
        setGenericName(n_genericName);
        setCategories(n_categories);
        setIcons(n_icons);
        setDescription(n_desc);
        setArch(n_archs.toArray(new String[0]));
        setOs(n_os.toArray(new String[0]));
        setOsdist(n_osdist.toArray(new String[0]));
        setPlatform(n_platform.toArray(new String[0]));
        setDesktopEnvironment(n_desktopEnvironment.toArray(new String[0]));
        setDependencies(n_deps.toArray(new NutsDependency[0]));
        setStandardDependencies(n_sdeps.toArray(new NutsDependency[0]));
        setProperties(n_props);
        return this;
    }

    @Override
    public NutsDescriptorBuilder applyProperties(Map<String, String> properties) {
        Function<String, String> map = new MapToFunction<>(properties);

        NutsId n_id = getId().builder().apply(map).build();
//        String n_alt = CoreNutsUtils.applyStringProperties(getAlternative(), map);
        String n_packaging = CoreNutsUtils.applyStringProperties(getPackaging(), map);
        String n_name = CoreNutsUtils.applyStringProperties(getName(), map);
        String n_desc = CoreNutsUtils.applyStringProperties(getDescription(), map);
        NutsArtifactCall n_executor = getExecutor();
        NutsArtifactCall n_installer = getInstaller();
        Map<String, String> n_props = new HashMap<>();
        Map<String, String> properties1 = getProperties();
        if (properties1 != null) {
            for (Map.Entry<String, String> ee : properties1.entrySet()) {
                n_props.put(CoreNutsUtils.applyStringProperties(ee.getKey(), map), CoreNutsUtils.applyStringProperties(ee.getValue(), map));
            }
        }

        LinkedHashSet<NutsDependency> n_deps = new LinkedHashSet<>();
        for (NutsDependency d2 : getDependencies()) {
            n_deps.add(applyNutsDependencyProperties(d2, map));
        }

        LinkedHashSet<NutsDependency> n_sdeps = new LinkedHashSet<>();
        for (NutsDependency d2 : getStandardDependencies()) {
            n_sdeps.add(applyNutsDependencyProperties(d2, map));
        }

        this.setId(n_id);
//        this.setAlternative(n_alt);
        this.setParents(getParents());
        this.setPackaging(n_packaging);
        this.setExecutable(isExecutable());
        this.setExecutor(n_executor);
        this.setInstaller(n_installer);
        this.setName(n_name);
        this.setDescription(n_desc);
        this.setGenericName(CoreNutsUtils.applyStringProperties(getGenericName(), map));
        this.setIcons(
                getIcons().stream()
                        .map(
                                x->CoreNutsUtils.applyStringProperties(x, map)
                        ).collect(Collectors.toList())
        );
        this.setCategories(
                getCategories().stream()
                        .map(
                                x->CoreNutsUtils.applyStringProperties(x, map)
                        ).collect(Collectors.toList())
        );
        this.setArch(CoreNutsUtils.applyStringProperties(getArch(), map));
        this.setOs(CoreNutsUtils.applyStringProperties(getOs(), map));
        this.setOsdist(CoreNutsUtils.applyStringProperties(getOsdist(), map));
        this.setPlatform(CoreNutsUtils.applyStringProperties(getPlatform(), map));
        this.setDesktopEnvironment(CoreNutsUtils.applyStringProperties(getDesktopEnvironment(), map));
        this.setDependencies(n_deps.toArray(new NutsDependency[0]));
        this.setStandardDependencies(n_sdeps.toArray(new NutsDependency[0]));
        this.setProperties(n_props);
        return this;
    }

    private NutsId applyNutsIdProperties(NutsId child, Function<String, String> properties) {
        return session.getWorkspace().id().builder()
                .setRepository(CoreNutsUtils.applyStringProperties(child.getRepository(), properties))
                .setGroupId(CoreNutsUtils.applyStringProperties(child.getGroupId(), properties))
                .setArtifactId(CoreNutsUtils.applyStringProperties(child.getArtifactId(), properties))
                .setVersion(CoreNutsUtils.applyStringProperties(child.getVersion().getValue(), properties))
                .setOs(CoreNutsUtils.applyStringProperties(child.getOs(), properties))
                .setOsdist(CoreNutsUtils.applyStringProperties(child.getOsdist(), properties))
                .setArch(CoreNutsUtils.applyStringProperties(child.getArch(), properties))
                .setClassifier(CoreNutsUtils.applyStringProperties(child.getClassifier(), properties))
                .setPackaging(CoreNutsUtils.applyStringProperties(child.getPackaging(), properties))
                .setProperties(CoreNutsUtils.applyMapProperties(child.getProperties(), properties))
                .build();
    }

    private NutsDependency applyNutsDependencyProperties(NutsDependency child, Function<String, String> properties) {
        NutsId[] exclusions = child.getExclusions();
        for (int i = 0; i < exclusions.length; i++) {
            exclusions[i] = applyNutsIdProperties(exclusions[i], properties);
        }
        return session.getWorkspace().dependency().builder()
                .setRepository(CoreNutsUtils.applyStringProperties(child.getRepository(), properties))
                .setGroupId(CoreNutsUtils.applyStringProperties(child.getGroupId(), properties))
                .setArtifactId(CoreNutsUtils.applyStringProperties(child.getArtifactId(), properties))
                .setVersion(CoreNutsUtils.applyStringProperties(child.getVersion(), properties,session.getWorkspace()))
                .setClassifier(CoreNutsUtils.applyStringProperties(child.getClassifier(), properties))
                .setScope(CoreNutsUtils.applyStringProperties(child.getScope(), properties))
                .setOptional(CoreNutsUtils.applyStringProperties(child.getOptional(), properties))
                .setOs(CoreNutsUtils.applyStringProperties(child.getOs(), properties))
                .setArch(CoreNutsUtils.applyStringProperties(child.getArch(), properties))
                .setType(CoreNutsUtils.applyStringProperties(child.getType(), properties))
                .setExclusions(exclusions)
                .setProperties(CoreNutsUtils.applyStringProperties(child.getPropertiesQuery(), properties))
                .build();
    }

    @Override
    public NutsDescriptorBuilder replaceDependency(Predicate<NutsDependency> filter, UnaryOperator<NutsDependency> converter) {
        if (converter == null) {
            return this;
        }
        ArrayList<NutsDependency> dependenciesList = new ArrayList<>();
        for (NutsDependency d : getDependencies()) {
            if (filter == null || filter.test(d)) {
                d = converter.apply(d);
                if (d != null) {
                    dependenciesList.add(d);
                    ;
                }
            } else {
                dependenciesList.add(d);
            }
        }
        this.dependencies = dependenciesList;
        return this;
    }

    @Override
    public NutsDescriptorBuilder removeDependency(Predicate<NutsDependency> dependency) {
        if (dependency == null) {
            return this;
        }
        for (Iterator<NutsDependency> it = dependencies.iterator(); it.hasNext(); ) {
            NutsDependency d = it.next();
            if (dependency.test(d)) {
                //do not add
                it.remove();
            }
        }
        return this;
    }

    @Override
    public String getGenericName() {
        return genericName;
    }

    @Override
    public NutsDescriptorBuilder setGenericName(String name) {
        this.genericName=name;
        return this;
    }

    @Override
    public List<String> getIcons() {
        return icons;
    }

    @Override
    public NutsDescriptorBuilder setIcons(List<String> icons) {
        this.icons =icons==null?new ArrayList<>() :new ArrayList<>(icons);
        return this;
    }

    @Override
    public List<String> getCategories() {
        return categories;
    }

    @Override
    public NutsDescriptorBuilder setCategories(List<String> categories) {
        this.categories =categories==null?new ArrayList<>() : new ArrayList<>(categories);
        return this;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        DefaultNutsDescriptorBuilder that = (DefaultNutsDescriptorBuilder) o;
        return executable == that.executable && application == that.application && Objects.equals(id, that.id) && Arrays.equals(parents, that.parents) && Objects.equals(packaging, that.packaging) && Objects.equals(executor, that.executor) && Objects.equals(installer, that.installer) && Objects.equals(name, that.name) && Objects.equals(icons, that.icons) && Objects.equals(categories, that.categories) && Objects.equals(genericName, that.genericName) && Objects.equals(description, that.description) && Objects.equals(arch, that.arch) && Objects.equals(os, that.os) && Objects.equals(osdist, that.osdist) && Objects.equals(platform, that.platform) && Objects.equals(desktopEnvironment, that.desktopEnvironment) && Objects.equals(locations, that.locations) && Objects.equals(classifierMappings, that.classifierMappings) && Objects.equals(dependencies, that.dependencies) && Objects.equals(standardDependencies, that.standardDependencies) && Objects.equals(properties, that.properties) && Objects.equals(session, that.session);
    }

    @Override
    public int hashCode() {
        int result = Objects.hash(id, packaging, executable, application, executor, installer, name, icons, categories, genericName, description, arch, os, osdist, platform, desktopEnvironment, locations, classifierMappings, dependencies, standardDependencies, properties, session);
        result = 31 * result + Arrays.hashCode(parents);
        return result;
    }
}
