package net.sqlind;

import java.io.InputStream;
import java.lang.reflect.Method;
import java.net.URL;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.StringTokenizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtMethod;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import net.sqlind.SQLQueryMapper.GenericQueryHandler.BeanWiringBehavior;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;


/**
 * SQLind
 * @since 2011
 * @version 1.1.8
 * @author Rives Davy.
 */
public class SQLQueryMapper {

    private int LRU_CACHE_SIZE =50 ;
   
    /**
     * Defines the LRU policy cache size
     * @param numberOfQuery
     */
    public void setCacheSize(int numberOfQuery) {
        LRU_CACHE_SIZE = numberOfQuery;
    }
   
    /**
     * Remove all entries from the LRU cache
     */
    public void clearCache() {
        synchronized (queryMap) {
            for (String key:lru)
                queryMap.remove(key);
            lru.clear();
        }
    }
   
    protected static Log log = LogFactory.getLog(SQLQueryMapper.class);
   
    private static SQLQueryMapper instance;

    
    private static synchronized  void initInstance(){
    	if (instance==null){
    		instance = new SQLQueryMapper();
		    log.info("SQLind Initialized");
    	}
    }
    /**
     * @return the singleton manager instance of SQLQueryMapper
     */
    public static SQLQueryMapper getInstance() {
        if (instance == null){
        	initInstance();
        
        }
        return instance;
    }

    // SQLMapper map
    protected static HashMap<String, SQLQueryMapper> instances = new HashMap<String, SQLQueryMapper>();
   
    // XML constant
    private static String ID = "id";
    private static String PARAMS = "params";
    private static String POLICY = "policy";

    private enum Tags {
        queries, query, var, include, section, inject
    }

    private enum WiringPolicies {
        debug,reload,lru,fast
    }

    protected enum Errors {
        ERROR_UNKNOWN, ERROR_INIT_WIRING, ERROR_DO_WIRING, ERROR_SET_WIRING, ERROR_SQL, ERROR_INIT_QUERY, ERROR_INIT_QUERIES
    }

    // Query transformation/manipulation patterns
   
    private static String INJECT_JOKER = "<SQLind-InjectPoint/>";
   
    private static Pattern INJECT_JOKER_PATTERN = Pattern.compile(INJECT_JOKER);
   
    private static Pattern SQL_FROM_PATTERN = Pattern.compile(
            "from\\s((?!where|union|left|right|full|outer|join|inner|\\)|$).)+(where|union|left|right|full|outer|join|inner|\\)|$)",
            Pattern.CASE_INSENSITIVE+Pattern.DOTALL);
   
    private static Pattern SQL_JOIN_PATTERN = Pattern.compile(
            "join\\s((?!on|where|union|left|right|full|outer|join|inner).)+(on|where|union|left|right|full|outer|join|inner)",
            Pattern.CASE_INSENSITIVE+Pattern.DOTALL);
   
    private static Pattern SQL_SELECT_PATTERN = Pattern.compile("select\\s*(distinct){0,1}\\s*(((?!from).)+)from.*",
            Pattern.CASE_INSENSITIVE);
    
    private static Pattern SQL_COLUMN_PATTERN = Pattern.compile("(^|,)(((?!as).)+as\\s*[^\\s,]+|[^,]+)",Pattern.CASE_INSENSITIVE);
    private static Pattern SQL_STAT_PATTERN = Pattern.compile("([a-z\\._0-9]+\\s*(,|$)|as\\s+[^\\s,]+)",Pattern.CASE_INSENSITIVE);
    private static Pattern SQL_ALIAS_PATTERN = Pattern.compile("([a-z0-9_]+)\\s*,?\\s*$",Pattern.CASE_INSENSITIVE);
    
 
    private final static Tracer console = new Tracer(){
		public void trace(String data) {System.out.println(data);}};
    
	private final static Tracer logger = new Tracer(){
		public void trace(String data) {log.info(data);}};
	    	
		
	/**
     * Util interface to handle logging stream
     */
    public interface Tracer {public void trace(String data);}
    
    /**
     * Structure class to define an
     * insert point in a query
     */
    public class InjectPoint {
        private InjectPoint(String name, Integer idx) {
            this.name = name;
            this.idx = idx;
        }

        private String name;
        private Integer idx;
        public String toString(){
            return idx+name;
        }
    }

    /**
     * Generic query result fetching interface
     */
    public interface QueryBehavior {
        public void fetch(ResultSet rs) throws Exception;
    }

    /**
     * Generic exception throws by the API. 
     */
    public class SQLMapperException extends Exception {
        private static final long serialVersionUID = 1L;

        public SQLMapperException(String msg, Throwable originalException) {
            super(msg, originalException);
        }
        
        public SQLMapperException(String msg) {
            super(msg);
        }
    }

    private static final long serialVersionUID = 1L;

    private String file;

    private HashMap<String, SQLQueryTemplate> queryMap = new HashMap<String, SQLQueryTemplate>();
    private List<String> lru = new ArrayList<String>();

    private static ThreadLocal<Context> context = new ThreadLocal<Context>() {
        protected synchronized Context initialValue() {
            return getInstance().new Context();
        }
    };

    /**
     * Retrieve a given SQL query
     * @param file the template file
     * @param id the query id
     * @param params optional sections id to enable
     * @return SQLQuery object
     */
    public SQLQueryHandler getSQLQuery(String file,String schema, String id, String... params)
            throws SQLMapperException {
        if (!instances.containsKey(file))
            synchronized (instances) {
                if (!instances.containsKey(file))
                	instances.put(file, new SQLQueryMapper(file));
            }
        return new SQLQueryHandler(instances.get(file).getQuery(schema, id, params));
    }
   
    protected SQLQueryMapper(String file) throws SQLMapperException {
        try {
            SchemaFactory factory = SchemaFactory.newInstance("http://www.w3.org/2001/XMLSchema");
            URL url = getClass().getClassLoader().getResource("net/sqlind/sqlind.xsd");
             InputStream fileIS= getClass().getClassLoader().getResourceAsStream(file);
            Schema schema = factory.newSchema(url);
            Validator validator = schema.newValidator();
            Source source = new StreamSource(fileIS);
            validator.validate(source);
        } catch (Exception e) {
            //throw new SQLMapperException("XML query definition validation failed", e);
            log.warn("failed to validate query file "+file,e);
        }
        this.file = file;
    }

    protected SQLQueryMapper() {
    }

    protected static Context getContext() {
        return context.get();
    }

    class Context {
        private StringBuilder queryKey = new StringBuilder();
        private StringBuilder strBuilder = new StringBuilder();
        String[] lastValues;

        private void clearStrBuilder() {
            strBuilder.delete(0, strBuilder.length());
        }

    }

    public interface SQLBeanAccess {
        public Object buildFromResultSet(BeanWiringBehavior behavior, ResultSet rs, Object bean, String[] lastValues)
                throws Exception;
    }

    protected static String shrinkSQL(String queryStr,List<InjectPoint> ips) {
       
            StringBuilder sb;
                sb = new StringBuilder(queryStr);
                int c=0;
                for (InjectPoint ip : ips)
                    sb.insert(ip.idx+(c++)*INJECT_JOKER.length(), INJECT_JOKER);
                trimQuery(sb);
                Matcher m=INJECT_JOKER_PATTERN.matcher(sb.toString());
                for (InjectPoint ip : ips){
                    m.find();
                    ip.idx=m.start();
                }
                sb = new StringBuilder(sb.toString().replaceAll(INJECT_JOKER,""));
               
           
           return sb.toString();
        }

    private static void trimQuery(StringBuilder sb) {
        StringTokenizer st = new StringTokenizer(sb.toString(),"\n");
        sb.delete(0,sb.length());
        while(st.hasMoreElements())
            sb.append((((String)st.nextElement())).trim()).append(" ");
    }

   
    // ------------------------- Query handler ---------------------------------
   
    public class SQLQueryHandler extends GenericQueryHandler{

        protected SQLQueryHandler(SQLQueryTemplate template) {
            super(template);
        }       
       
        /**
         * Perform a query using the 'one bean per rupture' wiring behavior
         * @param cnx the DB connection instance
         * @throws SQLMapperException
         * @return a T bean list
         */
        @SuppressWarnings("unchecked")
        public <T> List<T> getBeanList(Connection cnx,Class beanTypeClazz) throws SQLMapperException {
            List<T> list = new ArrayList<T>();
            performSelectQuery(cnx, new BeanListWiringBehavior<T>(beanTypeClazz, list));
            return list;
        }

   
        /**
         * Perform a query using the 'one bean per record' wiring behavior
         * @param cnx the DB connection instance
         * @throws SQLMapperException
         * @return a T bean list
         */
        @SuppressWarnings("unchecked")
        public <T> List<T> getSingleBeanList(Connection cnx,Class beanTypeClazz) throws SQLMapperException {
            List<T> list = new ArrayList<T>();
            performSelectQuery(cnx, new SingleBeanListWiringBehavior<T>(beanTypeClazz, list));
            return list;
        }
       
        /**
         * Perform a query and return a simple type result list
         * @param cnx the DB connection instance
         * @throws SQLMapperException
         * @return a List of T type instances.
         */
        @SuppressWarnings("unchecked")
        public <T>List<T> getSimpleResultList(Connection cnx,Class simpleTypeClazz) throws SQLMapperException {
            try {
                dumpQuery();
                ps = cnx.prepareStatement(performInjection(injectPointValues));
                fillInPreparedStatement(ps, parameterValues);
                ResultSet rs =ps.executeQuery();
                List<T> list = new ArrayList<T>();
                while(rs.next())
                    list.add((T)rs.getObject(1));
                return list;
            } catch (Exception e) {
                throw template.getException(Errors.ERROR_SQL, e,getSQLWithParameters());
            } finally {
                close();
            }
        }
       
        /**
         * Perform a query and return a single result from the first record
         * @param cnx the DB connection instance
         * @throws SQLMapperException
         * @return an instance of T if found, null otherwise
         */
        @SuppressWarnings("unchecked")
        public <T>T getSimpleResult(Connection cnx) throws SQLMapperException {
            try {
                dumpQuery();
                ps = cnx.prepareStatement(performInjection(injectPointValues));
                fillInPreparedStatement(ps, parameterValues);
                ResultSet rs =ps.executeQuery();
                if (rs.next())
                    return (T)rs.getObject(1);
                return null;
            } catch (Exception e) {
                throw template.getException(Errors.ERROR_SQL, e,getSQLWithParameters());
            } finally {
                close();
            }
        }
        /**
         * Perform an update over DB using current query definition.
         * @param cnx the DB connection instance
         * @throws SQLMapperException
         */
        public void performUpdateQuery(Connection cnx) throws SQLMapperException {
            try {
                dumpQuery();
                ps = cnx.prepareStatement(performInjection(injectPointValues));
                fillInPreparedStatement(ps, parameterValues);
                ps.executeUpdate();
            } catch (Exception e) {
                throw template.getException(Errors.ERROR_SQL, e,getSQLWithParameters());
            } finally {
                close();
            }
        }

       
        /**
         * Perform a select over DB using current query definition.
         * @param cnx the DB connection instance
         * @throws SQLMapperException
         */
        public void performSelectQuery(Connection cnx, QueryBehavior behaviour) throws SQLMapperException {
            try {
                dumpQuery();
                ps = cnx.prepareStatement(performInjection(injectPointValues));
                fillInPreparedStatement(ps, parameterValues);
                ResultSet rs = ps.executeQuery();
                values = new Object[template.getColumns().size()];
                getContext().lastValues = new String[values.length];
                while (rs.next())
                    behaviour.fetch(rs);
            } catch (SQLMapperException sqlme) {
                throw sqlme;
            } catch (Exception e) {
                throw template.getException(Errors.ERROR_SQL, e,getSQLWithParameters());
            } finally {
                close();
            }
        }
        protected void fillInPreparedStatement(PreparedStatement ps, HashMap<String, Object> pValues) throws SQLException {
            int j = 1;
            for (String paramId : template.getParameters())
                ps.setObject(j++, pValues.get(paramId));
        }
       
    }
    /**
     * Handle a query execution context
     */
    public class GenericQueryHandler {
        protected SQLQueryTemplate template;
        protected PreparedStatement ps = null;
        protected ResultSet rs = null;
        Object[] values = null;
        protected HashMap<String, Object> parameterValues = new HashMap<String, Object>(0);
        protected HashMap<String, String> injectPointValues = new HashMap<String, String>(0);
        //protected boolean debug=false;
        private Tracer tracer;
        /**
         * @return the list of insert points of the query.
         */
        public List<InjectPoint> getInjectPoints() {
            return template.getInjectPoints();
        }

        protected GenericQueryHandler(SQLQueryTemplate template) {
            if(template.getWiringPolicy()==WiringPolicies.reload)
                this.tracer=logger;
            else if(template.getWiringPolicy()==WiringPolicies.debug)
            	this.tracer=console;
            this.template = template;
        }

        /**
         * Generic query result set fetching implementation
         */
        public abstract class ResultSetFecthedBehaviour implements QueryBehavior {
            public void fetch(ResultSet rs) throws Exception {
                doForEach(rs);
            }
            /**
             * Called for each result.
             * @param rs
             */
            public abstract void doForEach(ResultSet rs) throws Exception;
        }

        /**
         * Generic query result bean wiring implementation
         */
        public abstract class BeanWiringBehavior<T> implements QueryBehavior {
            protected Class<T> clazz;
            protected T objBean;
            SQLQuery.WiringPolicy<T> wp;
            protected ResultSet rs;

            public BeanWiringBehavior(Class<T> clazz) throws SQLMapperException {
                this.clazz = clazz;
                wp = template.getWiringPolicyClass(clazz);
            }

            public void fetch(ResultSet rs) throws Exception {
                this.rs = rs;
                T bean = (T) wp.wire(this, rs, values);
                objBean = bean;
            }
            protected T getNewBean() throws Exception {
                return (T) clazz.newInstance();
            }
            /**
             * Call each time first level field values have changed
             * @param bean the first level object
             */
            public abstract void doForEachRupture(T bean);
            /**
             * Call each time sub level field values have changed
             * @param bean the sub level object
             */
            public abstract void doForEachSubRupture(Object bean) ;
            /**
             * Call for each records
             * @param bean the first level object
             */
            public abstract void doForEachRecord(T bean) ;
           
           
        }

        /**
         * Generic query result bean list wiring implementation.
         * Bean is added only if a first level rupture occurred.
         */
        public  class BeanListWiringBehavior<T> extends BeanWiringBehavior<T> {
            protected List<T> list;

            public BeanListWiringBehavior(Class<T> clazz, List<T> list) throws SQLMapperException {
                super(clazz);
                this.list = list;
            }
            @Override
            public void doForEachRecord(T bean) {}
            @Override
            public void doForEachRupture(T bean) {list.add(bean);}
            @Override
            public void doForEachSubRupture(Object bean) {}
        }
       
        /**
         * Generic query result bean list wiring implementation.
         * Bean list size will match the number of records retrieved by query.
         */
        public  class SingleBeanListWiringBehavior<T> extends BeanListWiringBehavior<T> {
            public SingleBeanListWiringBehavior(Class<T> clazz, List<T> list) throws SQLMapperException {
                super(clazz, list);
            }
            public void doForEachRupture(T bean) {}
            @Override
            public void doForEachRecord(T bean) {
           
                list.add(bean);
            }
       
        }

       
   
        protected void dumpQuery() {
            if (tracer!=null){
                tracer.trace("Processing query '"+template.getID()+"' : \n"+getSQLWithParameters());
                template.setSQL(shrinkSQL(template.getSQL(),template.getInjectPoints()));
            }
           
        }

        /**
         * @return the sql query string with valued parameters.
         */
        public String getSQLWithParameters(){
            String query = getInjectedSQL();
            List<String> params =template.getParameters();
            if (params!=null)
                for (String param : params){
                    Object paramValue =parameterValues.get(param);
                    String paramPattern =Pattern.quote(getParamBindingString(param));
                    if (paramValue==null)
                        query = query.replaceFirst(paramPattern,"!");
                    else{
                        if (paramValue instanceof String || paramValue instanceof Date)
                            query = query.replaceFirst(paramPattern,"'"+paramValue.toString()+"'");
                        else
                            query = query.replaceFirst(paramPattern,paramValue.toString());
                    }
                   
                }
            query = query.replaceAll("\n[\\s\\t]*\n","\n"); // clear blank lines
            return query;
        }
         /**
         * Close the query context
         */
        public void close() {
            try {
                if (ps != null)
                    ps.close();
            } catch (Exception e) {/* Nothing to do */
            }
            try {
                if (rs != null)
                    rs.close();
            } catch (Exception e) {/* Nothing to do */
            }
        }

        /**
         * Map a value to the given parameter
         * @param paramId
         * @param value
         * @throws SQLMapperException
         */
        public void setParameter(String paramId, Object value) throws SQLMapperException {
            parameterValues.put(paramId, value);
        }

        /**
         * Map a value to a given insert point.
         * @param injectPointId
         * @param value
         * @throws SQLMapperException
         */
        public void setInjection(String injectPointId, String value) throws SQLMapperException {
            injectPointValues.put(injectPointId, value);
        }

        /**
         * @return the query string before injection
         */
        public String getSQL() {
            return template.getSQL();
        }
       
        /**
         * @return the query string after injection
         */
        public String getInjectedSQL() {
            return performInjection(injectPointValues);
        }

       
        /**
         * Try to set the value of query parameters using the given bean
         * @param bean
         * @throws SQLMapperException
         */
        public void fillInParametersFromBean(Object bean) throws SQLMapperException {
            try {
                for (String paramId : template.getParameters())
                    parameterValues.put(paramId, BeanUtils.getProperty(bean, paramId));
            } catch (Exception e) {
                template.getException(Errors.ERROR_SET_WIRING, e, bean.getClass().getName());
            }
        }

        protected String performInjection(HashMap<String, String> ipValues) {
            getContext().clearStrBuilder();
            StringBuilder sb = getContext().strBuilder;
            int idx = 0;
           
            for (InjectPoint ip : template.getInjectPoints()) {
                sb.append(template.getSQL().substring(idx, ip.idx));
                String value = ipValues.get(ip.name);
                if (value != null)
                    sb.append(value);
                idx = ip.idx;
            }
            sb.append(template.getSQL().substring(idx, template.getSQL().length()));

            return sb.toString();

        }

    }

    // -------------------- Templated query------------------

    /**
     * Represents a SQL template definition
     */
    public interface SQLQueryTemplate {
        public List<String> getParameters();
        public List<InjectPoint> getInjectPoints();
        public String getSQL();
        public void setSQL(String queryString);
        public WiringPolicies getWiringPolicy();
        public String getID();
        public List<String> getColumns();
        public <Z> SQLQuery.WiringPolicy<Z> getWiringPolicyClass(Class<Z> clazz) throws SQLMapperException;
        public SQLMapperException getException(Errors err, Throwable originalException, String... infos);

    }

   
    private class SQLIncludedQuery extends SQLQuery {
        @Override
        public void shrink() {}
        public SQLIncludedQuery(String file, String schema, String id, String... params) throws SQLMapperException {
            super();
            createTemplate(file, schema, id, params);
        }
    }

     // Templated Query instance class
    private class SQLQuery extends DefaultHandler implements SQLQueryTemplate {
        private ParserBehaviour startTargParser;
        private ParserBehaviour endTagParser;
   
        String schema = null;
        WiringPolicies wiring = null;
        private List<InjectPoint> injectPoints = new ArrayList<InjectPoint>(0);
        private StringBuilder template = null;
        private String id;
        private String libId;
       
        private boolean found = false;
        private int ignore = 0;
        private String sql;

        private List<String> options = null;

        private List<String> parameters = new ArrayList<String>(0);
        private List<String> columnNames = new ArrayList<String>(5);
        @SuppressWarnings("unchecked")
        HashMap<String, WiringPolicy> wiringCache = null;

        // ERROR--------------------

        private class Info {
            String[] infos;
            StringBuilder ctx = new StringBuilder();

            Info(String[] infos) {
                ctx.append("\nQuery     :").append(id);
                ctx.append("\nSQL       :").append(getContext().strBuilder);
                ctx.append("\nparam     :").append(parameters);
                ctx.append("\ncolumns   :").append(columnNames);
                this.infos = infos;
            }

            String get(int i) {
                if (i < infos.length)
                    return infos[i];
                else
                    return "'?'";
            }
        }

        /**
         * Exception factory
         */
        public SQLMapperException getException(final Errors type, Throwable e, String... infos) {
            Info info = new Info(infos);
            e.printStackTrace();
            switch (type) {

            case ERROR_INIT_WIRING:
                return new SQLMapperException(info.ctx + "\nUnable to init wiring policy for bean " + info.get(0), e);
            case ERROR_DO_WIRING:
                return new SQLMapperException(info.ctx + "\nUnable to wire bean " + info.get(0), e);
            case ERROR_SET_WIRING:
                return new SQLMapperException(info.ctx
                        + "\nWiring is not activated on this query, please add a wiring policy to xml", e);
            case ERROR_SQL:
                return new SQLMapperException("\nUnable to perform query :\n "+info.get(0), e);
            case ERROR_INIT_QUERY:
                return new SQLMapperException(info.ctx
                        + "Unable to create query object from xml definition (check SQL syntax)", e);
            case ERROR_INIT_QUERIES:
                return new SQLMapperException(info.ctx + "Unable to parse queries xml definition", e);
            default:
                return new SQLMapperException(info.ctx + "Unexpected error : " + info.get(0), e);
            }
        }

        // ----------------- Wiring -------------------

        @SuppressWarnings("unchecked")
        private final Class[] NOARGType = new Class[0];
        private final Object[] NOARG = new Object[0];

        // List link
        private class ListWireringLink extends WireringLink {
            public ListWireringLink(Method setter, SQLind info) throws Exception {
                super(setter, info);
            }

            public <T> Object doIt(BeanWiringBehavior behavior, String[] strValues, T bean, Object[] values)
                    throws Exception {
                List<Object> targetList = (List) getter.invoke(bean, NOARG);
                Object target = null;
                if (targetList == null) {
                    targetList = new ArrayList<Object>();
                    setter.invoke(bean, targetList);

                } else
                    target = targetList.get(targetList.size() - 1);
                Object res = subWirerer.doWiring(behavior,true, strValues, target, values);
                if (res != target)
                    targetList.add(res);
                return target;
            }
        }

        // Aggregation link
        private class WireringLink {
            Method getter;
            Method setter;
            Class<? extends Object> subClazz;
            ReflectionWiring subWirerer;

            public WireringLink(Method setter, SQLind info) throws Exception {
                subClazz = info.link();
                getter = transformSetter2Getter(setter);
                this.setter = setter;
                subWirerer = new SQLQuery.ReflectionWiring(subClazz);
            }

            public <T> Object doIt(BeanWiringBehavior behavior, String[] strValues, T bean, Object[] values)
                    throws Exception {
                Object target = getter.invoke(bean, NOARG);
                Object res = subWirerer.doWiring(behavior,false, strValues, target, values);
                if (target != res)
                    setter.invoke(bean, res);
                return target;
            }
        }

        /**
         * Generic query result/bean fields wiring implementation
         * @param <Z>
         */
        public abstract class WiringPolicy<Z> {

            Class<Z> clazz;

            protected WiringPolicy(Class<Z> clazz) throws Exception {
                this.clazz = clazz;

            };

            protected void init() throws Exception {
                before();
                for (Method method : clazz.getMethods()) {
                    SQLind data = method.getAnnotation(SQLind.class);
                    if (data != null)
                        initSetter(data, method);
                }
                after();
            }

            protected void initSetter(SQLind data, Method method) throws Exception {
                for (String colName : data.column()) {
                    int idx = columnNames.indexOf(colName.toLowerCase());
                    if (idx != -1)
                        doColumnField(data, method, idx);
                }
                Class<? extends Object> clazz = data.link();
                if (clazz != Object.class) {
                    Class<? extends Object> rClazz = method.getParameterTypes()[0];
                    if (rClazz.getName().equals(clazz.getName()))
                        doLinkField(method, data);
                    else
                        doLinkList(method, data);
                }
            }

            protected abstract void doColumnField(SQLind data, Method method, int idx) throws Exception;

            protected abstract void doLinkField(Method method, SQLind data) throws Exception;

            protected abstract void doLinkList(Method method, SQLind data) throws Exception;

            protected abstract void before() throws Exception;

            protected abstract void after() throws Exception;

            protected abstract Z wire(BeanWiringBehavior<Z> behavior, ResultSet rs, Object... values)
                    throws SQLMapperException;
        }

        /**
         * @return the Wiring policy instance for the given class
         */
        @SuppressWarnings("unchecked")
        public <Z> WiringPolicy<Z> getWiringPolicyClass(Class<Z> clazz) throws SQLMapperException {
            try {
                getContext().clearStrBuilder();
                StringBuilder sb =getContext().strBuilder;
                sb.append(clazz.getName()).append(".").append(libId).append(".").append(id).append(String.valueOf(options));
                String key = sb.toString();
                if (!wiringCache.containsKey(key)) {
                    synchronized (instance) {
                        if (!wiringCache.containsKey(key)) {
                            switch (wiring) {
                            case fast:
                                wiringCache.put(key, new SQLQuery.TransformationWiring<Z>(clazz));
                                break;
                            default:
                                wiringCache.put(key, new SQLQuery.ReflectionWiring(clazz));
                            }
                        }
                    }
                }
                return (WiringPolicy<Z>) wiringCache.get(key);
            } catch (Exception e) {
                if (wiring == null)
                    throw getException(Errors.ERROR_SET_WIRING, e);
                else
                    throw getException(Errors.ERROR_INIT_WIRING, e, clazz.getName());
            }
        }

        private Method transformSetter2Getter(Method setter) throws NoSuchMethodException {
            String getterName = "get" + setter.getName().substring(3, setter.getName().length());
            return (setter.getDeclaringClass()).getMethod(getterName, NOARGType);
        }

       
        // Deep reflect wiring
        private class ReflectionWiring extends WiringPolicy {
            HashMap<Integer, Method> setterMap;
            List<WireringLink> linkMap;
            List<WireringLink> fieldMap;

            protected <Z> ReflectionWiring(Class<Z> clazz) throws Exception {
                super(clazz);
                init();
            }

            private Object doWiring(BeanWiringBehavior beahavior,boolean one2Many, String[] strValues, Object bean, Object... values)
                    throws Exception {
                boolean rupture = isRupture(strValues, bean);

                if (rupture) {
                    bean = clazz.newInstance();
                    for (Integer i : setterMap.keySet())
                        setterMap.get(i).invoke(bean, values[i]);
                    if(one2Many)
                        beahavior.doForEachSubRupture(bean);
                }
                wireLinks(beahavior, strValues, bean, values);

                return bean;
            }

            private void wireLinks(BeanWiringBehavior behavior, String[] strValues, Object bean, Object... values)
                    throws Exception {
                for (WireringLink field : fieldMap)
                    field.doIt(behavior, strValues, bean, values);
                for (WireringLink link : linkMap)
                    link.doIt(behavior, strValues, bean, values);
            }

            private boolean isRupture(String[] strValues, Object bean) {
                boolean rupture = false;

                String[] lastValues = getContext().lastValues;
                if (bean != null) {
                    rupture = isRuptureCase(strValues, lastValues);
                } else
                    rupture = true;
                return rupture;
            }

            private boolean isRuptureCase(String[] strValues, String[] lastValues) {

                for (Integer i : setterMap.keySet()) {
                    if (!strValues[i].equals(lastValues[i]))
                        return true;
                }
                for (WireringLink field : fieldMap) {
                    if (field.subWirerer.isRuptureCase(strValues, lastValues))
                        return true;
                }

                return false;
            }

            protected Object wire(BeanWiringBehavior behavior, ResultSet rs, Object... values)
                    throws SQLMapperException {
                try {
                    String[] strValues = new String[values.length];
                    for (int i = 0; i < values.length; i++) {
                        values[i] = rs.getObject(i + 1);
                        if (values[i] != null)
                            strValues[i] = values[i].toString();
                        else
                            strValues[i] = "NULL";
                    }
                    boolean rupture = isRupture(strValues, behavior.objBean);
                    Object ret = null;
                    if (rupture) {
                        ret = clazz.newInstance();
                        for (Integer i : setterMap.keySet()){
                            try{
                                setterMap.get(i).invoke(ret, values[i]);
                            }catch (Exception e) {
                                throw new Exception("Unable to set value : "+setterMap.get(i)+" --> "+values[i]);
                            }
                        }
                        behavior.doForEachRupture(ret);
                    } else
                        ret = behavior.objBean;
                    wireLinks(behavior, strValues, ret, values);
                    behavior.doForEachRecord(ret);
                    getContext().lastValues = strValues.clone();
                    return ret;

                } catch (Exception e) {
                    throw getException(Errors.ERROR_DO_WIRING, e, clazz.getName());
                }
            }

            @Override
            protected void before() {
                linkMap = new ArrayList<WireringLink>();
                fieldMap = new ArrayList<WireringLink>();
                setterMap = new HashMap<Integer, Method>();
            }

            @Override
            protected void after() {
            }

            @Override
            protected void doColumnField(SQLind data, Method method, int idx) throws Exception {
                setterMap.put(idx, method);
            }

            @Override
            protected void doLinkField(Method method, SQLind data) throws Exception {
                fieldMap.add(new WireringLink(method, data));
            }

            @Override
            protected void doLinkList(Method method, SQLind data) throws Exception {
                linkMap.add(new ListWireringLink(method, data));
            }
        }

        private class TransformationSubWiring<Z> extends TransformationWiring {

            protected TransformationSubWiring(Class clazz, List<String> beanIds) throws Exception {
                super(clazz);
                this.beanId = beanIds;
                init();
            }

            @Override
            protected void after() throws Exception {
           
                tmpSB.append("$1.doForEachSubRupture(");
                tmpSB.append(getBeanId());
                tmpSB.append(");\n");
                tmpSB.insert(0, ");\n");
                tmpSB.insert(0, getBeanId());
                tmpSB.insert(0, "().add(");
                tmpSB.insert(0, getGetterForBean());
                tmpSB.insert(0, "();\n");
                tmpSB.insert(0, clazz.getName());
                tmpSB.insert(0, "= new ");
                tmpSB.insert(0, getBeanId());
               
               
                doIfRupture(tmpSB, "else ", getBeanId(), " = ", getGetterForBean(), "().get(", getGetterForBean(),
                        "().size()-1);\n");

            }
           
        }

        private class TransformationWiring<Z> extends WiringPolicy {

            HashMap<String, List<Integer>> beanIdsmap;
            HashMap<Integer, Class> colType;
            SQLBeanAccess transfo;
            List<String> beanId;

            StringBuilder mainSB;
            StringBuilder secondarySB;
            StringBuilder tmpSB;

            protected String getAccessorString(Class type) {
                String str = type.getSimpleName();
                if (str.equals("Integer"))
                    return "Int";
                else
                    return str.substring(0, 1).toUpperCase() + str.substring(1, str.length());
            }

            protected String getPrimitive(Class type) {
                String str = type.getSimpleName();
                if (str.equals("Integer"))
                    return "int";
                else
                    return str.substring(0, 1).toLowerCase() + str.substring(1, str.length());
            }
           
            protected String getGetterForBean() {
                StringBuilder sb = new StringBuilder();
                int i = 0;
                for (; i < beanId.size() - 1; i++)
                    sb.append(beanId.get(i));
                return sb.append(".get").append(beanId.get(i).substring(3, 4).toUpperCase()).append(
                        beanId.get(i).substring(4, beanId.get(i).length())).toString();
            }

            protected String createNewBeanList(List<String> id) {
                StringBuilder sb = new StringBuilder();
                int i = 0;
                for (; i < id.size() - 1; i++)
                    sb.append(id.get(i));
                return sb.append(".set").append(id.get(i).substring(3, 4).toUpperCase()).append(
                        id.get(i).substring(4, id.get(i).length())).append("(new java.util.ArrayList());\n").toString();
            }

            protected String getBeanId() {

                if (beanId.size() == 1)
                    return "bean";
                StringBuilder sb = new StringBuilder();
                for (String bid : beanId)
                    sb.append(bid).append("_");
                return sb.toString();

            }

            private void registerCol(int id, Class classType) {
                String beanId = getBeanId();
                if (!beanIdsmap.containsKey(beanId))
                    beanIdsmap.put(beanId, new ArrayList<Integer>());
                beanIdsmap.get(beanId).add(id);
                colType.put(id + 1, classType);
            }

            protected TransformationWiring(Class<Z> clazz) throws Exception {
                super(clazz);
                init();
            }

            public Object wire(BeanWiringBehavior behavior, ResultSet rs, Object... values) throws SQLMapperException {
                try {

                     Object bean =
                     transfo.buildFromResultSet(behavior,rs,behavior.objBean,
                     getContext().lastValues);
                    return bean;
                } catch (Exception e) {
                    throw getException(Errors.ERROR_DO_WIRING, e, clazz.getName());
                }
            }

            @Override
            protected void before() throws Exception {
                colType = new HashMap<Integer, Class>();
                beanIdsmap = new HashMap<String, List<Integer>>();
                if (beanId == null) {
                    beanId = new ArrayList<String>();
                    beanId.add("bean");
                }
                secondarySB = new StringBuilder();
                tmpSB = new StringBuilder();
                mainSB = new StringBuilder();
                mainSB.append(clazz.getName()).append(" ").append(getBeanId()).append(";\n");

            }

            @Override
            protected void after() throws Exception {
                // Create transformation class
                ClassPool pool = ClassPool.getDefault();
               
               
                CtClass transfoClazz = pool.makeClass(clazz.getName() + "RSWirererOn_" +libId+"_"+ id+"_"+String.valueOf(options));
                transfoClazz.addInterface(pool.get(SQLBeanAccess.class.getName()));
                CtMethod meth = new CtMethod(pool.get(Object.class.getName()), "buildFromResultSet", new CtClass[] {
                        pool.get(BeanWiringBehavior.class.getName()), pool.get(ResultSet.class.getName()),
                        pool.get(Object.class.getName()), pool.get(String[].class.getName()) }, transfoClazz);
                meth.setExceptionTypes(new CtClass[] { pool.get(Exception.class.getName()) });
                StringBuilder methodBody = new StringBuilder("{\n");
                // Initialize variables
                fillInInitVariablesStatement(methodBody);
                // Insert process statements

                tmpSB.insert(0, "();\n");
                tmpSB.insert(0, clazz.getName());
                tmpSB.insert(0, "bean = new ");
                tmpSB.append("$1.doForEachRupture(");
                tmpSB.append(getBeanId());
                tmpSB.append(");\n");
                doIfRupture(tmpSB,"else bean = (",clazz.getName(),")$3;\n");

                 mainSB.append(secondarySB);

                methodBody.append(mainSB);
                // Update last values tables
                 doAtEnd(methodBody);
                 // return transformed bean
                methodBody.append("\nreturn bean;\n}");
            //DEBUG !
            //    System.out.println(methodBody);
                meth.setBody(methodBody.toString());
                transfoClazz.addMethod(meth);
                transfo = (SQLBeanAccess) pool.toClass(transfoClazz).newInstance();
               
            }

            protected void doAtEnd(StringBuilder methodBody) {
                 methodBody.append("$1.doForEachRecord(bean);\n");
                for (Integer id : colType.keySet())
                 methodBody.append("$4[").append(id-1).append("]=resultStr").append(id).append(";\n");
            }

            @SuppressWarnings("unchecked")
            private void fillInInitVariablesStatement(StringBuilder methodBody) {
                for (Integer colid : colType.keySet()) {
                    Class type = colType.get(colid);
                if (type.isPrimitive())  // float,double ...
                        doPrimitive(methodBody, colid, type);
                     else {
                         if (type != String.class) // Object
                            doObject(methodBody, colid, type);
                           
                         else  // String
                            doString(methodBody, colid, type);
                    }
                }
            }

            protected void doString(StringBuilder methodBody, Integer colid, Class type) {
                methodBody.append(type.getName()).append(" result").append(colid).append("= $2.get").append(
                        getAccessorString(type)).append("(").append(colid).append(");\n");
                doObjectStrVar(methodBody, colid);
            }

            protected void doObject(StringBuilder methodBody, Integer colid, Class type) {
                try {
                    type.getDeclaredField("TYPE");
                    methodBody.append(type.getName());
                        methodBody.append(" result").append(colid).append("= new ").append(type.getSimpleName()).append("( $2.get").append(
                                getAccessorString(type)).append("(").append(colid).append("));\n");
                        methodBody.append("String resultStr").append(colid).append("=String.valueOf(result").append(
                                colid).append(");\n");
                } catch (Exception e) {
                    doString(methodBody, colid, type);
                }
            }

            protected void doPrimitive(StringBuilder methodBody, Integer colid, Class type) {
                methodBody.append(type.getName());
                methodBody.append(" result").append(colid).append("= $2.get").append(
                        getAccessorString(type)).append("(").append(colid).append(");\n");
                methodBody.append("String resultStr").append(colid).append("=String.valueOf(result").append(
                        colid).append(");\n");
            }

            private void doObjectStrVar(StringBuilder methodBody, Integer colid) {
                methodBody.append("String resultStr").append(colid).append(";\n");
                methodBody.append("if (result").append(colid).append(" != null)\n");
                methodBody.append("resultStr").append(colid).append("=result").append(colid).append(
                        ".toString();\n");
                methodBody.append("else ").append("resultStr").append(colid).append("= \"NULL\";\n");
            }

            protected void doIfRupture(StringBuilder content, String... elseClause) {
                mainSB.append("if(!(");
                boolean first = true;
                for (String bId : beanIdsmap.keySet()) {

                    for (Integer colId : beanIdsmap.get(bId)) {
                        if (!first)
                            mainSB.append("&&");
                        mainSB.append("resultStr").append(colId + 1).append(".equals(");
                        mainSB.append("$4[").append(colId);
                        mainSB.append("])");
                        first = false;
                    }
                }
                mainSB.append(")){\n");
                mainSB.append(content);
                mainSB.append("}\n");
                for (String str : elseClause)
                    mainSB.append(str);
            }

            @Override
            protected void doColumnField(SQLind data, Method setter, int idx) throws Exception {
                Class ClassType = setter.getParameterTypes()[0];

                tmpSB.append(getBeanId()).append(".").append(setter.getName()).append("(result").append(idx + 1)
                        .append(");\n");
                registerCol(idx, ClassType);

            }

            @Override
            protected void doLinkField(Method setter, SQLind data) throws Exception {
                Class<? extends Object> subClazz = data.link();
                for (Method method : subClazz.getMethods()) {
                    SQLind subData = method.getAnnotation(SQLind.class);
                    if (subData != null) {
                        beanId.add(setter.getName());
                        tmpSB.append(subClazz.getName()).append(" ").append(getBeanId()).append("=new ").append(
                                subClazz.getName()).append("();");
                        initSetter(subData, method);
                        String subBeanId = getBeanId();
                        beanId.remove(beanId.size() - 1);
                        tmpSB.append(getBeanId()).append(".").append(setter.getName()).append("(").append(subBeanId)
                                .append(");\n");

                    }
                }

            }

            @SuppressWarnings("unchecked")
            @Override
            protected void doLinkList(Method setter, SQLind data) throws Exception {
                Class<? extends Object> subClazz = data.link();
                List<String> beanIds = new ArrayList<String>(beanId);
                beanIds.add(setter.getName());
                tmpSB.append(createNewBeanList(beanIds));
                TransformationSubWiring<Z> subTransfo = new TransformationSubWiring<Z>(subClazz, beanIds);
                secondarySB.append(subTransfo.mainSB);
                colType.putAll(subTransfo.colType);
            }
        }

        // -------------------------------

        public SQLQuery() {
        }

        private SQLQuery(String file, String schema, String id, String... params) throws SQLMapperException {
            try {
               
                createTemplate(file, schema, id, params);
           
                if (sql != null){
                   
                Matcher m = SQL_SELECT_PATTERN.matcher(sql);
                String str = null;
                if (m.find())
                    str = m.group(2);
                if(str!=null){
                	m= SQL_COLUMN_PATTERN.matcher(str);
        			while(m.find()){
        				Matcher m2= SQL_STAT_PATTERN.matcher(m.group(2));
        				while(m2.find()){
        					Matcher m3= SQL_ALIAS_PATTERN.matcher(m2.group(1));
        					if(m3.find())
        				      columnNames.add(m3.group(1));
        				}}
        	 	}}
                template = null;
                schema = null;
                id = null;
                startTargParser = null;
                endTagParser = null;
            } catch (Exception e) {
                throw getException(Errors.ERROR_INIT_QUERY, e);
            }
        }

        protected void createTemplate(String file, String schema, String id, String... params)
                throws SQLMapperException {
            this.id = id;
            boolean queryFound = false;
            options = new ArrayList<String>(1);
            this.schema = schema;
            template = new StringBuilder();
            for (String option : params)
                options.add(option);
            SAXParserFactory spf = SAXParserFactory.newInstance();
            try {
            SAXParser sp = spf.newSAXParser();
            sp.parse(getClass().getClassLoader().getResourceAsStream(file), SQLQueryMapper.SQLQuery.this);
            } catch (StopEvent se) {
            	queryFound = true;
            } catch (Exception e) {
                throw getException(Errors.ERROR_INIT_QUERIES, e);
            }
            if (!queryFound)
            	throw new SQLMapperException("Query '"+id+"' not found in file "+file);
            if (schema != null){
                buildSQLWithSchema(template.toString(),SQL_FROM_PATTERN,"from ");
                buildSQLWithSchema(sql,SQL_JOIN_PATTERN,"join ");
            }
            else
                sql = template.toString();
   
            shrink();
       
        }

        public void shrink() {
            if(wiring==WiringPolicies.lru||wiring==WiringPolicies.fast)   
                sql =shrinkSQL(sql,injectPoints);
        }

       
        private void buildSQLWithSchema(String template,Pattern pattern,String keyWord) {
           
            Matcher matcher = pattern.matcher(template);
            StringBuilder query = new StringBuilder();
            int idx = 0;
            while (matcher.find()) {
                String from = matcher.group(0).substring(keyWord.length(), matcher.group(0).length());
                query.append(template.substring(idx, matcher.start())).append(keyWord);
                String[] tables = from.split(",");
                int last = query.length();
                for (int j = 0; j < tables.length; j++) {
                    query.append(schema).append(".").append(tables[j].trim());
                    if (j < tables.length - 1)
                        query.append(",");
                    updateInjectPoints(last, (query.length() - last)-(tables[j].length()));
                    last = query.length();
                }
                idx = matcher.end();
            }
            query.append(template.substring(idx, template.length()));
            sql = query.toString();
        }

        private void updateInjectPoints(int idx, int offset) {
            for (int j = injectPoints.size() - 1; j >= 0; j--) {
                InjectPoint ip = injectPoints.get(j);
                if (ip.idx > idx)
                    ip.idx = ip.idx + offset;
                else
                    break;
            }
        }
   

        public String getSQL() {
            return sql;
        }

       

        public List<String> getParameters() {
            return parameters;
        }

        public List<InjectPoint> getInjectPoints() {
            return injectPoints;
        }

        public String getID() {
            return id;
        }

        public WiringPolicies getWiringPolicy() {
            return wiring;
        }

        public List<String> getColumns() {
            return columnNames;
        }

        public void setSQL(String queryString) {
            sql= queryString;   
        }

        // -------------------------------------------------PARSING----------------------------------------
        // ------------------------------------------------------------------------------------------------
        private class StopEvent extends SAXException {
            public StopEvent(){
                super("query found");
            }
            private static final long serialVersionUID = 1L;
        }

        /***************
         * Parse a start Tag
         */
        public void startElement(final String uri, final String localName, final String qName,
                final Attributes attributes) throws SAXException {

            if (startTargParser == null) {
                startTargParser = new ParserBehaviour() {
                    @Override
                    void doQuery() {

                        String currentId = attributes.getValue(ID);

                        if (id.equalsIgnoreCase(currentId)) {
                            found = true;
                            String wiringpolicy = attributes.getValue(POLICY);
                            wiringCache = new HashMap<String, WiringPolicy>(1);
                            wiring = WiringPolicies.valueOf(wiringpolicy);
                        }
                    }

                    @Override
                    void doVar() {
                    	String paramId =attributes.getValue(ID);
                        parameters.add(paramId);
                        template.append(getParamBindingString(paramId));
                    }

                    @Override
                    void doSection() {
                        String id = attributes.getValue(ID);
                        if (!options.contains(id) || ignore > 0)
                            ignore++;
                    }

                    @Override
                    void doInclude() throws SAXException {
                        String currentId = attributes.getValue(ID);
                        String param = attributes.getValue(PARAMS);
                        String[] params = new String[0];
                        List<String> allOptions = new ArrayList<String>(options);
                        if (param != null) {
                            params = param.split(",");
                            for (String xmlparam : params)
                                allOptions.add(xmlparam);
                        }
                        try {
                            params = (String[]) (allOptions.toArray(params));
                            SQLQuery q = new SQLIncludedQuery(file, schema, currentId, params);
                            parameters.addAll(q.parameters);
                            for (InjectPoint ip: q.getInjectPoints()){
                                ip.idx = ip.idx+template.length();
                                injectPoints.add(ip);
                            }
                           
                            template.append(q.template);
                        } catch (SQLMapperException e) {
                            throw new SAXException("Unable to include query " + currentId + " in query " + id, e);
                        }
                    }

                    @Override
                    void doInject() throws SAXException {
                        String currentId = attributes.getValue(ID);
                        injectPoints.add(new InjectPoint(currentId, template.length()));
                       
                    }
                   
                    @Override
                    void doQueries() throws SAXException {
                        libId=attributes.getValue(ID);
                    }
                };
            }
            startTargParser.parseTag(qName, found, ignore);
        }


		public void characters(char[] ch, int start, int length) throws SAXException {
            if (found && ignore == 0)
                    template.append(new String(ch, start, length));
        }

        public void endElement(String uri, String localName, String qName) throws SAXException {
            if (endTagParser == null) {
                endTagParser = new ParserBehaviour() {
                    @Override
                    void doQuery() throws SAXException {
                        if (found)
                            throw new StopEvent();
                    }

                    @Override
                    void doSection() {
                        if (ignore > 0)
                            ignore--;
                    }
                };
            }
            endTagParser.parseTag(qName, found, ignore);
        }

       
       
    }

    protected SQLQueryTemplate getQuery(String schema, String id, String... params) throws SQLMapperException {
        Context ctx = getContext();
        ctx.queryKey.delete(0, ctx.queryKey.length());
        ctx.queryKey.append(schema).append(".").append(id);
        for (String str : params)
            ctx.queryKey.append(".").append(str);
        String key = ctx.queryKey.toString();
        SQLQueryTemplate qt= queryMap.get(key);
        if (qt == null){
            SQLQuery q = new SQLQuery(file, schema, id, params);
            switch (q.getWiringPolicy()) {
                case fast:
                    synchronized (queryMap) {
                        if (!queryMap.containsKey(key)) // check we are the first to initialize.
                            queryMap.put(key, q);
                    }
                    break;
                case lru:
                    synchronized (queryMap) {
                        if (!queryMap.containsKey(key)){// check we are the first to initialize.
                            if(lru.size()>LRU_CACHE_SIZE){
                                queryMap.remove(lru.get(0));
                                lru.remove(0);
                            }
                            lru.add(key);
                            queryMap.put(key, q);
                        }
                    }
                    break;
                default:
                	return q;
            }
           
        }
        return queryMap.get(key);
       
    }

   

    private abstract class ParserBehaviour {
        void parseTag(String name, boolean found, int ignore) throws SAXException {
            Tags tag = null;
            if (name != null) {
                try {
                    tag = Tags.valueOf(name.toLowerCase());
                     if(tag==Tags.queries){
                         doQueries();
                         return;
                     }
                } catch (Exception e) {
                    throw new SAXException("Unknown tag '" + name + "'");
                }
                if (tag != Tags.query && !found)
                    return;

                if (tag != Tags.section && ignore > 0)
                    return;

                switch (tag) {
                case query:
                    doQuery();
                    break;
                case var:
                    doVar();
                    break;
                case section:
                    doSection();
                    break;
                case include:
                    doInclude();
                    break;
                case inject:
                    doInject();
                    break;
               
                default:
                    doOther();
                    break;
                }
            }
        }

        abstract void doQuery() throws SAXException;
        void doSection() throws SAXException {}
        void doVar() throws SAXException {}
        void doInclude() throws SAXException {}
        void doInject() throws SAXException {}
        void doOther() throws SAXException {}
        void doQueries() throws SAXException {}
       
    }
    protected String getParamBindingString(String paramId) {
		return "?";
	}

}