package net.sparkworks.cargo.client.impl;

import net.sparkworks.cargo.client.APIBaseClient;
import net.sparkworks.cargo.client.UnitConversionClient;
import net.sparkworks.cargo.client.config.CargoClientConfig;
import net.sparkworks.cargo.client.config.CargoInternalCommunicationConfiguration;
import net.sparkworks.cargo.client.config.CargoRestTemplateConfig;
import net.sparkworks.cargo.common.dto.UnitConversionCreateDTO;
import net.sparkworks.cargo.common.dto.UnitConversionDTO;
import net.sparkworks.cargo.common.dto.UnitConversionQueryDTO;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestOperations;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Collection;
import java.util.UUID;

@Component("CaRGOUnitConversionClientImplV2")
public class UnitConversionClientImpl extends APIBaseClient implements UnitConversionClient {
    
    private String accessToken;

    @Autowired
    public UnitConversionClientImpl(@Qualifier(CargoRestTemplateConfig.SW_CARGO_REST_TEMPLATE_NAME) RestOperations restTemplate,
                                    final CargoClientConfig sparkworksCargoClientConfig,
                                    final CargoInternalCommunicationConfiguration cargoInternalCommunicationConfiguration) {
        this.restTemplate = restTemplate;
        this.apiBaseUrl = sparkworksCargoClientConfig.getAddress() +
                (cargoInternalCommunicationConfiguration.isEnabled() ? INTERNAL_PATH : "") +
                API_V2;
    }
    
    public UnitConversionClientImpl(final String apiBaseUrl, final String clientId, final String clientSecret, final String username,
                                    final String password, final String oauth2TokenUrl, final String oauth2GrantType,
                                    final String oauth2Scope) {
        super(apiBaseUrl, clientId, clientSecret, username, password, oauth2TokenUrl, oauth2GrantType, oauth2Scope);
    }
    
    public UnitConversionClientImpl(final String apibaseUrl, final String accessToken){
        super(apibaseUrl);
        this.accessToken = accessToken;
    }
    
    @Override
    public Collection<UnitConversionDTO> listAll() {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<Collection<UnitConversionDTO>> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + CONVERSION).toUriString(),
                        HttpMethod.GET, httpEntity, new ParameterizedTypeReference<Collection<UnitConversionDTO>>() {});
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    @Override
    public UnitConversionDTO getByUUID(final UUID uuid) {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<UnitConversionDTO> response = restTemplate.
                exchange(UriComponentsBuilder.
                        fromUriString(apiBaseUrl + CONVERSION_BY_UUID).
                        buildAndExpand(uuid).
                        toUriString(), HttpMethod.GET, httpEntity, UnitConversionDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    @Override
    public UnitConversionDTO create(final UUID sourceUuid, final UUID targetUuid, final String method) {
        final UnitConversionCreateDTO unitConversionCreateDTO = UnitConversionCreateDTO.builder()
                .sourceUuid(sourceUuid)
                .targetUuid(targetUuid)
                .method(method)
                .build();
        HttpEntity<UnitConversionCreateDTO> httpEntity = new HttpEntity<>(unitConversionCreateDTO, prepareHeaders(accessToken));
        ResponseEntity<UnitConversionDTO> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + CONVERSION).toUriString(),
                        HttpMethod.POST, httpEntity, UnitConversionDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    @Override
    public Collection<UnitConversionDTO> query(final UUID sourceUuid, final UUID targetUuid, final UUID uuid) {
        final UnitConversionQueryDTO queryDTO = UnitConversionQueryDTO.builder()
                .sourceUuid(sourceUuid)
                .targetUuid(targetUuid)
                .uuid(uuid)
                .build();
        HttpEntity<UnitConversionQueryDTO> httpEntity = new HttpEntity<>(queryDTO, prepareHeaders(accessToken));
        ResponseEntity<Collection<UnitConversionDTO>> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + CONVERSION_BY_QUERY).toUriString(),
                        HttpMethod.POST, httpEntity, new ParameterizedTypeReference<Collection<UnitConversionDTO>>() {});
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    
    @Override
    public void delete(UUID uuid) {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<Void> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + CONVERSION_BY_UUID).
                                buildAndExpand(uuid).
                                toUriString(),
                        HttpMethod.DELETE, httpEntity, Void.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
    }
    
    
}
