package net.sparkworks.cargo.client.impl;

import net.sparkworks.cargo.client.APIBaseClient;
import net.sparkworks.cargo.client.UnitClient;
import net.sparkworks.cargo.client.config.CargoClientConfig;
import net.sparkworks.cargo.client.config.CargoInternalCommunicationConfiguration;
import net.sparkworks.cargo.client.config.CargoRestTemplateConfig;
import net.sparkworks.cargo.common.dto.UnitCreateDTO;
import net.sparkworks.cargo.common.dto.UnitDTO;
import net.sparkworks.cargo.common.dto.UnitQueryDTO;
import net.sparkworks.cargo.common.dto.UnitUpdateDTO;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestOperations;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Collection;
import java.util.UUID;

@Component("CaRGOUnitClientImplV2")
public class UnitClientImpl extends APIBaseClient implements UnitClient {
    
    private String accessToken;

    @Autowired
    public UnitClientImpl(@Qualifier(CargoRestTemplateConfig.SW_CARGO_REST_TEMPLATE_NAME) RestOperations restTemplate,
                          final CargoClientConfig sparkworksCargoClientConfig,
                          final CargoInternalCommunicationConfiguration cargoInternalCommunicationConfiguration) {
        this.restTemplate = restTemplate;
        this.apiBaseUrl = sparkworksCargoClientConfig.getAddress() +
                (cargoInternalCommunicationConfiguration.isEnabled() ? INTERNAL_PATH : "") +
                API_V2;
    }
    
    public UnitClientImpl(final String apiBaseUrl, final String clientId, final String clientSecret, final String username,
                          final String password, final String oauth2TokenUrl, final String oauth2GrantType,
                          final String oauth2Scope) {
        super(apiBaseUrl, clientId, clientSecret, username, password, oauth2TokenUrl, oauth2GrantType, oauth2Scope);
    }
    
    public UnitClientImpl(final String apibaseUrl, final String accessToken){
        super(apibaseUrl);
        this.accessToken = accessToken;
    }
    
    @Override
    public Collection<UnitDTO> listAll() {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<Collection<UnitDTO>> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + UNIT).toUriString(),
                        HttpMethod.GET, httpEntity, new ParameterizedTypeReference<Collection<UnitDTO>>() {});
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    @Override
    public UnitDTO getByUUID(final UUID uuid) {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<UnitDTO> response = restTemplate.
                exchange(UriComponentsBuilder.
                        fromUriString(apiBaseUrl + UNIT_BY_UUID).
                        buildAndExpand(uuid).
                        toUriString(), HttpMethod.GET, httpEntity, UnitDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    @Override
    public UnitDTO create(final String name) {
        final UnitCreateDTO unitCreateDTO = UnitCreateDTO.builder().name(name).build();
        HttpEntity<UnitCreateDTO> httpEntity = new HttpEntity<>(unitCreateDTO, prepareHeaders(accessToken));
        ResponseEntity<UnitDTO> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + UNIT).toUriString(),
                        HttpMethod.POST, httpEntity, UnitDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    @Override
    public UnitDTO query(final String name, final UUID uuid) {
        final UnitQueryDTO queryDTO = UnitQueryDTO.builder().name(name).uuid(uuid).build();
        HttpEntity<UnitQueryDTO> httpEntity = new HttpEntity<>(queryDTO, prepareHeaders(accessToken));
        ResponseEntity<UnitDTO> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + UNIT_BY_QUERY).toUriString(),
                        HttpMethod.POST, httpEntity, UnitDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    
    @Override
    public void delete(UUID uuid) {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<Void> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + UNIT_BY_UUID).
                                buildAndExpand(uuid).
                                toUriString(),
                        HttpMethod.DELETE, httpEntity, Void.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
    }
    
    @Override
    public UnitDTO update(UUID uuid, String name) {
        UnitUpdateDTO updateDTO = UnitUpdateDTO.builder()
                .name(name)
                .build();
        
        HttpEntity<UnitUpdateDTO> httpEntity = new HttpEntity<>(updateDTO, prepareHeaders(accessToken));
        ResponseEntity<UnitDTO> response = restTemplate.
                exchange(UriComponentsBuilder.
                                fromUriString(apiBaseUrl + UNIT_BY_UUID).
                                buildAndExpand(uuid).toUriString(),
                        HttpMethod.POST, httpEntity, UnitDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
}
