package net.sparkworks.cargo.client.impl;

import net.sparkworks.cargo.client.APIBaseClient;
import net.sparkworks.cargo.client.ResourceClient;
import net.sparkworks.cargo.client.config.CargoClientConfig;
import net.sparkworks.cargo.client.config.CargoInternalCommunicationConfiguration;
import net.sparkworks.cargo.client.config.CargoRestTemplateConfig;
import net.sparkworks.cargo.common.dto.ResourceCreateDTO;
import net.sparkworks.cargo.common.dto.ResourceDTO;
import net.sparkworks.cargo.common.dto.ResourceQueryDTO;
import net.sparkworks.cargo.common.dto.ResourceUpdateDTO;
import net.sparkworks.cargo.common.enumerator.ResourceType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestOperations;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Collection;
import java.util.UUID;

@Component("CaRGOResourceClientV2")
public class ResourceClientImpl extends APIBaseClient implements ResourceClient {
    
    private String accessToken;

    @Autowired
    public ResourceClientImpl(@Qualifier(CargoRestTemplateConfig.SW_CARGO_REST_TEMPLATE_NAME) RestOperations restTemplate,
                              final CargoClientConfig sparkworksCargoClientConfig,
                              final CargoInternalCommunicationConfiguration cargoInternalCommunicationConfiguration) {
        this.restTemplate = restTemplate;
        this.apiBaseUrl = sparkworksCargoClientConfig.getAddress() +
                (cargoInternalCommunicationConfiguration.isEnabled() ? INTERNAL_PATH : "") + API_V2;
    }
    
    public ResourceClientImpl(final String apiBaseUrl, final String clientId, final String clientSecret, final String username,
                              final String password, final String oauth2TokenUrl, final String oauth2GrantType,
                              final String oauth2Scope) {
        super(apiBaseUrl, clientId, clientSecret, username, password, oauth2TokenUrl, oauth2GrantType, oauth2Scope);
    }
    
    public ResourceClientImpl(final String apibaseUrl, final String accessToken) {
        super(apibaseUrl);
        this.accessToken = accessToken;
    }

    @Override
    public Collection<ResourceDTO> listAll() {
        MultiValueMap<String, String> headers = new HttpHeaders();
        if(this.accessToken != null) headers.add(AUTHORIZATION_HEADER, AUTH_BEARER_PREFIX + this.accessToken);
        HttpEntity httpEntity = new HttpEntity(headers);

        ResponseEntity<Collection<ResourceDTO>> response = restTemplate.
                exchange(UriComponentsBuilder.
                        fromUriString(apiBaseUrl + RESOURCE).
                        toUriString(), HttpMethod.GET, httpEntity, new ParameterizedTypeReference<Collection<ResourceDTO>>() {});
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }

    @Override
    public ResourceDTO create(final String systemName, final String userFriendlyName, final ResourceType type,
                              final String phenomenon, final String unit, final String controls, final UUID groupUUID) {

        final ResourceCreateDTO resourceCreateDTO = ResourceCreateDTO.builder()
                .systemName(systemName)
                .userFriendlyName(userFriendlyName)
                .type(type)
                .phenomenon(phenomenon)
                .unit(unit)
                .controls(controls)
                .groupUUID(groupUUID)
                .build();
    
        HttpEntity<ResourceCreateDTO> httpEntity = new HttpEntity<>(resourceCreateDTO, prepareHeaders(accessToken));
        ResponseEntity<ResourceDTO> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + RESOURCE).toUriString(),
                        HttpMethod.POST, httpEntity, ResourceDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }

    @Override
    public ResourceDTO getByUUID(UUID uuid) {
        MultiValueMap<String, String> headers = new HttpHeaders();
        if(this.accessToken != null) headers.add(AUTHORIZATION_HEADER, AUTH_BEARER_PREFIX + this.accessToken);
        HttpEntity httpEntity = new HttpEntity(headers);

        ResponseEntity<ResourceDTO> response = restTemplate.
                exchange(UriComponentsBuilder.
                        fromUriString(apiBaseUrl + RESOURCE_BY_UUID).
                        buildAndExpand(uuid).
                        toUriString(), HttpMethod.GET, httpEntity, ResourceDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    @Override
    public ResourceDTO getBySystemName(final String systemName) {
        final ResourceQueryDTO resourceQueryDTO= ResourceQueryDTO.builder()
                .systemName(systemName)
                .build();
        HttpEntity<ResourceQueryDTO> httpEntity = new HttpEntity<>(resourceQueryDTO, prepareHeaders(accessToken));
        ResponseEntity<ResourceDTO> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + RESOURCE_BY_QUERY).toUriString(),
                        HttpMethod.POST, httpEntity, ResourceDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    
    
    @Override
    public Long count() {
        MultiValueMap<String, String> headers = new HttpHeaders();
        if(this.accessToken != null) headers.add(AUTHORIZATION_HEADER, AUTH_BEARER_PREFIX + this.accessToken);
        HttpEntity httpEntity = new HttpEntity(headers);

        ResponseEntity<Long> response = restTemplate.
                exchange(UriComponentsBuilder.
                        fromUriString(apiBaseUrl + RESOURCE_COUNT).
                        toUriString(), HttpMethod.GET, httpEntity, Long.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }

    @Override
    public void delete(UUID uuid) {
        MultiValueMap<String, String> headers = new HttpHeaders();
        if(this.accessToken != null) headers.add(AUTHORIZATION_HEADER, AUTH_BEARER_PREFIX + this.accessToken);
        HttpEntity httpEntity = new HttpEntity(headers);

        ResponseEntity<String> response = restTemplate.
                exchange(UriComponentsBuilder.
                        fromUriString(apiBaseUrl + RESOURCE_BY_UUID).
                        buildAndExpand(uuid).
                        toUriString(), HttpMethod.DELETE, httpEntity, String.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
    }

    @Override
    public ResourceDTO move(final UUID resourceUuid, final UUID targetGroupUuid) {
        MultiValueMap<String, String> headers = new HttpHeaders();
        if(this.accessToken != null) headers.add(AUTHORIZATION_HEADER, AUTH_BEARER_PREFIX + this.accessToken);
        HttpEntity httpEntity = new HttpEntity(headers);

        ResponseEntity<ResourceDTO> response = restTemplate.
                exchange(UriComponentsBuilder.
                        fromUriString(apiBaseUrl + MOVE_RESOURCE_BY_UUID).
                        buildAndExpand(resourceUuid, targetGroupUuid).
                        toUriString(), HttpMethod.POST, httpEntity, ResourceDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    @Override
    public ResourceDTO update(final UUID uuid,
                              final String userFriendlyName,
                              final ResourceType type,
                              final String controls,
                              final UUID phenomenonUuid,
                              final UUID unitUuid) {
        ResourceUpdateDTO updateDTO = ResourceUpdateDTO.builder()
                .userFriendlyName(userFriendlyName)
                .type(type)
                .controls(controls)
                .phenomenonUuid(phenomenonUuid)
                .unitUuid(unitUuid)
                .build();
    
        HttpEntity<ResourceUpdateDTO> httpEntity = new HttpEntity<>(updateDTO, prepareHeaders(accessToken));
        ResponseEntity<ResourceDTO> response = restTemplate.
                exchange(UriComponentsBuilder.
                        fromUriString(apiBaseUrl + RESOURCE_BY_UUID).
                        buildAndExpand(uuid).
                        toUriString(), HttpMethod.POST, httpEntity, ResourceDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
}
