package net.sparkworks.cargo.client.impl;

import net.sparkworks.cargo.client.APIBaseClient;
import net.sparkworks.cargo.client.GroupClient;
import net.sparkworks.cargo.client.config.CargoClientConfig;
import net.sparkworks.cargo.client.config.CargoInternalCommunicationConfiguration;
import net.sparkworks.cargo.client.config.CargoRestTemplateConfig;
import net.sparkworks.cargo.common.dto.GatewayDTO;
import net.sparkworks.cargo.common.dto.GroupCreateDTO;
import net.sparkworks.cargo.common.dto.GroupDTO;
import net.sparkworks.cargo.common.dto.GroupNodeDTO;
import net.sparkworks.cargo.common.dto.GroupUpdateDTO;
import net.sparkworks.cargo.common.dto.ResourceDTO;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestOperations;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Collection;
import java.util.UUID;

@Component("CaRGOGroupClientImplV2")
public class GroupClientImpl extends APIBaseClient implements GroupClient {
    
    private String accessToken;
    
    @Autowired
    public GroupClientImpl(@Qualifier(CargoRestTemplateConfig.SW_CARGO_REST_TEMPLATE_NAME) RestOperations restTemplate,
                           final CargoClientConfig sparkworksCargoClientConfig,
                           final CargoInternalCommunicationConfiguration cargoInternalCommunicationConfiguration) {
        this.restTemplate = restTemplate;
        this.apiBaseUrl = sparkworksCargoClientConfig.getAddress() +
                (cargoInternalCommunicationConfiguration.isEnabled() ? INTERNAL_PATH : "") + API_V2;
    }
    
    public GroupClientImpl(final String apiBaseUrl, final String clientId, final String clientSecret, final String username,
                           final String password, final String oauth2TokenUrl, final String oauth2GrantType,
                           final String oauth2Scope) {
        super(apiBaseUrl, clientId, clientSecret, username, password, oauth2TokenUrl, oauth2GrantType, oauth2Scope);
    }
    
    public GroupClientImpl(final String apibaseUrl, final String accessToken){
        super(apibaseUrl);
        this.accessToken = accessToken;
    }


    @Override
    public Collection<GroupDTO> listAll() {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<Collection<GroupDTO>> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + GROUP).toUriString(), HttpMethod.GET, httpEntity, new ParameterizedTypeReference<Collection<GroupDTO>>() {});
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    @Override
    public GroupDTO create(final String name, final String path) {
        final GroupCreateDTO groupCreateDTO = GroupCreateDTO.builder()
                .name(name)
                .parentPath(path)
                .build();
        HttpEntity<GroupCreateDTO> httpEntity = new HttpEntity<>(groupCreateDTO, prepareHeaders(accessToken));
        ResponseEntity<GroupDTO> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + GROUP).toUriString(), HttpMethod.POST, httpEntity, GroupDTO.class);
        if (!response.getStatusCode().is2xxSuccessful())
            throw new RestClientException(response.toString());
        return response.getBody();
    }

    @Override
    public GroupDTO getByUUID(UUID uuid) {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<GroupDTO> response = restTemplate.
                exchange(UriComponentsBuilder.
                        fromUriString(apiBaseUrl + GROUP_BY_UUID).
                        buildAndExpand(uuid).
                        toUriString(), HttpMethod.GET, httpEntity, GroupDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }

    @Override
    public Long count() {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<Long> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + GROUP_COUNT).toUriString(),
                        HttpMethod.GET, httpEntity, Long.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }

    @Override
    public void delete(UUID uuid) {
        HttpEntity<GroupDTO> httpEntity = new HttpEntity<>(prepareHeaders(accessToken));
        ResponseEntity<Void> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + GROUP_BY_UUID).
                                buildAndExpand(uuid).
                                toUriString(),
                        HttpMethod.DELETE, httpEntity, Void.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());

    }

    @Override
    public void move(UUID sourceUUID, UUID targetUUID) {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<Void> response = restTemplate.
                exchange(UriComponentsBuilder.
                        fromUriString(apiBaseUrl + GROUP_MOVE).
                        buildAndExpand(sourceUUID,targetUUID).
                        toUriString(), HttpMethod.POST, httpEntity, Void.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());

    }

    @Override
    public Collection<ResourceDTO> getGroupResources(UUID uuid) {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<Collection<ResourceDTO>> response = restTemplate.
                exchange(UriComponentsBuilder.
                        fromUriString(apiBaseUrl + GROUP_RESOURCES).
                        buildAndExpand(uuid).
                        toUriString(), HttpMethod.GET, httpEntity, new ParameterizedTypeReference<Collection<ResourceDTO>>() {});
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }

    @Override
    public Collection<GroupDTO> getSubGroups(UUID uuid, int depth) {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<Collection<GroupDTO>> response = restTemplate.
                exchange(UriComponentsBuilder.
                        fromUriString(apiBaseUrl + GROUP_SUBGROUPS).
                        buildAndExpand(uuid, depth).
                        toUriString(), HttpMethod.GET, httpEntity, new ParameterizedTypeReference<Collection<GroupDTO>>() {});
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }

    @Override
    public Collection<GatewayDTO> getGroupGateways(UUID uuid) {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<Collection<GatewayDTO>> response = restTemplate.
                exchange(UriComponentsBuilder.
                        fromUriString(apiBaseUrl + GROUP_GATEWAY).
                        buildAndExpand(uuid).
                        toUriString(), HttpMethod.GET, httpEntity, new ParameterizedTypeReference<Collection<GatewayDTO>>() {});
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }

    @Override
    public Collection<GroupDTO> getRootGroups() {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<Collection<GroupDTO>> response = restTemplate.
                exchange(UriComponentsBuilder.
                        fromUriString(apiBaseUrl + GROUP_MAIN).
                        toUriString(), HttpMethod.GET, httpEntity, new ParameterizedTypeReference<Collection<GroupDTO>>() {});
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    @Override
    public GroupDTO update(UUID uuid, String name) {
        GroupUpdateDTO groupUpdateDTO = GroupUpdateDTO.builder()
                .name(name)
                .build();
        HttpEntity<GroupUpdateDTO> httpEntity = new HttpEntity<>(groupUpdateDTO, prepareHeaders(accessToken));
        ResponseEntity<GroupDTO> response = restTemplate.
                exchange(UriComponentsBuilder.
                        fromUriString(apiBaseUrl + GROUP_BY_UUID).
                        buildAndExpand(uuid).
                        toUriString(), HttpMethod.POST, httpEntity, GroupDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    @Override
    public GroupNodeDTO getGroupsTree() {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<GroupNodeDTO> response = restTemplate.
                exchange(UriComponentsBuilder.
                        fromHttpUrl(apiBaseUrl + GROUP_TREE).
                        toUriString(), HttpMethod.GET, httpEntity, GroupNodeDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
}
