package net.sparkworks.cargo.client.impl;

import net.sparkworks.cargo.client.APIBaseClient;
import net.sparkworks.cargo.client.GatewayClient;
import net.sparkworks.cargo.client.config.CargoClientConfig;
import net.sparkworks.cargo.client.config.CargoInternalCommunicationConfiguration;
import net.sparkworks.cargo.client.config.CargoRestTemplateConfig;
import net.sparkworks.cargo.common.dto.GatewayCreateDTO;
import net.sparkworks.cargo.common.dto.GatewayDTO;
import net.sparkworks.cargo.common.dto.GatewayQueryDTO;
import net.sparkworks.cargo.common.dto.ResourceDTO;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestOperations;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Collection;
import java.util.UUID;

@Component("CaRGOGatewayClientV2")
public class GatewayClientImpl extends APIBaseClient implements GatewayClient {
    
    private String accessToken;

    @Autowired
    public GatewayClientImpl(@Qualifier(CargoRestTemplateConfig.SW_CARGO_REST_TEMPLATE_NAME) RestOperations restTemplate,
                             final CargoClientConfig sparkworksCargoClientConfig,
                             final CargoInternalCommunicationConfiguration cargoInternalCommunicationConfiguration) {
        this.restTemplate = restTemplate;
        this.apiBaseUrl = sparkworksCargoClientConfig.getAddress() +
                (cargoInternalCommunicationConfiguration.isEnabled() ? INTERNAL_PATH : "") +
                API_V2;
    }
    
    public GatewayClientImpl(final String apiBaseUrl, final String clientId, final String clientSecret, final String username,
                             final String password, final String oauth2TokenUrl, final String oauth2GrantType,
                             final String oauth2Scope) {
        super(apiBaseUrl, clientId, clientSecret, username, password, oauth2TokenUrl, oauth2GrantType, oauth2Scope);
    }
    
    public GatewayClientImpl(final String apibaseUrl, final String accessToken){
        super(apibaseUrl);
        this.accessToken = accessToken;
    }
    
    @Override
    public Collection<GatewayDTO> listAll() {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<Collection<GatewayDTO>> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + GATEWAY).toUriString(),
                        HttpMethod.GET, httpEntity, new ParameterizedTypeReference<Collection<GatewayDTO>>() {});
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    @Override
    public GatewayDTO getByUUID(final UUID uuid) {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<GatewayDTO> response = restTemplate.
                exchange(UriComponentsBuilder.
                        fromUriString(apiBaseUrl + GATEWAY_BY_UUID).
                        buildAndExpand(uuid).
                        toUriString(), HttpMethod.GET, httpEntity, GatewayDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    @Override
    public GatewayDTO getBySystemName(final String systemName) {
        final GatewayQueryDTO gatewayQueryDTO = GatewayQueryDTO.builder()
                .systemName(systemName)
                .build();
        HttpEntity<GatewayQueryDTO> httpEntity = new HttpEntity<>(gatewayQueryDTO, prepareHeaders(accessToken));
        ResponseEntity<GatewayDTO> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + GATEWAY_BY_QUERY).toUriString(),
                        HttpMethod.POST, httpEntity, GatewayDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    @Override
    public GatewayDTO create(final String systemName, final UUID groupUUID) {
        final GatewayCreateDTO gatewayCreateDTO = GatewayCreateDTO.builder()
                .systemName(systemName)
                .groupUUID(groupUUID)
                .build();
        HttpEntity<GatewayCreateDTO> httpEntity = new HttpEntity<>(gatewayCreateDTO, prepareHeaders(accessToken));
        ResponseEntity<GatewayDTO> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + GATEWAY).toUriString(),
                        HttpMethod.POST, httpEntity, GatewayDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }

    @Override
    public Long count() {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<Long> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + GATEWAY_COUNT).toUriString(),
                        HttpMethod.GET, httpEntity, Long.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }


    @Override
    public void delete(UUID uuid) {
        HttpEntity<GatewayDTO> httpEntity = new HttpEntity<>(prepareHeaders(accessToken));
        ResponseEntity<GatewayDTO> response = restTemplate.
                exchange(UriComponentsBuilder.fromUriString(apiBaseUrl + GATEWAY_BY_UUID).
                                buildAndExpand(uuid).
                                toUriString(),
                        HttpMethod.DELETE, httpEntity, GatewayDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
    }

    @Override
    public Collection<ResourceDTO> getResources(UUID gatewayUuid) {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<Collection<ResourceDTO>> response = restTemplate.
                exchange(UriComponentsBuilder.
                        fromUriString(apiBaseUrl + GATEWAY_RESOURCE).
                        buildAndExpand(gatewayUuid).
                        toUriString(), HttpMethod.GET, httpEntity, new ParameterizedTypeReference<Collection<ResourceDTO>>() {});
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }

}
