package net.sparkworks.cargo.client.impl;

import net.sparkworks.cargo.client.APIBaseClient;
import net.sparkworks.cargo.client.DataClient;
import net.sparkworks.cargo.client.config.CargoClientConfig;
import net.sparkworks.cargo.client.config.CargoInternalCommunicationConfiguration;
import net.sparkworks.cargo.client.config.CargoRestTemplateConfig;
import net.sparkworks.cargo.common.CargoRoutes;
import net.sparkworks.cargo.common.dto.data.LatestDTO;
import net.sparkworks.cargo.common.dto.data.PublishDataListDTO;
import net.sparkworks.cargo.common.dto.data.QueryLatestResourceDataDTO;
import net.sparkworks.cargo.common.dto.data.QueryLatestResourceDataResultDTO;
import net.sparkworks.cargo.common.dto.data.QueryTimeRangeResourceDataDTO;
import net.sparkworks.cargo.common.dto.data.QueryTimeRangeResourceDataResultDTO;
import net.sparkworks.cargo.common.dto.data.SummaryDTO;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestOperations;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.UUID;

@Component("CaRGODataClientV2")
public class DataClientImpl extends APIBaseClient implements DataClient {
    
    private String accessToken;
    
    @Autowired
    public DataClientImpl(@Qualifier(CargoRestTemplateConfig.SW_CARGO_REST_TEMPLATE_NAME) RestOperations restTemplate,
                          final CargoClientConfig sparkworksClientConfig,
                          final CargoInternalCommunicationConfiguration internalCommunicationConfiguration) {
        this.restTemplate = restTemplate;
        this.apiBaseUrl = sparkworksClientConfig.getAddress() +
                (internalCommunicationConfiguration.isEnabled() ? CargoRoutes.INTERNAL_PATH : "") +
                CargoRoutes.API_V2;
    }
    
    public DataClientImpl(final String apiBaseUrl, final String clientId, final String clientSecret, final String username,
                          final String password, final String oauth2TokenUrl, final String oauth2GrantType,
                          final String oauth2Scope) {
        super(apiBaseUrl, clientId, clientSecret, username, password, oauth2TokenUrl, oauth2GrantType, oauth2Scope);
    }
    
    public DataClientImpl(final String apibaseUrl, final String accessToken){
        super(apibaseUrl);
        this.accessToken = accessToken;
    }
    
    @Override
    public SummaryDTO getLatestSummary(final UUID resourceUuid) {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<SummaryDTO> response = restTemplate.
                exchange(UriComponentsBuilder.
                                fromUriString(apiBaseUrl + CargoRoutes.RESOURCE_SUMMARY).
                                buildAndExpand(resourceUuid).toUriString(),
                        HttpMethod.GET, httpEntity, SummaryDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    @Override
    public LatestDTO getLatestValues(final UUID resourceUuid) {
        HttpEntity httpEntity = new HttpEntity(prepareHeaders(accessToken));
        ResponseEntity<LatestDTO> response = restTemplate.
                exchange(UriComponentsBuilder.
                                fromUriString(apiBaseUrl + CargoRoutes.RESOURCE_LATEST).
                                buildAndExpand(resourceUuid).toUriString(),
                        HttpMethod.GET, httpEntity, LatestDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    @Override
    public QueryLatestResourceDataResultDTO queryLatestResourcesData(final QueryLatestResourceDataDTO queryLatestResourceDataDTO) {
        HttpEntity<QueryLatestResourceDataDTO> httpEntity =
                new HttpEntity<>(queryLatestResourceDataDTO, prepareHeaders(accessToken));
        ResponseEntity<QueryLatestResourceDataResultDTO> response = restTemplate.
                exchange(UriComponentsBuilder.
                                fromUriString(apiBaseUrl + CargoRoutes.RESOURCE_QUERY_LATEST).toUriString(),
                        HttpMethod.POST, httpEntity, QueryLatestResourceDataResultDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
    
    @Override
    public QueryTimeRangeResourceDataResultDTO queryTimeRangeResourcesData(QueryTimeRangeResourceDataDTO queryTimeRangeResourceDataDTO) {
        HttpEntity<QueryTimeRangeResourceDataDTO> httpEntity =
                new HttpEntity<>(queryTimeRangeResourceDataDTO, prepareHeaders(accessToken));
        ResponseEntity<QueryTimeRangeResourceDataResultDTO> response = restTemplate.
                exchange(UriComponentsBuilder.
                                fromUriString(apiBaseUrl + CargoRoutes.RESOURCE_QUERY_TIMERANGE).toUriString(),
                        HttpMethod.POST, httpEntity, QueryTimeRangeResourceDataResultDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }

    @Override
    public PublishDataListDTO publishData(PublishDataListDTO publishDataListDTO){
        HttpEntity<PublishDataListDTO> httpEntity = new HttpEntity<>(publishDataListDTO, prepareHeaders(accessToken));
        ResponseEntity<PublishDataListDTO> response = restTemplate.
                exchange(apiBaseUrl + CargoRoutes.RESOURCE_DATA_PUBLISH,
                        HttpMethod.POST,httpEntity , PublishDataListDTO.class);
        if (!response.getStatusCode().is2xxSuccessful()) throw new RestClientException(response.toString());
        return response.getBody();
    }
}
