package net.sparkworks.cargo.client;

import net.sparkworks.cargo.common.dto.ResourceDTO;
import net.sparkworks.cargo.common.enumerator.ResourceType;

import java.util.Collection;
import java.util.UUID;

public interface ResourceClient {
    
    /**
     * Retrieve all the Resources the user has access to.
     *
     * @return a {@link Collection} containing all the {@link ResourceDTO} the user has access to.
     */
    Collection<ResourceDTO> listAll();
    
    /**
     * Retrieve a Resource by its UUID
     *
     * @param uuid the {@link UUID} of the Resource to retrieve
     * @return the {@link ResourceDTO} found
     */
    ResourceDTO getByUUID(UUID uuid);
    
    /**
     * Retrieve a Resource by its name
     *
     * @param systemName the system name of the Resource to retrieve
     * @return the {@link ResourceDTO} found
     */
    ResourceDTO getBySystemName(final String systemName);
    
    /**
     * Create a new Resource.
     *
     * @param systemName the uri of the new Resource
     * @param userFriendlyName the user friendly name of the Resource
     * @param type Sensor/Actuator
     * @param phenomenon the phenomenon name of the Resource observes
     * @param unit the unit of measurement the Resource
     * @param controls // TODO: 3/12/2018 add description to controls
     * @param groupUUID the UUID of the Group the new Resource will belong to
     * @return the new {@link ResourceDTO}
     */
    ResourceDTO create(final String systemName, final String userFriendlyName, final ResourceType type, final String phenomenon, final String unit, final String controls, final UUID groupUUID);
    
    /**
     * Retrieve the number of the Resources the user has access to.
     *
     * @return the number of the Resources the user has access to
     */
    Long count();
    
    /**
     * Delete a Resource by its UUID.
     *
     * @param uuid the {@link UUID} of the Resource to delete
     */
    void delete(UUID uuid);
    
    /**
     * Move a Resource to a Group
     *
     * @param resourceUuid the Resource {@link UUID}
     * @param targetGroupUuid the Group {@link UUID}
     * @return the updated {@link ResourceDTO}
     */
    ResourceDTO move(final UUID resourceUuid, final UUID targetGroupUuid);
    
    ResourceDTO update(UUID uuid, String userFriendlyName, ResourceType typeUuid, String controls, UUID phenomenonUuid, UUID unitUuid);
}
