package net.sparkworks.cargo.client;


import net.sparkworks.cargo.common.dto.GatewayDTO;
import net.sparkworks.cargo.common.dto.GroupDTO;
import net.sparkworks.cargo.common.dto.GroupNodeDTO;
import net.sparkworks.cargo.common.dto.ResourceDTO;

import java.util.Collection;
import java.util.UUID;

public interface GroupClient {
    
    /**
     * Retrieve a list of all the Groups the user has access to.
     *
     * @return a {@link Collection} containing all the {@link GroupDTO} the user has access to.
     */
    Collection<GroupDTO> listAll();
    
    /**
     * Create a new Group.
     *
     * @param name the name of the new Group
     * @param path the path of the Parent Group
     * @return the new {@link GroupDTO}
     */
    GroupDTO create(final String name, final String path);

    /**
     * Retrieve a Group by its UUID if the user has access to it.
     *
     * @param uuid the {@link UUID} of the Group to retrieve
     * @return the {@link GroupDTO} found by the UUID
     */
    GroupDTO getByUUID(UUID uuid);
    
    /**
     * Retrieve the number of the Groups the user has access to.
     *
     * @return the number of the Groups the user has access to
     */
    Long count();
    
    /**
     * Delete a Group by its UUID.
     *
     * @param uuid the {@link UUID} of the Group to delete
     */
    void delete(UUID uuid);

    /**
     * Move a Group under another Group.
     * e.g.
     * (using invalid UUID just for the purpose of readability in the example)
     * GroupA = {uuid=111, path=ROOT.PATH.TO.GROUPA}
     * GroupB = {uuid=333, path=ROOT.PATH.TO.ANOTHER.GROUPB}
     * GroupC = {uuid=555, path=ROOT.PATH.TO.GROUPA.GROUPC}
     * After executing move(111, 333) the Groups data will be like:
     * GroupA = {uuid=111, path=ROOT.PATH.TO.ANOTHER.GROUPB.GROUPA}
     * GroupB = {uuid=333, path=ROOT.PATH.TO.ANOTHER.GROUPB}
     * GroupC = {uuid=555, path=ROOT.PATH.TO.ANOTHER.GROUPB.GROUPA.GROUPC}
     *
     * @param sourceUUID the UUID of the Group to be moved
     * @param targetUUID the UUID of the target Group
     */
    void move(UUID sourceUUID,UUID targetUUID);

    /**
     * Retrieve the Resources of a Group with specific UUID.
     *
     * @param uuid the the UUID of the Group to get the Resources from
     * @return a {@link Collection} of {@link ResourceDTO}s that belong to the Group
     */
    Collection<ResourceDTO> getGroupResources(UUID uuid);

    /**
     * Retrieve the SubGroups of a specific Group in certain depth.
     *
     * @param uuid the UUID of the Group to get the SubGroups from
     * @param depth the depth of the SubGroups
     * @return a {@link Collection} of {@link GroupDTO} containing the SubGroups of the Group
     */
    Collection<GroupDTO> getSubGroups(UUID uuid, int depth);

    /**
     * Retrieve the Gateways that belong to a specific Group
     *
     * @param uuid the {@link UUID} of the Group
     * @return a {@link Collection} of {@link GatewayDTO} that belong to the Group
     */
    Collection<GatewayDTO> getGroupGateways(UUID uuid);
    
    /**
     * Retrieve the Main Groups of the current User
     *
     * @return a {@link Collection} of {@link GroupDTO} that contains the main Groups of the user
     */
    Collection<GroupDTO> getRootGroups();
    
    /**
     * Update a Group by its UUID
     *
     * @param uuid the {@link UUID} of the Group
     * @param name the new name of the Group
     * @return the updated {@link GroupDTO}
     */
    GroupDTO update(UUID uuid, String name);
    
    /**
     * Retrieve the Tree structure of the Groups starting from the Root
     * @return a {@link GroupNodeDTO} that contains all the Tree
     */
    GroupNodeDTO getGroupsTree();
}
