package net.sparkworks.cargo.client;

import net.sparkworks.cargo.common.dto.GatewayDTO;
import net.sparkworks.cargo.common.dto.ResourceDTO;

import java.util.Collection;
import java.util.UUID;

public interface GatewayClient {

    /**
     * Retrieve a list of all the Gateways the user has access to.
     *
     * @return {@link Collection} {@link GatewayDTO} containst all the Gateways the user has access to.
     */
    Collection<GatewayDTO> listAll();

    /**
     * Retrieve a Gateway by its UUID if the user has access to it.
     *
     * @param uuid the UUID of the Gateway to retrieve
     * @return the {@link GatewayDTO} found with the requested UUID
     */
    GatewayDTO getByUUID(UUID uuid);
    
    /**
     * Retrieve a Gateway by its name if the user has access to it.
     *
     * @param systemName the system name of the Gateway to retrieve
     * @return the {@link GatewayDTO} found with the requested system name
     */
    GatewayDTO getBySystemName(final String systemName);

    /**
     * Create a new Gateway.
     *
     * @param systemName the name of the new Gateway
     * @param groupUUID the Group UUID of the Group the Gateway will belong to
     * @return the new {@link GatewayDTO}
     */
    GatewayDTO create(final String systemName, final UUID groupUUID);

    /**
     * Retrieve the number of the Gateways the user has access to.
     *
     * @return the number of the Gateways the user has access to
     */
    Long count();
    
    /**
     * Delete a Gateway by its UUID.
     *
     * @param uuid the {@link UUID} of the Gateway to delete
     */
    void delete(UUID uuid);

    /**
     * Retrieve the Resources of a Gateway with specific UUID.
     *
     * @param gatewayUuid the Gateway UUID to retrieve the Resources from
     * @return a {@link Collection} containing the {@link ResourceDTO}s of the Gateway
     */
    Collection<ResourceDTO> getResources(UUID gatewayUuid);
}
