package net.sparkworks.cargo.client;

import net.sparkworks.cargo.common.dto.data.LatestDTO;
import net.sparkworks.cargo.common.dto.data.PublishDataListDTO;
import net.sparkworks.cargo.common.dto.data.QueryLatestResourceDataDTO;
import net.sparkworks.cargo.common.dto.data.QueryLatestResourceDataResultDTO;
import net.sparkworks.cargo.common.dto.data.QueryTimeRangeResourceDataDTO;
import net.sparkworks.cargo.common.dto.data.QueryTimeRangeResourceDataResultDTO;
import net.sparkworks.cargo.common.dto.data.SummaryDTO;

import java.util.UUID;

public interface DataClient {
    
    /**
     * Retrieve latest summary values of a Resource
     * Retrieve a summary of the data for a specific Resource by its resource ID.
     * The summary contains the 42 latest aggregated values for intervals of 5 minutes, 1 hour, 1 day and 1 month.
     * A Spark Works Accounts authenticated common user is able to query Resources that has permissions on.
     * An administrator is able to query any Resource of the Platform
     *
     * @param resourceUuid the Resource {@link UUID}
     * @return {@link SummaryDTO} containing the latest summary values
     */
    SummaryDTO getLatestSummary(UUID resourceUuid);
    
    /**
     * Retrieve latest values of a Resource
     * Retrieve the latest aggregated values for a specific Resource by its resource ID
     * The response contains the latest aggregated values for intervals of 5 minutes, 1 hour, 1 day and 1 month.
     * A Spark Works Accounts authenticated common user is able to query Resources that has permissions on.
     * An administrator is able to query any Resource of the Platform
     *
     * @param resourceUuid the Resource {@link UUID}
     * @return {@link LatestDTO} containing latest values
     */
    LatestDTO getLatestValues(UUID resourceUuid);
    
    /**
     * Retrieve latest values of criteria matching Resources
     * Query the latest aggregated values for multiple Resources.
     * The values can be converted to a requested unit of measurement.
     * The response contains the latest aggregated values for intervals of 5 minutes, 1 hour, 1 day and 1 month.
     * A Spark Works Accounts authenticated common user is able to query Resources that has permissions on. 
     * An administrator is able to query any Resource of the Platform
     *
     * @param queryLatestResourceDataDTO contains a list of query criteria {@link net.sparkworks.cargo.common.dto.data.QueryResourceDataCriteriaDTO}
     * @return {@link QueryLatestResourceDataResultDTO} that contains a {@link java.util.Map} that holds the requested data response for each query criteria
     */
    QueryLatestResourceDataResultDTO queryLatestResourcesData(QueryLatestResourceDataDTO queryLatestResourceDataDTO);
    
    /**
     * Retrieve latest values of criteria matching Resources within a requested time window
     * Query the aggregated historical values for multiple Resources within a requested time window.
     * The values can be retrieved in intervals of 5minutes, 1 hour, 1 day, 1 month and can be converted to a requested unit of measurement.
     * A Spark Works Accounts authenticated common user is able to query Resources that has permissions on.
     * An administrator is able to query any Resource of the Platform
     *
     * @param queryTimeRangeResourceDataDTO cointains a list of query criteria for time range requests
     * @return {@link QueryTimeRangeResourceDataResultDTO} that contains a {@link java.util.Map} that holds the requested data response for each query criteria
     */         
    QueryTimeRangeResourceDataResultDTO queryTimeRangeResourcesData(QueryTimeRangeResourceDataDTO queryTimeRangeResourceDataDTO);

    /**
     * Publish data to the provided Resource.
     *
     * @param publishDataListDTO {@link PublishDataListDTO} that holds the info of the resources and the data to publish on these resource
     * @return {@link PublishDataListDTO} the list of the published data
     */
    PublishDataListDTO publishData(PublishDataListDTO publishDataListDTO);
}
