package net.spals.appbuilder.message.kinesis.producer

import javax.validation.constraints.NotNull

import com.amazonaws.auth.AWSCredentialsProvider
import com.amazonaws.regions.Regions
import com.amazonaws.services.kinesis.producer.{KinesisProducer, KinesisProducerConfiguration}
import com.google.inject.Provider
import com.netflix.governator.annotations.Configuration
import com.typesafe.config.ConfigException
import net.spals.appbuilder.annotations.service.AutoBindProvider

import scala.util.Try

/**
  * A [[Provider]] of the AWS [[KinesisProducer]].
  *
  * @author tkral
  */
@AutoBindProvider
private[producer] class KinesisProducerProvider() extends Provider[KinesisProducer] {

  @NotNull
  @Configuration("messageProducer.kinesis.credentialsProvider")
  private var credentialsProviderClassName: String = null

  @NotNull
  @Configuration("messageProducer.kinesis.endpoint")
  private var endpoint: String = null

  override def get(): KinesisProducer = {
    val config = new KinesisProducerConfiguration()
      .setCredentialsProvider(Class.forName(credentialsProviderClassName).newInstance.asInstanceOf[AWSCredentialsProvider])

    endpoint match {
      case httpEndpoint if httpEndpoint.startsWith("http://") =>
        config.setKinesisEndpoint(httpEndpoint)
      case regionEndpoint if Try(Regions.fromName(regionEndpoint)).isSuccess =>
        config.setRegion(regionEndpoint)
      case _ => throw new ConfigException.BadValue("messageProducer.kinesis.endpoint",
        s"Unrecognized Kinesis endpoint. Value is neither an http endpoint nor a known region: $endpoint")
    }

    new KinesisProducer(config)
  }
}
