/*
 * Copyright (c) 2015 Goldman Sachs.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v. 1.0 which accompany this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 */

package org.eclipse.collections.impl.stack.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.EmptyStackException;

import org.eclipse.collections.api.FloatIterable;
import org.eclipse.collections.api.LazyFloatIterable;
import org.eclipse.collections.api.bag.primitive.MutableFloatBag;
import org.eclipse.collections.api.block.function.primitive.FloatToObjectFunction;
import org.eclipse.collections.api.block.function.primitive.ObjectFloatToObjectFunction;
import org.eclipse.collections.api.block.predicate.primitive.FloatPredicate;
import org.eclipse.collections.api.block.procedure.primitive.FloatProcedure;
import org.eclipse.collections.api.iterator.FloatIterator;
import org.eclipse.collections.api.list.primitive.FloatList;
import org.eclipse.collections.api.list.primitive.MutableFloatList;
import org.eclipse.collections.api.set.primitive.MutableFloatSet;
import org.eclipse.collections.api.stack.MutableStack;
import org.eclipse.collections.api.stack.primitive.FloatStack;
import org.eclipse.collections.api.stack.primitive.ImmutableFloatStack;
import org.eclipse.collections.api.stack.primitive.MutableFloatStack;
import org.eclipse.collections.impl.bag.mutable.primitive.FloatHashBag;
import org.eclipse.collections.impl.factory.primitive.FloatStacks;
import org.eclipse.collections.impl.iterator.UnmodifiableFloatIterator;
import org.eclipse.collections.impl.lazy.primitive.LazyFloatIterableAdapter;
import org.eclipse.collections.impl.list.mutable.primitive.FloatArrayList;
import org.eclipse.collections.impl.set.mutable.primitive.FloatHashSet;
import org.eclipse.collections.impl.stack.mutable.ArrayStack;
import net.jcip.annotations.NotThreadSafe;

/**
 * FloatArrayStack is similar to {@link ArrayStack}, and is memory-optimized for float primitives.
 * This file was automatically generated from template file primitiveArrayStack.stg.
 */
@NotThreadSafe
public final class FloatArrayStack
        implements MutableFloatStack, Externalizable
{
    private static final long serialVersionUID = 1L;

    private transient FloatArrayList delegate;

    public FloatArrayStack()
    {
        this.delegate = new FloatArrayList();
    }

    private FloatArrayStack(int size)
    {
        this.delegate = new FloatArrayList(size);
    }

    private FloatArrayStack(float... items)
    {
        this.delegate = new FloatArrayList(items);
    }

    public static FloatArrayStack newStackFromTopToBottom(float... items)
    {
        FloatArrayStack stack = new FloatArrayStack(items.length);
        for (int i = items.length - 1; i >= 0; i--)
        {
            stack.push(items[i]);
        }
        return stack;
    }

    public static FloatArrayStack newStackWith(float... items)
    {
        return new FloatArrayStack(items);
    }

    public static FloatArrayStack newStack(FloatIterable items)
    {
        FloatArrayStack stack = new FloatArrayStack(items.size());
        stack.delegate = FloatArrayList.newList(items);
        return stack;
    }

    public static FloatArrayStack newStackFromTopToBottom(FloatIterable items)
    {
        FloatArrayStack stack = new FloatArrayStack(items.size());
        stack.delegate = FloatArrayList.newList(items).reverseThis();
        return stack;
    }

    public void push(float item)
    {
        this.delegate.add(item);
    }

    public float pop()
    {
        this.checkEmptyStack();
        return this.delegate.removeAtIndex(this.delegate.size() - 1);
    }

    private void checkEmptyStack()
    {
        if (this.delegate.isEmpty())
        {
            throw new EmptyStackException();
        }
    }

    public FloatList pop(int count)
    {
        this.checkPositiveValueForCount(count);
        this.checkSizeLessThanCount(count);
        if (count == 0)
        {
            return new FloatArrayList(0);
        }
        MutableFloatList subList = new FloatArrayList(count);
        while (count > 0)
        {
            subList.add(this.pop());
            count--;
        }
        return subList;
    }

    private void checkPositiveValueForCount(int count)
    {
        if (count < 0)
        {
            throw new IllegalArgumentException("Count must be positive but was " + count);
        }
    }

    private void checkSizeLessThanCount(int count)
    {
        if (this.delegate.size() < count)
        {
            throw new IllegalArgumentException("Count must be less than size: Count = " + count + " Size = " + this.delegate.size());
        }
    }

    public MutableFloatStack select(FloatPredicate predicate)
    {
        return newStackFromTopToBottom(this.delegate.asReversed().select(predicate));
    }

    public MutableFloatStack reject(FloatPredicate predicate)
    {
        return newStackFromTopToBottom(this.delegate.asReversed().reject(predicate));
    }

    public float peek()
    {
        this.checkEmptyStack();
        return this.delegate.getLast();
    }

    public FloatList peek(int count)
    {
        this.checkPositiveValueForCount(count);
        this.checkSizeLessThanCount(count);
        if (count == 0)
        {
            return new FloatArrayList(0);
        }
        MutableFloatList subList = new FloatArrayList(count);
        int index = this.delegate.size() - 1;
        for (int i = 0; i < count; i++)
        {
            subList.add(this.delegate.get(index - i));
        }
        return subList;
    }

    public float peekAt(int index)
    {
        this.rangeCheck(index);
        return this.delegate.get(this.delegate.size() - 1 - index);
    }

    private void rangeCheck(int index)
    {
        if (index < 0 || index > this.delegate.size() - 1)
        {
            throw new IllegalArgumentException("Index " + index + " out of range.Should be between 0 and " + (this.delegate.size() - 1));
        }
    }

    public FloatIterator floatIterator()
    {
        return new UnmodifiableFloatIterator(this.delegate.asReversed().floatIterator());
    }

    public void forEach(FloatProcedure procedure)
    {
        this.each(procedure);
    }

    /**
     * @since 7.0.
     */
    public void each(FloatProcedure procedure)
    {
        this.delegate.asReversed().forEach(procedure);
    }

    public int size()
    {
        return this.delegate.size();
    }

    public boolean isEmpty()
    {
        return this.delegate.isEmpty();
    }

    public boolean notEmpty()
    {
        return this.delegate.notEmpty();
    }

    public int count(FloatPredicate predicate)
    {
        return this.delegate.asReversed().count(predicate);
    }

    public boolean anySatisfy(FloatPredicate predicate)
    {
        return this.delegate.asReversed().anySatisfy(predicate);
    }

    public boolean allSatisfy(FloatPredicate predicate)
    {
        return this.delegate.asReversed().allSatisfy(predicate);
    }

    public boolean noneSatisfy(FloatPredicate predicate)
    {
        return this.delegate.asReversed().noneSatisfy(predicate);
    }

    public float detectIfNone(FloatPredicate predicate, float ifNone)
    {
        return this.delegate.asReversed().detectIfNone(predicate, ifNone);
    }

    public <V> MutableStack<V> collect(FloatToObjectFunction<? extends V> function)
    {
        return ArrayStack.newStackFromTopToBottom(this.delegate.asReversed().collect(function));
    }

    public <V> V injectInto(V injectedValue, ObjectFloatToObjectFunction<? super V, ? extends V> function)
    {
        return this.delegate.asReversed().injectInto(injectedValue, function);
    }

    public double sum()
    {
        return this.delegate.sum();
    }

    public float max()
    {
        return this.delegate.max();
    }

    public float min()
    {
        return this.delegate.min();
    }

    public float minIfEmpty(float defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public float maxIfEmpty(float defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public double average()
    {
        return this.delegate.average();
    }

    public double median()
    {
        return this.delegate.median();
    }

    public float[] toSortedArray()
    {
        return this.delegate.toSortedArray();
    }

    public float[] toArray()
    {
        return this.delegate.asReversed().toArray();
    }

    public boolean contains(float value)
    {
        return this.delegate.asReversed().contains(value);
    }

    public boolean containsAll(float... source)
    {
        return this.delegate.asReversed().containsAll(source);
    }

    public boolean containsAll(FloatIterable source)
    {
        return this.delegate.asReversed().containsAll(source);
    }

    public void clear()
    {
        this.delegate.clear();
    }

    @Override
    public boolean equals(Object otherStack)
    {
        if (otherStack == this)
        {
            return true;
        }
        if (!(otherStack instanceof FloatStack))
        {
            return false;
        }
        FloatStack stack = (FloatStack) otherStack;
        if (this.size() != stack.size())
        {
            return false;
        }
        for (int i = 0; i < this.size(); i++)
        {
            if (Float.compare(this.peekAt(i), stack.peekAt(i)) != 0)
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int hashCode = 1;
        FloatIterable iterable = this.delegate.asReversed();
        FloatIterator iterator = iterable.floatIterator();
        while (iterator.hasNext())
        {
            float item = iterator.next();
            hashCode = 31 * hashCode + Float.floatToIntBits(item);
        }
        return hashCode;
    }

    @Override
    public String toString()
    {
        return this.delegate.asReversed().toString();
    }

    public String makeString()
    {
        return this.delegate.asReversed().makeString();
    }

    public String makeString(String separator)
    {
        return this.delegate.asReversed().makeString(separator);
    }

    public String makeString(String start, String separator, String end)
    {
        return this.delegate.asReversed().makeString(start, separator, end);
    }

    public void appendString(Appendable appendable)
    {
        this.delegate.asReversed().appendString(appendable);
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.delegate.asReversed().appendString(appendable, separator);
    }

    public void appendString(
            Appendable appendable,
            String start,
            String separator,
            String end)
    {
        this.delegate.asReversed().appendString(appendable, start, separator, end);
    }

    public MutableFloatList toList()
    {
        return FloatArrayList.newList(this);
    }

    public MutableFloatList toSortedList()
    {
        return FloatArrayList.newList(this).sortThis();
    }

    public MutableFloatSet toSet()
    {
        return FloatHashSet.newSet(this);
    }

    public MutableFloatBag toBag()
    {
        return FloatHashBag.newBag(this);
    }

    public LazyFloatIterable asLazy()
    {
        return new LazyFloatIterableAdapter(this);
    }

    public MutableFloatStack asUnmodifiable()
    {
        return new UnmodifiableFloatStack(this);
    }

    public MutableFloatStack asSynchronized()
    {
        return new SynchronizedFloatStack(this);
    }

    public ImmutableFloatStack toImmutable()
    {
        return FloatStacks.immutable.withAll(this.delegate);
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        FloatIterator iterator = this.delegate.asReversed().floatIterator();
        while (iterator.hasNext())
        {
            float each = iterator.next();
            out.writeFloat(each);
        }
    }

    public void readExternal(ObjectInput in) throws IOException
    {
        int size = in.readInt();
        float[] array = new float[size];
        for (int i = size - 1; i >= 0; i--)
        {
            array[i] = in.readFloat();
        }
        this.delegate = FloatArrayList.newListWith(array);
    }
}
