// =================================================================================================
// Copyright 2015 Twitter, Inc.
// -------------------------------------------------------------------------------------------------
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this work except in compliance with the License.
// You may obtain a copy of the License in the LICENSE file, or at:
//
//  http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// =================================================================================================

package com.twitter.common.metrics;

import java.util.Map;
import java.util.Set;

import com.google.common.base.Preconditions;
import com.google.common.base.Supplier;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;

import com.twitter.common.collections.BoundedQueue;
import com.twitter.common.util.Clock;

/**
 * A {@link MetricListener} implementation that behaves as {@link TimeSeriesProvider}.
 */
public class TimeSeriesListeningProvider implements TimeSeriesProvider, MetricListener {

  private final BoundedQueue<Number> timestamps;
  private final LoadingCache<String, BoundedQueue<Number>> timeseries;
  private final Clock clock;

  /**
   * Creates a new {@link TimeSeriesListeningProvider} with given {@code retainedSampleLimit}
   * and {@code clock}.
   *
   * @param size the window size for timeseries
   * @param clock the {@code Clock} instance for sample timestamps
   */
  public TimeSeriesListeningProvider(final int size, final Clock clock) {
    Preconditions.checkNotNull(clock, "Clock can not be null.");
    Preconditions.checkArgument(size > 0, "Timeseries size must be positive.");

    this.clock = clock;
    this.timestamps = new BoundedQueue<Number>(size);

    Supplier<BoundedQueue<Number>> supplier = new Supplier<BoundedQueue<Number>>() {
      @Override
      public BoundedQueue<Number> get() {
        // In order to issue a consistent timeseries, we have to fill it with zeros, to
        // indicate that it wasn't collected previously. Although, we keep room for a
        // single collection that should happen right after this call.
        BoundedQueue<Number> ts = new BoundedQueue<Number>(size);
        for (int i = 0; i < timestamps.size() - 1; i++) {
          ts.add(0L);
        }

        return ts;
      }
    };

    this.timeseries = CacheBuilder.newBuilder().build(CacheLoader.from(supplier));
  }

  @Override
  public synchronized Set<String> getAvailableSeries() {
    return ImmutableSet.copyOf(timeseries.asMap().keySet());
  }

  @Override
  public synchronized Iterable<Number> getTimeSeries(String name) {
    if (!timeseries.asMap().containsKey(name)) { return null; }
    return Iterables.unmodifiableIterable(timeseries.getUnchecked(name));
  }

  @Override
  public synchronized Iterable<Number> getTimestamps() {
    return Iterables.unmodifiableIterable(timestamps);
  }

  @Override
  public synchronized void updateStats(Map<String, Number> samples) {
    // sample timestamp
    timestamps.add(clock.nowMillis());

    // sample timeseries
    for (Map.Entry<String, Number> sample : samples.entrySet()) {
      BoundedQueue<Number> ts = timeseries.getUnchecked(sample.getKey());
      ts.add(sample.getValue());
    }
  }
}
