/*
 * Decompiled with CFR 0.152.
 */
package net.sourceforge.pmd.docs;

import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import net.sourceforge.pmd.Rule;
import net.sourceforge.pmd.RuleSet;
import net.sourceforge.pmd.RuleSetFactory;
import net.sourceforge.pmd.RuleSetNotFoundException;
import net.sourceforge.pmd.docs.FileWriter;
import net.sourceforge.pmd.docs.RuleSetUtils;
import net.sourceforge.pmd.docs.SidebarGenerator;
import net.sourceforge.pmd.lang.Language;
import net.sourceforge.pmd.lang.rule.RuleReference;
import net.sourceforge.pmd.lang.rule.XPathRule;
import net.sourceforge.pmd.properties.MultiValuePropertyDescriptor;
import net.sourceforge.pmd.properties.PropertyDescriptor;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;

public class RuleDocGenerator {
    private static final Logger LOG = Logger.getLogger(RuleDocGenerator.class.getName());
    private static final String LANGUAGE_INDEX_FILENAME_PATTERN = "docs/pages/pmd/rules/${language.tersename}.md";
    private static final String LANGUAGE_INDEX_PERMALINK_PATTERN = "pmd_rules_${language.tersename}.html";
    private static final String RULESET_INDEX_FILENAME_PATTERN = "docs/pages/pmd/rules/${language.tersename}/${ruleset.name}.md";
    private static final String RULESET_INDEX_PERMALINK_PATTERN = "pmd_rules_${language.tersename}_${ruleset.name}.html";
    private static final String DEPRECATION_LABEL_SMALL = "<span style=\"border-radius: 0.25em; color: #fff; padding: 0.2em 0.6em 0.3em; display: inline; background-color: #d9534f; font-size: 75%;\">Deprecated</span> ";
    private static final String DEPRECATION_LABEL = "<span style=\"border-radius: 0.25em; color: #fff; padding: 0.2em 0.6em 0.3em; display: inline; background-color: #d9534f;\">Deprecated</span> ";
    private static final String DEPRECATED_RULE_PROPERTY_MARKER = "deprecated!";
    private static final String GITHUB_SOURCE_LINK = "https://github.com/pmd/pmd/blob/master/";
    private static final Map<String, String> LANGUAGE_HIGHLIGHT_MAPPER = new HashMap<String, String>();
    private final Path root;
    private final FileWriter writer;

    public RuleDocGenerator(FileWriter writer, Path root) {
        this.writer = Objects.requireNonNull(writer, "A file writer must be provided");
        this.root = Objects.requireNonNull(root, "Root directory must be provided");
        Path docsDir = root.resolve("docs");
        if (!Files.exists(docsDir, new LinkOption[0]) || !Files.isDirectory(docsDir, new LinkOption[0])) {
            throw new IllegalArgumentException("Couldn't find \"docs\" subdirectory");
        }
    }

    public void generate(Iterator<RuleSet> registeredRulesets, List<String> additionalRulesets) {
        try {
            Map<Language, List<RuleSet>> sortedRulesets = this.sortRulesets(registeredRulesets);
            Map<Language, List<RuleSet>> sortedAdditionalRulesets = this.sortRulesets(this.resolveAdditionalRulesets(additionalRulesets));
            this.generateLanguageIndex(sortedRulesets, sortedAdditionalRulesets);
            this.generateRuleSetIndex(sortedRulesets);
            this.generateSidebar(sortedRulesets);
        }
        catch (IOException | RuleSetNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    private void generateSidebar(Map<Language, List<RuleSet>> sortedRulesets) throws IOException {
        SidebarGenerator generator = new SidebarGenerator(this.writer, this.root);
        generator.generateSidebar(sortedRulesets);
    }

    private Iterator<RuleSet> resolveAdditionalRulesets(List<String> additionalRulesets) throws RuleSetNotFoundException {
        if (additionalRulesets == null) {
            return Collections.emptyIterator();
        }
        ArrayList<RuleSet> rulesets = new ArrayList<RuleSet>();
        RuleSetFactory ruleSetFactory = new RuleSetFactory();
        for (String filename : additionalRulesets) {
            try {
                if (!filename.contains("pmd-test") && !filename.contains("pmd-core")) {
                    rulesets.add(ruleSetFactory.createRuleSet(filename));
                    continue;
                }
                LOG.fine("Ignoring ruleset " + filename);
            }
            catch (IllegalArgumentException e) {
                LOG.log(Level.WARNING, "ruleset file " + filename + " ignored (" + e.getMessage() + ")", e);
            }
        }
        return rulesets.iterator();
    }

    private Path getAbsoluteOutputPath(String filename) {
        return this.root.resolve(FilenameUtils.normalize((String)filename));
    }

    private Map<Language, List<RuleSet>> sortRulesets(Iterator<RuleSet> rulesets) throws RuleSetNotFoundException {
        TreeMap<Language, List<RuleSet>> rulesetsByLanguage = new TreeMap<Language, List<RuleSet>>();
        while (rulesets.hasNext()) {
            RuleSet ruleset = rulesets.next();
            Language language = RuleDocGenerator.getRuleSetLanguage(ruleset);
            if (!rulesetsByLanguage.containsKey(language)) {
                rulesetsByLanguage.put(language, new ArrayList());
            }
            ((List)rulesetsByLanguage.get(language)).add(ruleset);
        }
        for (List rulesetsOfOneLanguage : rulesetsByLanguage.values()) {
            Collections.sort(rulesetsOfOneLanguage, new Comparator<RuleSet>(){

                @Override
                public int compare(RuleSet o1, RuleSet o2) {
                    return o1.getName().compareToIgnoreCase(o2.getName());
                }
            });
        }
        return rulesetsByLanguage;
    }

    private static Language getRuleSetLanguage(RuleSet ruleset) {
        Collection rules = ruleset.getRules();
        if (rules.isEmpty()) {
            throw new RuntimeException("Ruleset " + ruleset.getFileName() + " is empty!");
        }
        return ((Rule)rules.iterator().next()).getLanguage();
    }

    private void generateLanguageIndex(Map<Language, List<RuleSet>> rulesets, Map<Language, List<RuleSet>> sortedAdditionalRulesets) throws IOException {
        for (Map.Entry<Language, List<RuleSet>> entry : rulesets.entrySet()) {
            String languageTersename = entry.getKey().getTerseName();
            String filename = LANGUAGE_INDEX_FILENAME_PATTERN.replace("${language.tersename}", languageTersename);
            Path path = this.getAbsoluteOutputPath(filename);
            LinkedList<String> lines = new LinkedList<String>();
            lines.add("---");
            lines.add("title: " + entry.getKey().getName() + " Rules");
            lines.add("tags: [rule_references, " + languageTersename + "]");
            lines.add("summary: Index of all built-in rules available for " + entry.getKey().getName());
            lines.add("language_name: " + entry.getKey().getName());
            lines.add("permalink: " + LANGUAGE_INDEX_PERMALINK_PATTERN.replace("${language.tersename}", languageTersename));
            lines.add("folder: pmd/rules");
            lines.add("---");
            for (RuleSet ruleset : entry.getValue()) {
                lines.add("## " + ruleset.getName());
                lines.add("");
                lines.add("{% include callout.html content=\"" + RuleDocGenerator.getRuleSetDescriptionSingleLine(ruleset).replaceAll("\"", "'") + "\" %}");
                lines.add("");
                for (Rule rule : this.getSortedRules(ruleset)) {
                    String link = RULESET_INDEX_PERMALINK_PATTERN.replace("${language.tersename}", languageTersename).replace("${ruleset.name}", RuleSetUtils.getRuleSetFilename(ruleset));
                    if (rule instanceof RuleReference) {
                        RuleReference ref = (RuleReference)rule;
                        if (ruleset.getFileName().equals(ref.getRuleSetReference().getRuleSetFileName())) {
                            lines.add("*   [" + rule.getName() + "](" + link + "#" + rule.getName().toLowerCase(Locale.ROOT) + "): " + DEPRECATION_LABEL_SMALL + "The rule has been renamed. Use instead [" + ref.getRule().getName() + "](" + link + "#" + ref.getRule().getName().toLowerCase(Locale.ROOT) + ").");
                            continue;
                        }
                        String otherLink = RULESET_INDEX_PERMALINK_PATTERN.replace("${language.tersename}", languageTersename).replace("${ruleset.name}", RuleSetUtils.getRuleSetFilename(ref.getRuleSetReference().getRuleSetFileName()));
                        lines.add("*   [" + rule.getName() + "](" + link + "#" + rule.getName().toLowerCase(Locale.ROOT) + "): " + DEPRECATION_LABEL_SMALL + "The rule has been moved to another ruleset. Use instead [" + ref.getRule().getName() + "](" + otherLink + "#" + ref.getRule().getName().toLowerCase(Locale.ROOT) + ").");
                        continue;
                    }
                    link = link + "#" + rule.getName().toLowerCase(Locale.ROOT);
                    lines.add("*   [" + rule.getName() + "](" + link + "): " + (rule.isDeprecated() ? DEPRECATION_LABEL_SMALL : "") + RuleDocGenerator.getShortRuleDescription(rule));
                }
                lines.add("");
            }
            List<RuleSet> additionalRulesetsForLanguage = sortedAdditionalRulesets.get(entry.getKey());
            if (additionalRulesetsForLanguage != null) {
                lines.add("## Additional rulesets");
                lines.add("");
                for (RuleSet ruleset : additionalRulesetsForLanguage) {
                    boolean deprecated = RuleSetUtils.isRuleSetDeprecated(ruleset);
                    String rulesetName = ruleset.getName() + " (`" + RuleSetUtils.getRuleSetClasspath(ruleset) + "`)";
                    if (!deprecated) {
                        lines.add("*   " + rulesetName + ":");
                        lines.add("");
                        lines.add("    " + RuleDocGenerator.getRuleSetDescriptionSingleLine(ruleset));
                        lines.add("");
                    } else {
                        lines.add("*   " + rulesetName + ":");
                        lines.add("");
                        lines.add("    <span style=\"border-radius: 0.25em; color: #fff; padding: 0.2em 0.6em 0.3em; display: inline; background-color: #d9534f; font-size: 75%;\">Deprecated</span>  This ruleset is for backwards compatibility.");
                        lines.add("");
                    }
                    lines.add("    It contains the following rules:");
                    lines.add("");
                    StringBuilder rules = new StringBuilder();
                    for (Rule rule : this.getSortedRules(ruleset)) {
                        if (rules.length() == 0) {
                            rules.append("    ");
                        } else {
                            rules.append(", ");
                        }
                        Rule resolvedRule = RuleSetUtils.resolveRuleReferences(rule);
                        if (resolvedRule instanceof RuleReference) {
                            RuleReference ref = (RuleReference)resolvedRule;
                            String otherLink = RULESET_INDEX_PERMALINK_PATTERN.replace("${language.tersename}", languageTersename).replace("${ruleset.name}", RuleSetUtils.getRuleSetFilename(ref.getRuleSetReference().getRuleSetFileName()));
                            rules.append("[").append(ref.getName()).append("](");
                            rules.append(otherLink).append("#").append(ref.getRule().getName().toLowerCase(Locale.ROOT)).append(")");
                            continue;
                        }
                        rules.append(rule.getName());
                    }
                    lines.add(rules.toString());
                    lines.add("");
                }
                lines.add("");
            }
            System.out.println("Generated " + path);
            this.writer.write(path, lines);
        }
    }

    private static String getShortRuleDescription(Rule rule) {
        return StringUtils.abbreviate((String)StringUtils.stripToEmpty((String)rule.getDescription().replaceAll("\n|\r", "").replaceAll("\\|", "\\\\|").replaceAll("`", "'").replaceAll("\\*", "")), (int)100);
    }

    private static String getRuleSetDescriptionSingleLine(RuleSet ruleset) {
        String description = ruleset.getDescription();
        description = description.replaceAll("\\n|\\r", " ");
        description = StringUtils.stripToEmpty((String)description);
        return description;
    }

    private static List<String> toLines(String s) {
        return Arrays.asList(s.split("\r\n|\n"));
    }

    private void generateRuleSetIndex(Map<Language, List<RuleSet>> rulesets) throws IOException {
        for (Map.Entry<Language, List<RuleSet>> entry : rulesets.entrySet()) {
            Language language = entry.getKey();
            String languageTersename = language.getTerseName();
            String languageName = language.getName();
            for (RuleSet ruleset : entry.getValue()) {
                String rulesetFilename = RuleSetUtils.getRuleSetFilename(ruleset);
                String filename = RULESET_INDEX_FILENAME_PATTERN.replace("${language.tersename}", languageTersename).replace("${ruleset.name}", rulesetFilename);
                Path path = this.getAbsoluteOutputPath(filename);
                String permalink = RULESET_INDEX_PERMALINK_PATTERN.replace("${language.tersename}", languageTersename).replace("${ruleset.name}", rulesetFilename);
                LinkedList<String> lines = new LinkedList<String>();
                lines.add("---");
                lines.add("title: " + ruleset.getName());
                lines.add("summary: " + RuleDocGenerator.getRuleSetDescriptionSingleLine(ruleset));
                lines.add("permalink: " + permalink);
                lines.add("folder: pmd/rules/" + languageTersename);
                lines.add("sidebaractiveurl: /" + LANGUAGE_INDEX_PERMALINK_PATTERN.replace("${language.tersename}", languageTersename));
                lines.add("editmepath: ../" + this.getRuleSetSourceFilepath(ruleset));
                lines.add("keywords: " + this.getRuleSetKeywords(ruleset));
                lines.add("language: " + languageName);
                lines.add("---");
                for (Rule rule : this.getSortedRules(ruleset)) {
                    lines.add("## " + rule.getName());
                    lines.add("");
                    if (rule instanceof RuleReference) {
                        RuleReference ref = (RuleReference)rule;
                        if (ruleset.getFileName().equals(ref.getRuleSetReference().getRuleSetFileName())) {
                            lines.add(DEPRECATION_LABEL);
                            lines.add("");
                            lines.add("This rule has been renamed. Use instead: [" + ref.getRule().getName() + "](#" + ref.getRule().getName().toLowerCase(Locale.ROOT) + ")");
                            lines.add("");
                        } else {
                            String otherLink = RULESET_INDEX_PERMALINK_PATTERN.replace("${language.tersename}", languageTersename).replace("${ruleset.name}", RuleSetUtils.getRuleSetFilename(ref.getRuleSetReference().getRuleSetFileName()));
                            lines.add(DEPRECATION_LABEL);
                            lines.add("");
                            lines.add("The rule has been moved to another ruleset. Use instead: [" + ref.getRule().getName() + "](" + otherLink + "#" + ref.getRule().getName().toLowerCase(Locale.ROOT) + ")");
                            lines.add("");
                        }
                    }
                    if (rule.isDeprecated()) {
                        lines.add(DEPRECATION_LABEL);
                        lines.add("");
                    }
                    if (rule.getSince() != null) {
                        lines.add("**Since:** PMD " + rule.getSince());
                        lines.add("");
                    }
                    lines.add("**Priority:** " + rule.getPriority() + " (" + rule.getPriority().getPriority() + ")");
                    lines.add("");
                    if (rule.getMinimumLanguageVersion() != null) {
                        lines.add("**Minimum Language Version:** " + rule.getLanguage().getName() + " " + rule.getMinimumLanguageVersion().getVersion());
                        lines.add("");
                    }
                    lines.addAll(RuleDocGenerator.toLines(RuleDocGenerator.stripIndentation(rule.getDescription())));
                    lines.add("");
                    if (rule instanceof XPathRule || rule instanceof RuleReference && ((RuleReference)rule).getRule() instanceof XPathRule) {
                        lines.add("**This rule is defined by the following XPath expression:**");
                        lines.add("``` xpath");
                        lines.addAll(RuleDocGenerator.toLines(StringUtils.stripToEmpty((String)((String)rule.getProperty((PropertyDescriptor)XPathRule.XPATH_DESCRIPTOR)))));
                        lines.add("```");
                    } else {
                        lines.add("**This rule is defined by the following Java class:** [" + rule.getRuleClass() + "](" + GITHUB_SOURCE_LINK + this.getRuleClassSourceFilepath(rule.getRuleClass()) + ")");
                    }
                    lines.add("");
                    if (!rule.getExamples().isEmpty()) {
                        lines.add("**Example(s):**");
                        lines.add("");
                        for (String example : rule.getExamples()) {
                            lines.add("``` " + RuleDocGenerator.mapLanguageForHighlighting(languageTersename));
                            lines.addAll(RuleDocGenerator.toLines(StringUtils.stripToEmpty((String)example)));
                            lines.add("```");
                            lines.add("");
                        }
                    }
                    ArrayList properties = new ArrayList(rule.getPropertyDescriptors());
                    properties.remove(Rule.VIOLATION_SUPPRESS_REGEX_DESCRIPTOR);
                    properties.remove(Rule.VIOLATION_SUPPRESS_XPATH_DESCRIPTOR);
                    properties.remove(XPathRule.XPATH_DESCRIPTOR);
                    properties.remove(XPathRule.VERSION_DESCRIPTOR);
                    if (!properties.isEmpty()) {
                        lines.add("**This rule has the following properties:**");
                        lines.add("");
                        lines.add("|Name|Default Value|Description|Multivalued|");
                        lines.add("|----|-------------|-----------|-----------|");
                        for (PropertyDescriptor propertyDescriptor : properties) {
                            String description = propertyDescriptor.description();
                            if (description != null && description.toLowerCase(Locale.ROOT).startsWith(DEPRECATED_RULE_PROPERTY_MARKER)) {
                                description = DEPRECATION_LABEL_SMALL + description.substring(DEPRECATED_RULE_PROPERTY_MARKER.length());
                            }
                            String defaultValue = "";
                            if (propertyDescriptor.defaultValue() != null) {
                                if (propertyDescriptor.isMultiValue()) {
                                    MultiValuePropertyDescriptor multiPropertyDescriptor = (MultiValuePropertyDescriptor)propertyDescriptor;
                                    defaultValue = multiPropertyDescriptor.asDelimitedString((Object)((List)multiPropertyDescriptor.defaultValue()));
                                    defaultValue = defaultValue.replaceAll(Pattern.quote(String.valueOf(multiPropertyDescriptor.multiValueDelimiter())), " " + multiPropertyDescriptor.multiValueDelimiter() + " ");
                                } else {
                                    defaultValue = String.valueOf(propertyDescriptor.defaultValue());
                                }
                            }
                            String multiValued = "no";
                            if (propertyDescriptor.isMultiValue()) {
                                MultiValuePropertyDescriptor multiValuePropertyDescriptor = (MultiValuePropertyDescriptor)propertyDescriptor;
                                multiValued = "yes. Delimiter is '" + multiValuePropertyDescriptor.multiValueDelimiter() + "'.";
                            }
                            lines.add("|" + propertyDescriptor.name() + "|" + defaultValue.replace("|", "\\|") + "|" + description + "|" + multiValued.replace("|", "\\|") + "|");
                        }
                        lines.add("");
                    }
                    lines.add("**Use this rule by referencing it:**");
                    lines.add("``` xml");
                    lines.add("<rule ref=\"category/" + languageTersename + "/" + rulesetFilename + ".xml/" + rule.getName() + "\" />");
                    lines.add("```");
                    lines.add("");
                }
                this.writer.write(path, lines);
                System.out.println("Generated " + path);
            }
        }
    }

    private static String stripIndentation(String description) {
        int indentation;
        if (description == null || description.isEmpty()) {
            return "";
        }
        String stripped = StringUtils.stripStart((String)description, (String)"\n\r");
        stripped = StringUtils.stripEnd((String)stripped, (String)"\n\r ");
        int strLen = stripped.length();
        for (indentation = 0; Character.isWhitespace(stripped.charAt(indentation)) && indentation < strLen; ++indentation) {
        }
        String[] lines = stripped.split("\\n");
        String prefix = StringUtils.repeat((char)' ', (int)indentation);
        StringBuilder result = new StringBuilder(stripped.length());
        if (StringUtils.isNotEmpty((CharSequence)prefix)) {
            for (int i = 0; i < lines.length; ++i) {
                String line = lines[i];
                if (i > 0) {
                    result.append("\n");
                }
                result.append(StringUtils.removeStart((String)line, (String)prefix));
            }
        } else {
            result.append(stripped);
        }
        return result.toString();
    }

    private static String mapLanguageForHighlighting(String languageTersename) {
        if (LANGUAGE_HIGHLIGHT_MAPPER.containsKey(languageTersename)) {
            return LANGUAGE_HIGHLIGHT_MAPPER.get(languageTersename);
        }
        return languageTersename;
    }

    private String getRuleSetKeywords(RuleSet ruleset) {
        LinkedList<String> ruleNames = new LinkedList<String>();
        for (Rule rule : ruleset.getRules()) {
            ruleNames.add(rule.getName());
        }
        return ruleset.getName() + ", " + StringUtils.join(ruleNames, (String)", ");
    }

    private List<Rule> getSortedRules(RuleSet ruleset) {
        ArrayList<Rule> sortedRules = new ArrayList<Rule>(ruleset.getRules());
        Collections.sort(sortedRules, new Comparator<Rule>(){

            @Override
            public int compare(Rule o1, Rule o2) {
                return o1.getName().compareToIgnoreCase(o2.getName());
            }
        });
        return sortedRules;
    }

    private String getRuleSetSourceFilepath(RuleSet ruleset) throws IOException {
        final String rulesetFilename = FilenameUtils.normalize((String)StringUtils.chomp((String)ruleset.getFileName()));
        final LinkedList foundPathResult = new LinkedList();
        Files.walkFileTree(this.root, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                String path = file.toString();
                if (path.contains("src") && path.endsWith(rulesetFilename)) {
                    foundPathResult.add(file);
                    return FileVisitResult.TERMINATE;
                }
                return super.visitFile(file, attrs);
            }
        });
        if (!foundPathResult.isEmpty()) {
            Path foundPath = (Path)foundPathResult.get(0);
            foundPath = this.root.relativize(foundPath);
            return FilenameUtils.normalize((String)foundPath.toString(), (boolean)true);
        }
        return StringUtils.chomp((String)ruleset.getFileName());
    }

    private String getRuleClassSourceFilepath(String ruleClass) throws IOException {
        final String relativeSourceFilename = ruleClass.replaceAll("\\.", Matcher.quoteReplacement(File.separator)) + ".java";
        final LinkedList foundPathResult = new LinkedList();
        Files.walkFileTree(this.root, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                String path = file.toString();
                if (path.contains("src") && path.endsWith(relativeSourceFilename)) {
                    foundPathResult.add(file);
                    return FileVisitResult.TERMINATE;
                }
                return super.visitFile(file, attrs);
            }
        });
        if (!foundPathResult.isEmpty()) {
            Path foundPath = (Path)foundPathResult.get(0);
            foundPath = this.root.relativize(foundPath);
            return FilenameUtils.normalize((String)foundPath.toString(), (boolean)true);
        }
        return FilenameUtils.normalize((String)relativeSourceFilename, (boolean)true);
    }

    static {
        LANGUAGE_HIGHLIGHT_MAPPER.put("ecmascript", "javascript");
        LANGUAGE_HIGHLIGHT_MAPPER.put("pom", "xml");
        LANGUAGE_HIGHLIGHT_MAPPER.put("apex", "java");
        LANGUAGE_HIGHLIGHT_MAPPER.put("plsql", "sql");
    }
}

