/*
 * Decompiled with CFR 0.152.
 */
package net.sourceforge.pmd.internal.util;

import java.io.IOException;
import java.io.Reader;
import java.io.UncheckedIOException;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.sql.SQLException;
import java.util.HashSet;
import java.util.List;
import net.sourceforge.pmd.AbstractConfiguration;
import net.sourceforge.pmd.cpd.CPDConfiguration;
import net.sourceforge.pmd.internal.util.FileUtil;
import net.sourceforge.pmd.internal.util.IOUtil;
import net.sourceforge.pmd.lang.document.FileCollector;
import net.sourceforge.pmd.lang.document.FileId;
import net.sourceforge.pmd.lang.document.InternalApiBridge;
import net.sourceforge.pmd.util.database.DBMSMetadata;
import net.sourceforge.pmd.util.database.DBURI;
import net.sourceforge.pmd.util.database.SourceObject;
import net.sourceforge.pmd.util.log.PmdReporter;
import net.sourceforge.pmd.util.log.internal.ErrorsAsWarningsReporter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class FileCollectionUtil {
    private static final Logger LOG = LoggerFactory.getLogger(FileCollectionUtil.class);

    private FileCollectionUtil() {
    }

    public static void collectFiles(CPDConfiguration cpdConfiguration, FileCollector collector) {
        if (cpdConfiguration.isSkipDuplicates()) {
            HashSet alreadyAddedFileNamesWithSize = new HashSet();
            collector.setFileFilter(fileId -> {
                Path path = Paths.get(fileId.getAbsolutePath(), new String[0]);
                if (!Files.isRegularFile(path, new LinkOption[0])) {
                    LOG.debug("Path {} is not a regular file, skipping in fileFilter", (Object)path);
                    return true;
                }
                try {
                    String signature = path.getFileName() + "_" + Files.size(path);
                    if (!alreadyAddedFileNamesWithSize.add(signature)) {
                        LOG.info("Skipping {} since it appears to be a duplicate file and --skip-duplicate-files is set", (Object)path);
                        return false;
                    }
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
                return true;
            });
        }
        FileCollectionUtil.collectFiles((AbstractConfiguration)cpdConfiguration, collector);
    }

    public static void collectFiles(AbstractConfiguration configuration, FileCollector collector) {
        if (configuration.getSourceEncoding() != null) {
            collector.setCharset(configuration.getSourceEncoding());
            collector.setRecursive(configuration.collectFilesRecursively());
        }
        FileCollectionUtil.collectFiles(collector, configuration.getInputPathList());
        if (configuration.getUri() != null) {
            FileCollectionUtil.collectDB(collector, configuration.getUri());
        }
        if (configuration.getInputFile() != null) {
            FileCollectionUtil.collectFileList(collector, configuration.getInputFile());
        }
        if (configuration.getIgnoreFile() != null || !configuration.getExcludes().isEmpty()) {
            LOG.debug("Now collecting files to exclude.");
            ErrorsAsWarningsReporter mutedLog = new ErrorsAsWarningsReporter(collector.getReporter());
            try (FileCollector excludeCollector = InternalApiBridge.newCollector(collector, (PmdReporter)mutedLog);){
                if (configuration.getIgnoreFile() != null) {
                    FileCollectionUtil.collectFileList(excludeCollector, configuration.getIgnoreFile());
                }
                FileCollectionUtil.collectFiles(excludeCollector, configuration.getExcludes());
                collector.exclude(excludeCollector);
            }
        }
    }

    public static void collectFiles(FileCollector collector, List<Path> filePaths) {
        for (Path rootLocation : filePaths) {
            try {
                FileCollectionUtil.addRoot(collector, rootLocation);
            }
            catch (IOException e) {
                collector.getReporter().errorEx("Error collecting {0}", new Object[]{rootLocation}, e);
            }
        }
    }

    public static void collectFileList(FileCollector collector, Path fileList) {
        List<Path> filePaths;
        LOG.debug("Reading file list {}.", (Object)fileList);
        if (!Files.exists(fileList, new LinkOption[0])) {
            collector.getReporter().error("No such file {0}", fileList);
            return;
        }
        try {
            filePaths = FileUtil.readFilelistEntries(fileList);
        }
        catch (IOException e) {
            collector.getReporter().errorEx("Error reading {0}", new Object[]{fileList}, e);
            return;
        }
        FileCollectionUtil.collectFiles(collector, filePaths);
    }

    private static void addRoot(FileCollector collector, Path path) throws IOException {
        String pathStr = path.toString();
        if (!Files.exists(path, new LinkOption[0])) {
            collector.getReporter().error("No such file {0}", path);
            return;
        }
        if (Files.isDirectory(path, new LinkOption[0])) {
            LOG.debug("Adding directory {}.", (Object)path);
            collector.addDirectory(path);
        } else if (pathStr.endsWith(".zip") || pathStr.endsWith(".jar")) {
            collector.addZipFileWithContent(path);
        } else if (Files.isRegularFile(path, new LinkOption[0])) {
            LOG.debug("Adding regular file {}.", (Object)path);
            collector.addFile(path);
        } else {
            LOG.debug("Ignoring {}: not a regular file or directory", (Object)path);
        }
    }

    public static void collectDB(FileCollector collector, URI uri) {
        try {
            LOG.debug("Connecting to {}", (Object)uri);
            DBURI dbUri = new DBURI(uri);
            DBMSMetadata dbmsMetadata = new DBMSMetadata(dbUri);
            LOG.trace("DBMSMetadata retrieved");
            List<SourceObject> sourceObjectList = dbmsMetadata.getSourceObjectList();
            LOG.trace("Located {} database source objects", (Object)sourceObjectList.size());
            for (SourceObject sourceObject : sourceObjectList) {
                String falseFilePath = sourceObject.getPseudoFileName();
                LOG.trace("Adding database source object {}", (Object)falseFilePath);
                try {
                    Reader sourceCode = dbmsMetadata.getSourceCode(sourceObject);
                    try {
                        String source = IOUtil.readToString(sourceCode);
                        collector.addSourceFile(FileId.fromPathLikeString(falseFilePath), source);
                    }
                    finally {
                        if (sourceCode == null) continue;
                        sourceCode.close();
                    }
                }
                catch (SQLException ex) {
                    collector.getReporter().warnEx("Cannot get SourceCode for {0}  - skipping ...", new Object[]{falseFilePath}, ex);
                }
            }
        }
        catch (ClassNotFoundException e) {
            collector.getReporter().errorEx("Cannot get files from DB - probably missing database JDBC driver", e);
        }
        catch (Exception e) {
            collector.getReporter().errorEx("Cannot get files from DB - ''{0}''", new Object[]{uri}, e);
        }
    }
}

