/*
 * Decompiled with CFR 0.152.
 */
package net.sourceforge.pmd.lang.document;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Collection;
import java.util.List;
import net.sourceforge.pmd.AbstractConfiguration;
import net.sourceforge.pmd.PMDConfiguration;
import net.sourceforge.pmd.PmdAnalysis;
import net.sourceforge.pmd.internal.util.FileCollectionUtil;
import net.sourceforge.pmd.internal.util.IOUtil;
import net.sourceforge.pmd.lang.DummyLanguageModule;
import net.sourceforge.pmd.lang.Language;
import net.sourceforge.pmd.lang.LanguageRegistry;
import net.sourceforge.pmd.lang.LanguageVersion;
import net.sourceforge.pmd.lang.LanguageVersionDiscoverer;
import net.sourceforge.pmd.lang.document.FileCollector;
import net.sourceforge.pmd.lang.document.FileId;
import net.sourceforge.pmd.lang.document.TestMessageReporter;
import net.sourceforge.pmd.lang.document.TextFile;
import net.sourceforge.pmd.util.CollectionUtil;
import net.sourceforge.pmd.util.log.PmdReporter;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;

class FileCollectorTest {
    private static final Path RESOURCES = Paths.get("src/test/resources/net/sourceforge/pmd/lang/document/filecollectortest/", new String[0]);
    @TempDir
    private Path tempFolder;

    FileCollectorTest() {
    }

    @Test
    void testAddFile() throws IOException {
        Path foo = this.newFile(this.tempFolder, "foo.dummy");
        Path bar = this.newFile(this.tempFolder, "bar.unknown");
        FileCollector collector = this.newCollector();
        Assertions.assertTrue((boolean)collector.addFile(foo), (String)"should be dummy language");
        Assertions.assertFalse((boolean)collector.addFile(bar), (String)"should be unknown language");
        this.assertCollected(collector, CollectionUtil.listOf((Object)FileId.fromPath((Path)foo), (Object[])new FileId[0]));
    }

    @Test
    void testAddFileForceLanguage() throws IOException {
        Path bar = this.newFile(this.tempFolder, "bar.unknown");
        DummyLanguageModule dummy = DummyLanguageModule.getInstance();
        FileCollector collector = this.newCollector(dummy.getDefaultVersion());
        Assertions.assertTrue((boolean)collector.addFile(bar, (Language)dummy), (String)"should be unknown language");
        this.assertCollected(collector, CollectionUtil.listOf((Object)FileId.fromPath((Path)bar), (Object[])new FileId[0]));
        this.assertNoErrors(collector);
    }

    @Test
    void testAddFileNotExists() {
        FileCollector collector = this.newCollector();
        Assertions.assertFalse((boolean)collector.addFile(this.tempFolder.resolve("does_not_exist.dummy")));
        Assertions.assertEquals((int)1, (int)collector.getReporter().numErrors());
    }

    @Test
    void testAddFileNotAFile() throws IOException {
        Path dir = this.tempFolder.resolve("src");
        Files.createDirectories(dir, new FileAttribute[0]);
        FileCollector collector = this.newCollector();
        Assertions.assertFalse((boolean)collector.addFile(dir));
        Assertions.assertEquals((int)1, (int)collector.getReporter().numErrors());
    }

    @Test
    void testAddDirectory() throws IOException {
        Path root = this.tempFolder;
        Path foo = this.newFile(root, "src/foo.dummy");
        this.newFile(root, "src/bar.unknown");
        Path bar = this.newFile(root, "src/x/bar.dummy");
        FileCollector collector = this.newCollector();
        collector.addDirectory(root.resolve("src"));
        this.assertCollected(collector, CollectionUtil.listOf((Object)FileId.fromPath((Path)foo), (Object[])new FileId[]{FileId.fromPath((Path)bar)}));
    }

    @Test
    void testGetApplicableFiles() {
        FileCollector collector = this.newCollector();
        FileCollectionUtil.collectFileList((FileCollector)collector, (Path)RESOURCES.resolve("filelist.txt"));
        List applicableFiles = collector.getCollectedFiles();
        MatcherAssert.assertThat((Object)applicableFiles, (Matcher)Matchers.hasSize((int)2));
        MatcherAssert.assertThat((Object)((TextFile)applicableFiles.get(0)).getFileId().getFileName(), (Matcher)Matchers.equalTo((Object)"anotherfile.dummy"));
        MatcherAssert.assertThat((Object)((TextFile)applicableFiles.get(1)).getFileId().getFileName(), (Matcher)Matchers.equalTo((Object)"somefile.dummy"));
    }

    @Test
    void testGetApplicableFilesMultipleLines() {
        FileCollector collector = this.newCollector();
        FileCollectionUtil.collectFileList((FileCollector)collector, (Path)RESOURCES.resolve("filelist2.txt"));
        List applicableFiles = collector.getCollectedFiles();
        MatcherAssert.assertThat((Object)applicableFiles, (Matcher)Matchers.hasSize((int)2));
        FileCollectorTest.assertFilenameIs((TextFile)applicableFiles.get(0), "anotherfile.dummy");
        FileCollectorTest.assertFilenameIs((TextFile)applicableFiles.get(1), "somefile.dummy");
    }

    @Test
    void testGetApplicableFilesWithIgnores() {
        FileCollector collector = this.newCollector();
        PMDConfiguration configuration = new PMDConfiguration();
        configuration.setInputFilePath(RESOURCES.resolve("filelist3.txt"));
        configuration.setIgnoreFilePath(RESOURCES.resolve("ignorelist.txt"));
        FileCollectionUtil.collectFiles((AbstractConfiguration)configuration, (FileCollector)collector);
        List applicableFiles = collector.getCollectedFiles();
        MatcherAssert.assertThat((Object)applicableFiles, (Matcher)Matchers.hasSize((int)2));
        FileCollectorTest.assertFilenameIs((TextFile)applicableFiles.get(0), "somefile2.dummy");
        FileCollectorTest.assertFilenameIs((TextFile)applicableFiles.get(1), "somefile4.dummy");
    }

    @Test
    void testRelativizeWith() {
        PMDConfiguration conf = new PMDConfiguration();
        conf.setInputFilePath(RESOURCES.resolve("filelist2.txt"));
        conf.addRelativizeRoot(Paths.get("src/test/resources", new String[0]));
        try (PmdAnalysis pmd = PmdAnalysis.create((PMDConfiguration)conf);){
            List files = pmd.files().getCollectedFiles();
            MatcherAssert.assertThat((Object)files, (Matcher)Matchers.hasSize((int)2));
            FileCollectorTest.assertHasName((TextFile)files.get(0), IOUtil.normalizePath((String)"net/sourceforge/pmd/lang/document/filecollectortest/src/anotherfile.dummy"), pmd);
            FileCollectorTest.assertHasName((TextFile)files.get(1), IOUtil.normalizePath((String)"net/sourceforge/pmd/lang/document/filecollectortest/src/somefile.dummy"), pmd);
        }
    }

    @Test
    void testRelativizeWithOtherDir() {
        PMDConfiguration conf = new PMDConfiguration();
        conf.setInputFilePath(RESOURCES.resolve("filelist4.txt"));
        conf.addRelativizeRoot(RESOURCES.resolve("src"));
        try (PmdAnalysis pmd = PmdAnalysis.create((PMDConfiguration)conf);){
            List files = pmd.files().getCollectedFiles();
            MatcherAssert.assertThat((Object)files, (Matcher)Matchers.hasSize((int)3));
            FileCollectorTest.assertHasName((TextFile)files.get(0), ".." + IOUtil.normalizePath((String)"/otherSrc/somefile.dummy"), pmd);
            FileCollectorTest.assertHasName((TextFile)files.get(1), "anotherfile.dummy", pmd);
            FileCollectorTest.assertHasName((TextFile)files.get(2), "somefile.dummy", pmd);
        }
    }

    @Test
    void testRelativizeWithSeveralDirs() {
        PMDConfiguration conf = new PMDConfiguration();
        conf.setInputFilePath(RESOURCES.resolve("filelist4.txt"));
        conf.addRelativizeRoot(RESOURCES.resolve("src"));
        conf.addRelativizeRoot(RESOURCES.resolve("otherSrc"));
        try (PmdAnalysis pmd = PmdAnalysis.create((PMDConfiguration)conf);){
            List files = pmd.files().getCollectedFiles();
            MatcherAssert.assertThat((Object)files, (Matcher)Matchers.hasSize((int)3));
            FileCollectorTest.assertHasName((TextFile)files.get(0), "somefile.dummy", pmd);
            FileCollectorTest.assertHasName((TextFile)files.get(1), "anotherfile.dummy", pmd);
            FileCollectorTest.assertHasName((TextFile)files.get(2), "somefile.dummy", pmd);
        }
    }

    @Test
    void testUseAbsolutePaths() {
        PMDConfiguration conf = new PMDConfiguration();
        conf.setInputFilePath(RESOURCES.resolve("filelist4.txt"));
        conf.addRelativizeRoot(RESOURCES.toAbsolutePath().getRoot());
        try (PmdAnalysis pmd = PmdAnalysis.create((PMDConfiguration)conf);){
            List files = pmd.files().getCollectedFiles();
            MatcherAssert.assertThat((Object)files, (Matcher)Matchers.hasSize((int)3));
            FileCollectorTest.assertHasName((TextFile)files.get(0), RESOURCES.resolve("otherSrc/somefile.dummy").toAbsolutePath().toString(), pmd);
            FileCollectorTest.assertHasName((TextFile)files.get(1), RESOURCES.resolve("src/anotherfile.dummy").toAbsolutePath().toString(), pmd);
            FileCollectorTest.assertHasName((TextFile)files.get(2), RESOURCES.resolve("src/somefile.dummy").toAbsolutePath().toString(), pmd);
        }
    }

    @Test
    void testGetApplicableFilesWithDirAndIgnores() {
        PMDConfiguration configuration = new PMDConfiguration();
        configuration.addInputPath(RESOURCES.resolve("src"));
        configuration.setIgnoreFilePath(RESOURCES.resolve("ignorelist.txt"));
        FileCollector collector = this.newCollector();
        FileCollectionUtil.collectFiles((AbstractConfiguration)configuration, (FileCollector)collector);
        List applicableFiles = collector.getCollectedFiles();
        MatcherAssert.assertThat((Object)applicableFiles, (Matcher)Matchers.hasSize((int)4));
        FileCollectorTest.assertFilenameIs((TextFile)applicableFiles.get(0), "anotherfile.dummy");
        FileCollectorTest.assertFilenameIs((TextFile)applicableFiles.get(1), "somefile.dummy");
        FileCollectorTest.assertFilenameIs((TextFile)applicableFiles.get(2), "somefile2.dummy");
        FileCollectorTest.assertFilenameIs((TextFile)applicableFiles.get(3), "somefile4.dummy");
    }

    public static void assertHasName(TextFile textFile, String expected, PmdAnalysis pmd) {
        MatcherAssert.assertThat((Object)pmd.fileNameRenderer().getDisplayName(textFile), (Matcher)Matchers.equalTo((Object)expected));
    }

    private static void assertFilenameIs(TextFile textFile, String suffix) {
        MatcherAssert.assertThat((Object)textFile.getFileId().getFileName(), (Matcher)Matchers.is((Object)suffix));
    }

    private Path newFile(Path root, String path) throws IOException {
        Path resolved = root.resolve(path);
        Files.createDirectories(resolved.getParent(), new FileAttribute[0]);
        Files.createFile(resolved, new FileAttribute[0]);
        return resolved;
    }

    private void assertCollected(FileCollector collector, List<FileId> expected) {
        List actual = CollectionUtil.map((Collection)collector.getCollectedFiles(), TextFile::getFileId);
        Assertions.assertEquals(expected, (Object)actual);
    }

    private void assertNoErrors(FileCollector collector) {
        Assertions.assertEquals((int)0, (int)collector.getReporter().numErrors(), (String)"No errors expected");
    }

    private FileCollector newCollector() {
        return this.newCollector(null);
    }

    private FileCollector newCollector(LanguageVersion forcedVersion) {
        LanguageVersionDiscoverer discoverer = new LanguageVersionDiscoverer(LanguageRegistry.PMD, forcedVersion);
        return FileCollector.newCollector((LanguageVersionDiscoverer)discoverer, (PmdReporter)new TestMessageReporter());
    }
}

