/*
 * Decompiled with CFR 0.152.
 */
package net.sourceforge.pmd.internal.util;

import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.concurrent.BrokenBarrierException;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.Semaphore;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import net.sourceforge.pmd.internal.util.ClasspathClassLoader;
import net.sourceforge.pmd.internal.util.IOUtil;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ValueSource;

class ClasspathClassLoaderTest {
    @TempDir
    private Path tempDir;
    private static final String CUSTOM_JAR_RESOURCE = "net/sourceforge/pmd/Sample.txt";
    private static final String CUSTOM_JAR_RESOURCE2 = "net/sourceforge/pmd/Sample2.txt";
    private static final String CUSTOM_JAR_RESOURCE_CONTENT = "Test\n";

    ClasspathClassLoaderTest() {
    }

    @Test
    void loadEmptyClasspathWithParent() throws IOException {
        try (ClasspathClassLoader loader = new ClasspathClassLoader("", ClasspathClassLoader.class.getClassLoader());
             InputStream resource = loader.getResourceAsStream("java/lang/Object.class");){
            Assertions.assertNotNull((Object)resource);
            try (DataInputStream data = new DataInputStream(resource);){
                this.assertClassFile(data, Integer.valueOf(System.getProperty("java.specification.version")));
            }
        }
    }

    @Test
    void loadEmptyClasspathNoParent() throws IOException {
        try (ClasspathClassLoader loader = new ClasspathClassLoader("", null);
             InputStream resource = loader.getResourceAsStream("java/lang/Object.class");){
            Assertions.assertNotNull((Object)resource);
            try (DataInputStream data = new DataInputStream(resource);){
                this.assertClassFile(data, Integer.valueOf(System.getProperty("java.specification.version")));
            }
        }
    }

    private Path prepareCustomJar() throws IOException {
        Path jarPath = this.tempDir.resolve("custom.jar");
        try (ZipOutputStream out = new ZipOutputStream(Files.newOutputStream(jarPath, new OpenOption[0]));){
            out.putNextEntry(new ZipEntry(CUSTOM_JAR_RESOURCE));
            out.write(CUSTOM_JAR_RESOURCE_CONTENT.getBytes(StandardCharsets.UTF_8));
            out.putNextEntry(new ZipEntry(CUSTOM_JAR_RESOURCE2));
            out.write(CUSTOM_JAR_RESOURCE_CONTENT.getBytes(StandardCharsets.UTF_8));
        }
        return jarPath;
    }

    @Test
    void loadFromJar() throws IOException {
        Path jarPath = this.prepareCustomJar();
        String classpath = jarPath.toString();
        try (ClasspathClassLoader loader = new ClasspathClassLoader(classpath, null);
             InputStream in = loader.getResourceAsStream(CUSTOM_JAR_RESOURCE);){
            Assertions.assertNotNull((Object)in);
            String s = IOUtil.readToString((InputStream)in, (Charset)StandardCharsets.UTF_8);
            Assertions.assertEquals((Object)CUSTOM_JAR_RESOURCE_CONTENT, (Object)s);
        }
    }

    @Test
    void loadMultithreadedFromJar() throws IOException, InterruptedException {
        Path jarPath = this.prepareCustomJar();
        final String classpath = jarPath.toString();
        int numberOfThreads = 2;
        final CyclicBarrier waitForClosed = new CyclicBarrier(numberOfThreads);
        final Semaphore grabResource = new Semaphore(1);
        final ArrayList caughtExceptions = new ArrayList();
        class ThreadRunnable
        extends Thread {
            private final int number;

            ThreadRunnable(int number) {
                super("Thread" + number);
                this.number = number;
            }

            @Override
            public void run() {
                try (ClasspathClassLoader loader = new ClasspathClassLoader(classpath, null);){
                    InputStream stream;
                    grabResource.acquire();
                    try {
                        stream = loader.getResourceAsStream(ClasspathClassLoaderTest.CUSTOM_JAR_RESOURCE);
                    }
                    finally {
                        grabResource.release();
                    }
                    try (InputStream in = stream;){
                        Assertions.assertNotNull((Object)in);
                        if (this.number > 0) {
                            waitForClosed.await();
                        }
                        String s = IOUtil.readToString((InputStream)in, (Charset)StandardCharsets.UTF_8);
                        Assertions.assertEquals((Object)ClasspathClassLoaderTest.CUSTOM_JAR_RESOURCE_CONTENT, (Object)s);
                    }
                }
                catch (Exception e) {
                    caughtExceptions.add(e);
                    throw new RuntimeException(e);
                }
                finally {
                    try {
                        if (this.number == 0) {
                            waitForClosed.await();
                        }
                    }
                    catch (InterruptedException e) {
                        throw new RuntimeException(e);
                    }
                    catch (BrokenBarrierException e) {
                        throw new RuntimeException(e);
                    }
                }
            }
        }
        ArrayList<ThreadRunnable> threads = new ArrayList<ThreadRunnable>(numberOfThreads);
        for (int i = 0; i < numberOfThreads; ++i) {
            threads.add(new ThreadRunnable(i));
        }
        threads.forEach(Thread::start);
        for (Thread thread : threads) {
            thread.join();
        }
        Assertions.assertTrue((boolean)caughtExceptions.isEmpty());
    }

    @ParameterizedTest
    @ValueSource(ints={11, 17, 21})
    void loadFromJava(int javaVersion) throws IOException {
        Path javaHome = Paths.get(System.getProperty("user.home"), "openjdk" + javaVersion);
        Assumptions.assumeTrue((boolean)Files.isDirectory(javaHome, new LinkOption[0]), (String)("Couldn't find java" + javaVersion + " installation at " + javaHome));
        Path jrtfsPath = javaHome.resolve("lib/jrt-fs.jar");
        Assertions.assertTrue((boolean)Files.isRegularFile(jrtfsPath, new LinkOption[0]), (String)("java" + javaVersion + " installation is incomplete. " + jrtfsPath + " not found!"));
        String classPath = jrtfsPath.toString();
        try (ClasspathClassLoader loader = new ClasspathClassLoader(classPath, null);){
            Assertions.assertEquals((Object)javaHome.toString(), (Object)loader.javaHome);
            try (InputStream stream = loader.getResourceAsStream("java/lang/Object.class");){
                this.assertClassFile(stream, javaVersion);
            }
            Assertions.assertNull((Object)loader.getResourceAsStream("ClassInDefaultPackage.class"));
            stream = loader.getResourceAsStream("java.base/module-info.class");
            try {
                this.assertClassFile(stream, javaVersion);
            }
            finally {
                if (stream != null) {
                    stream.close();
                }
            }
        }
    }

    private void assertClassFile(InputStream inputStream, int javaVersion) throws IOException {
        Assertions.assertNotNull((Object)inputStream);
        try (DataInputStream data = new DataInputStream(inputStream);){
            int magicNumber = data.readInt();
            Assertions.assertEquals((int)-889275714, (int)magicNumber);
            data.readUnsignedShort();
            int majorVersion = data.readUnsignedShort();
            Assertions.assertEquals((int)(44 + javaVersion), (int)majorVersion);
        }
    }

    private static byte[] readBytes(InputStream stream) throws IOException {
        Assertions.assertNotNull((Object)stream);
        ByteArrayOutputStream data = new ByteArrayOutputStream();
        try (InputStream inputStream = stream;){
            int count;
            byte[] buffer = new byte[8192];
            while ((count = inputStream.read(buffer)) != -1) {
                data.write(buffer, 0, count);
            }
        }
        return data.toByteArray();
    }

    @Test
    void findModuleInfoFromJar() throws IOException {
        try (ClasspathClassLoader loader = new ClasspathClassLoader("", ClasspathClassLoader.class.getClassLoader());){
            String junitJupiterApiModule = "org.junit.platform.suite.api/module-info.class";
            URL resource = loader.getResource(junitJupiterApiModule);
            Assertions.assertNotNull((Object)resource);
            MatcherAssert.assertThat((Object)resource.toString(), (Matcher)Matchers.endsWith((String)".jar!/module-info.class"));
            byte[] fromUrl = ClasspathClassLoaderTest.readBytes(resource.openStream());
            byte[] fromStream = ClasspathClassLoaderTest.readBytes(loader.getResourceAsStream(junitJupiterApiModule));
            Assertions.assertArrayEquals((byte[])fromUrl, (byte[])fromStream, (String)"getResource and getResourceAsStream should return the same module");
        }
    }
}

