/*
 * Decompiled with CFR 0.152.
 */
package net.sourceforge.pmd.cpd;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import net.sourceforge.pmd.cpd.CPDConfiguration;
import net.sourceforge.pmd.cpd.CPDListener;
import net.sourceforge.pmd.cpd.CPDReport;
import net.sourceforge.pmd.cpd.CpdAnalysis;
import net.sourceforge.pmd.cpd.Match;
import net.sourceforge.pmd.internal.util.IOUtil;
import net.sourceforge.pmd.lang.DummyLanguageModule;
import net.sourceforge.pmd.lang.Language;
import net.sourceforge.pmd.lang.ast.LexException;
import net.sourceforge.pmd.lang.ast.impl.javacc.MalformedSourceException;
import net.sourceforge.pmd.lang.document.FileId;
import net.sourceforge.pmd.lang.document.TextFile;
import net.sourceforge.pmd.reporting.Report;
import net.sourceforge.pmd.util.log.PmdReporter;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.condition.EnabledOnOs;
import org.junit.jupiter.api.condition.OS;
import org.junit.jupiter.api.io.TempDir;
import org.mockito.Mockito;

class CpdAnalysisTest {
    private static final String BASE_TEST_RESOURCE_PATH = "src/test/resources/net/sourceforge/pmd/cpd/files/";
    private static final String TARGET_TEST_RESOURCE_PATH = "target/classes/net/sourceforge/pmd/cpd/files/";
    @TempDir
    private Path tempDir;
    private CPDConfiguration config = new CPDConfiguration();

    CpdAnalysisTest() {
    }

    @BeforeEach
    void setup() {
        this.config.setOnlyRecognizeLanguage((Language)DummyLanguageModule.getInstance());
        this.config.setMinimumTileSize(10);
    }

    private void prepareSymLinks() throws Exception {
        Runtime runtime = Runtime.getRuntime();
        if (!new File(TARGET_TEST_RESOURCE_PATH, "symlink-for-real-file.txt").exists()) {
            runtime.exec(new String[]{"ln", "-s", "src/test/resources/net/sourceforge/pmd/cpd/files/real-file.txt", "target/classes/net/sourceforge/pmd/cpd/files/symlink-for-real-file.txt"}).waitFor();
        }
        if (!new File(BASE_TEST_RESOURCE_PATH, "this-is-a-broken-sym-link-for-test").exists()) {
            runtime.exec(new String[]{"ln", "-s", "broken-sym-link", "target/classes/net/sourceforge/pmd/cpd/files/this-is-a-broken-sym-link-for-test"}).waitFor();
        }
    }

    @Test
    @EnabledOnOs(value={OS.LINUX})
    void testFileSectionWithBrokenSymlinks() throws Exception {
        this.prepareSymLinks();
        FileCountAssertListener listener = new FileCountAssertListener(0);
        try (CpdAnalysis cpd = CpdAnalysis.create((CPDConfiguration)this.config);){
            cpd.setCpdListener((CPDListener)listener);
            Assertions.assertFalse((boolean)cpd.files().addFile(Paths.get(BASE_TEST_RESOURCE_PATH, "this-is-a-broken-sym-link-for-test")));
            cpd.performAnalysis();
        }
        listener.verify();
    }

    @Test
    @EnabledOnOs(value={OS.LINUX})
    void testFileAddedAsSymlinkAndReal() throws Exception {
        this.prepareSymLinks();
        FileCountAssertListener listener = new FileCountAssertListener(1);
        try (CpdAnalysis cpd = CpdAnalysis.create((CPDConfiguration)this.config);){
            cpd.setCpdListener((CPDListener)listener);
            Assertions.assertTrue((boolean)cpd.files().addFile(Paths.get(BASE_TEST_RESOURCE_PATH, "real-file.txt")));
            Assertions.assertFalse((boolean)cpd.files().addFile(Paths.get(BASE_TEST_RESOURCE_PATH, "symlink-for-real-file.txt")));
            cpd.performAnalysis();
        }
        listener.verify();
    }

    @Test
    @EnabledOnOs(value={OS.LINUX})
    void testNoFileAddedAsSymlink() throws Exception {
        this.prepareSymLinks();
        FileCountAssertListener listener = new FileCountAssertListener(0);
        try (CpdAnalysis cpd = CpdAnalysis.create((CPDConfiguration)this.config);){
            cpd.setCpdListener((CPDListener)listener);
            Assertions.assertFalse((boolean)cpd.files().addFile(Paths.get(BASE_TEST_RESOURCE_PATH, "symlink-for-real-file.txt")));
            cpd.performAnalysis();
        }
        listener.verify();
    }

    @Test
    void testFileAddedWithRelativePath() throws Exception {
        FileCountAssertListener listener = new FileCountAssertListener(1);
        try (CpdAnalysis cpd = CpdAnalysis.create((CPDConfiguration)this.config);){
            cpd.setCpdListener((CPDListener)listener);
            Assertions.assertTrue((boolean)cpd.files().addFile(Paths.get("./src/test/resources/net/sourceforge/pmd/cpd/files/", "real-file.txt")));
            cpd.performAnalysis();
        }
        listener.verify();
    }

    @Test
    void testFileOrderRelevance() throws Exception {
        Path dup1 = Paths.get("./src/test/resources/net/sourceforge/pmd/cpd/files/", "dup1.txt");
        Path dup2 = Paths.get("./src/test/resources/net/sourceforge/pmd/cpd/files/", "dup2.txt");
        try (CpdAnalysis cpd = CpdAnalysis.create((CPDConfiguration)this.config);){
            Assertions.assertTrue((boolean)cpd.files().addFile(dup2));
            Assertions.assertTrue((boolean)cpd.files().addFile(dup1));
            cpd.performAnalysis(report -> {
                List matches = report.getMatches();
                Assertions.assertFalse((boolean)matches.isEmpty());
                for (Match match : matches) {
                    Assertions.assertEquals((Object)"dup1.txt", (Object)match.getFirstMark().getFileId().getFileName());
                    Assertions.assertEquals((Object)"dup2.txt", (Object)match.getSecondMark().getFileId().getFileName());
                }
            });
        }
        cpd = CpdAnalysis.create((CPDConfiguration)this.config);
        try {
            Assertions.assertTrue((boolean)cpd.files().addFile(dup1));
            Assertions.assertTrue((boolean)cpd.files().addFile(dup2));
            cpd.performAnalysis(report -> {
                List matches = report.getMatches();
                Assertions.assertFalse((boolean)matches.isEmpty());
                for (Match match : matches) {
                    Assertions.assertEquals((Object)"dup1.txt", (Object)match.getFirstMark().getFileId().getFileName());
                    Assertions.assertEquals((Object)"dup2.txt", (Object)match.getSecondMark().getFileId().getFileName());
                }
            });
        }
        finally {
            if (cpd != null) {
                cpd.close();
            }
        }
    }

    @Test
    void testNoSkipLexicalErrors() throws IOException {
        PmdReporter reporter = (PmdReporter)Mockito.mock(PmdReporter.class);
        this.config.setReporter(reporter);
        this.config.setSkipLexicalErrors(false);
        try (CpdAnalysis cpd = CpdAnalysis.create((CPDConfiguration)this.config);){
            Assertions.assertTrue((boolean)cpd.files().addSourceFile(FileId.fromPathLikeString((String)"foo.dummy"), ":throw_lex_source_exception:"));
            Assertions.assertTrue((boolean)cpd.files().addSourceFile(FileId.fromPathLikeString((String)"foo2.dummy"), ":throw_malformed_source_exception:"));
            cpd.performAnalysis();
        }
        ((PmdReporter)Mockito.verify((Object)reporter)).errorEx((String)Mockito.eq((Object)"Error while tokenizing"), (Throwable)Mockito.any(LexException.class));
        ((PmdReporter)Mockito.verify((Object)reporter)).errorEx((String)Mockito.eq((Object)"Error while tokenizing"), (Throwable)Mockito.any(MalformedSourceException.class));
        ((PmdReporter)Mockito.verify((Object)reporter)).error((String)Mockito.eq((Object)"Errors were detected while lexing source, exiting because --skip-lexical-errors is unset."), new Object[0]);
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{reporter});
    }

    @Test
    void reportShouldContainProcessingErrors() throws IOException {
        AtomicReference report = new AtomicReference();
        PmdReporter reporter = (PmdReporter)Mockito.mock(PmdReporter.class);
        this.config.setReporter(reporter);
        try (CpdAnalysis cpd = CpdAnalysis.create((CPDConfiguration)this.config);){
            Assertions.assertTrue((boolean)cpd.files().addSourceFile(FileId.fromPathLikeString((String)"foo.dummy"), ":throw_lex_source_exception:"));
            Assertions.assertTrue((boolean)cpd.files().addSourceFile(FileId.fromPathLikeString((String)"foo2.dummy"), ":throw_malformed_source_exception:"));
            cpd.performAnalysis(report::set);
        }
        Assertions.assertNotNull(report.get(), (String)"CPD aborted early without producing a report");
        List processingErrors = ((CPDReport)report.get()).getProcessingErrors();
        Assertions.assertEquals((int)2, (int)processingErrors.size());
        Report.ProcessingError error1 = (Report.ProcessingError)processingErrors.get(0);
        Assertions.assertEquals((Object)"foo.dummy", (Object)error1.getFileId().getFileName());
        MatcherAssert.assertThat((Object)error1.getDetail(), (Matcher)Matchers.containsString((String)LexException.class.getSimpleName()));
        Report.ProcessingError error2 = (Report.ProcessingError)processingErrors.get(1);
        Assertions.assertEquals((Object)"foo2.dummy", (Object)error2.getFileId().getFileName());
        MatcherAssert.assertThat((Object)error2.getDetail(), (Matcher)Matchers.containsString((String)MalformedSourceException.class.getSimpleName()));
        ((PmdReporter)Mockito.verify((Object)reporter)).errorEx((String)Mockito.eq((Object)"Skipping file"), (Throwable)Mockito.any(LexException.class));
        ((PmdReporter)Mockito.verify((Object)reporter)).errorEx((String)Mockito.eq((Object)"Skipping file"), (Throwable)Mockito.any(MalformedSourceException.class));
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{reporter});
    }

    @Test
    void reportToNonExistentFile(@TempDir Path tmpDir) throws IOException {
        Path reportFile = tmpDir.resolve("cpd.txt");
        Assertions.assertFalse((boolean)Files.exists(reportFile, new LinkOption[0]), (String)("Report file " + reportFile + " should not exist"));
        this.testReportFile(reportFile);
    }

    @Test
    void reportToExistingFileShouldOverwrite(@TempDir Path tmpDir) throws IOException {
        Path reportFile = tmpDir.resolve("cpd.txt");
        Assertions.assertFalse((boolean)Files.exists(reportFile, new LinkOption[0]), (String)("Report file " + reportFile + " should not exist"));
        String sentinel = "EMPTY_FILE";
        Files.write(reportFile, "EMPTY_FILE".getBytes(), new OpenOption[0]);
        Assertions.assertTrue((boolean)Files.exists(reportFile, new LinkOption[0]), (String)("Report file " + reportFile + " should have been created"));
        String reportContents = this.testReportFile(reportFile);
        MatcherAssert.assertThat((Object)reportContents, (Matcher)Matchers.not((Matcher)Matchers.containsString((String)"EMPTY_FILE")));
    }

    private String testReportFile(Path reportFile) throws IOException {
        PmdReporter reporter = (PmdReporter)Mockito.mock(PmdReporter.class);
        this.config.setReporter(reporter);
        this.config.setRendererName("text");
        this.config.setReportFile(reportFile);
        Path dup1 = Paths.get("./src/test/resources/net/sourceforge/pmd/cpd/files/", "dup1.txt");
        Path dup2 = Paths.get("./src/test/resources/net/sourceforge/pmd/cpd/files/", "dup2.txt");
        try (CpdAnalysis cpd = CpdAnalysis.create((CPDConfiguration)this.config);){
            Assertions.assertTrue((boolean)cpd.files().addFile(dup2));
            Assertions.assertTrue((boolean)cpd.files().addFile(dup1));
            cpd.performAnalysis();
        }
        Assertions.assertTrue((boolean)Files.exists(reportFile, new LinkOption[0]), (String)("Report file " + reportFile + " should have been created"));
        String reportContents = IOUtil.readFileToString((File)reportFile.toFile());
        MatcherAssert.assertThat((Object)reportContents, (Matcher)Matchers.containsString((String)"duplication in the following files"));
        MatcherAssert.assertThat((Object)reportContents, (Matcher)Matchers.containsString((String)dup1.toAbsolutePath().normalize().toString()));
        MatcherAssert.assertThat((Object)reportContents, (Matcher)Matchers.containsString((String)dup2.toAbsolutePath().normalize().toString()));
        return reportContents;
    }

    @Test
    void testSkipLexicalErrors() throws IOException {
        PmdReporter reporter = (PmdReporter)Mockito.mock(PmdReporter.class);
        this.config.setReporter(reporter);
        try (CpdAnalysis cpd = CpdAnalysis.create((CPDConfiguration)this.config);){
            Assertions.assertTrue((boolean)cpd.files().addSourceFile(FileId.fromPathLikeString((String)"foo.dummy"), ":throw_lex_source_exception:"));
            Assertions.assertTrue((boolean)cpd.files().addSourceFile(FileId.fromPathLikeString((String)"foo2.dummy"), ":throw_malformed_source_exception:"));
            cpd.performAnalysis();
        }
        ((PmdReporter)Mockito.verify((Object)reporter)).errorEx((String)Mockito.eq((Object)"Skipping file"), (Throwable)Mockito.any(LexException.class));
        ((PmdReporter)Mockito.verify((Object)reporter)).errorEx((String)Mockito.eq((Object)"Skipping file"), (Throwable)Mockito.any(MalformedSourceException.class));
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{reporter});
    }

    @Test
    void duplicatedFilesShouldBeSkipped() throws IOException {
        String filename = "file1.dummy";
        Path aFile1 = Files.createDirectory(this.tempDir.resolve("a"), new FileAttribute[0]).resolve(filename).toAbsolutePath();
        Path bFile1 = Files.createDirectory(this.tempDir.resolve("b"), new FileAttribute[0]).resolve(filename).toAbsolutePath();
        Files.write(aFile1, "Same content".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        Files.write(bFile1, "Same content".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        this.config.setSkipDuplicates(true);
        this.config.setInputPathList(Arrays.asList(this.tempDir));
        try (CpdAnalysis cpd = CpdAnalysis.create((CPDConfiguration)this.config);){
            List collectedFiles = cpd.files().getCollectedFiles();
            collectedFiles.stream().map(TextFile::getFileId).map(FileId::getAbsolutePath).forEach(System.out::println);
            Assertions.assertEquals((int)1, (int)collectedFiles.size());
            String collectedFile = ((TextFile)collectedFiles.get(0)).getFileId().getAbsolutePath();
            Assertions.assertTrue((collectedFile.equals(aFile1.toString()) || collectedFile.equals(bFile1.toString()) ? 1 : 0) != 0);
        }
    }

    private static class FileCountAssertListener
    implements CPDListener {
        private int expectedFilesCount;
        private int files;

        FileCountAssertListener(int expectedFilesCount) {
            this.expectedFilesCount = expectedFilesCount;
            this.files = 0;
        }

        public void addedFile(int fileCount) {
            ++this.files;
            if (this.files > this.expectedFilesCount) {
                Assertions.fail((String)"File was added!");
            }
        }

        public void phaseUpdate(int phase) {
        }

        public void verify() {
            Assertions.assertEquals((int)this.expectedFilesCount, (int)this.files, (String)("Expected " + this.expectedFilesCount + " files, but " + this.files + " have been added."));
        }
    }
}

