/*
 * Decompiled with CFR 0.152.
 */
package net.sourceforge.pmd.cache.internal;

import com.github.stefanbirkner.systemlambda.SystemLambda;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import net.sourceforge.pmd.PmdCoreTestUtils;
import net.sourceforge.pmd.cache.internal.FileAnalysisCache;
import net.sourceforge.pmd.lang.Language;
import net.sourceforge.pmd.lang.LanguageVersion;
import net.sourceforge.pmd.lang.document.FileId;
import net.sourceforge.pmd.lang.document.FileLocation;
import net.sourceforge.pmd.lang.document.TextDocument;
import net.sourceforge.pmd.lang.document.TextFile;
import net.sourceforge.pmd.lang.document.TextFileContent;
import net.sourceforge.pmd.lang.document.TextRange2d;
import net.sourceforge.pmd.lang.rule.Rule;
import net.sourceforge.pmd.lang.rule.internal.RuleSets;
import net.sourceforge.pmd.reporting.FileAnalysisListener;
import net.sourceforge.pmd.reporting.InternalApiBridge;
import net.sourceforge.pmd.reporting.Report;
import net.sourceforge.pmd.reporting.RuleViolation;
import net.sourceforge.pmd.util.CollectionUtil;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;
import org.mockito.Mockito;
import org.mockito.stubbing.Answer;

class FileAnalysisCacheTest {
    @TempDir
    private Path tempFolder;
    private File unexistingCacheFile;
    private File newCacheFile;
    private File emptyCacheFile;
    private TextDocument sourceFile;
    private TextFile sourceFileBackend;
    private final LanguageVersion dummyVersion = PmdCoreTestUtils.dummyVersion();

    FileAnalysisCacheTest() {
    }

    @BeforeEach
    public void setUp() throws IOException {
        this.unexistingCacheFile = this.tempFolder.resolve("non-existing-file.cache").toFile();
        this.newCacheFile = this.tempFolder.resolve("pmd-analysis.cache").toFile();
        this.emptyCacheFile = Files.createTempFile(this.tempFolder, null, null, new FileAttribute[0]).toFile();
        Path sourceFile = this.tempFolder.resolve("Source.java");
        Files.write(sourceFile, (Iterable<? extends CharSequence>)CollectionUtil.listOf((Object)"dummy text", (Object[])new String[0]), new OpenOption[0]);
        this.sourceFileBackend = TextFile.forPath((Path)sourceFile, (Charset)Charset.defaultCharset(), (LanguageVersion)this.dummyVersion);
        this.sourceFile = TextDocument.create((TextFile)this.sourceFileBackend);
    }

    @Test
    void testLoadFromNonExistingFile() throws IOException {
        FileAnalysisCache cache = new FileAnalysisCache(this.unexistingCacheFile);
        Assertions.assertNotNull((Object)cache, (String)"Cache creation from non existing file failed.");
    }

    @Test
    void testLoadFromEmptyFile() throws IOException {
        FileAnalysisCache cache = new FileAnalysisCache(this.emptyCacheFile);
        Assertions.assertNotNull((Object)cache, (String)"Cache creation from empty file failed.");
    }

    @Test
    void testLoadFromDirectoryShouldntThrow() throws IOException {
        new FileAnalysisCache(this.tempFolder.toFile());
    }

    @Test
    void testLoadFromUnreadableFileShouldntThrow() throws IOException {
        this.emptyCacheFile.setReadable(false);
        new FileAnalysisCache(this.emptyCacheFile);
    }

    @Test
    void testStoreCreatesFile() throws Exception {
        FileAnalysisCache cache = new FileAnalysisCache(this.unexistingCacheFile);
        cache.persist();
        Assertions.assertTrue((boolean)this.unexistingCacheFile.exists(), (String)"Cache file doesn't exist after store");
    }

    @Test
    void testStoreOnUnwritableFileShouldntThrow() throws IOException {
        this.emptyCacheFile.setWritable(false);
        FileAnalysisCache cache = new FileAnalysisCache(this.emptyCacheFile);
        cache.persist();
    }

    @Test
    void testStorePersistsFilesWithViolations() throws IOException {
        FileAnalysisCache cache = new FileAnalysisCache(this.newCacheFile);
        cache.checkValidity((RuleSets)Mockito.mock(RuleSets.class), (ClassLoader)Mockito.mock(ClassLoader.class), (Collection)CollectionUtil.setOf((Object)this.sourceFileBackend, (Object[])new TextFile[0]));
        FileAnalysisListener cacheListener = cache.startFileAnalysis(this.sourceFile);
        cache.isUpToDate(this.sourceFile);
        RuleViolation rv = (RuleViolation)Mockito.mock(RuleViolation.class);
        TextRange2d textLocation = TextRange2d.range2d((int)1, (int)2, (int)3, (int)4);
        Mockito.when((Object)rv.getLocation()).thenReturn((Object)FileLocation.range((FileId)this.sourceFile.getFileId(), (TextRange2d)textLocation));
        Rule rule = (Rule)Mockito.mock(Rule.class, (Answer)Mockito.RETURNS_SMART_NULLS);
        Mockito.when((Object)rule.getLanguage()).thenReturn((Object)((Language)Mockito.mock(Language.class)));
        Mockito.when((Object)rv.getRule()).thenReturn((Object)rule);
        cacheListener.onRuleViolation(rv);
        cache.persist();
        FileAnalysisCache reloadedCache = new FileAnalysisCache(this.newCacheFile);
        reloadedCache.checkValidity((RuleSets)Mockito.mock(RuleSets.class), (ClassLoader)Mockito.mock(ClassLoader.class), (Collection)CollectionUtil.setOf((Object)this.sourceFileBackend, (Object[])new TextFile[0]));
        Assertions.assertTrue((boolean)reloadedCache.isUpToDate(this.sourceFile), (String)"Cache believes unmodified file with violations is not up to date");
        List cachedViolations = reloadedCache.getCachedViolations(this.sourceFile);
        Assertions.assertEquals((int)1, (int)cachedViolations.size(), (String)"Cached rule violations count mismatch");
        RuleViolation cachedViolation = (RuleViolation)cachedViolations.get(0);
        Assertions.assertSame((Object)this.sourceFile.getFileId(), (Object)cachedViolation.getFileId());
        Assertions.assertEquals((int)textLocation.getStartLine(), (int)cachedViolation.getBeginLine());
        Assertions.assertEquals((int)textLocation.getStartColumn(), (int)cachedViolation.getBeginColumn());
        Assertions.assertEquals((int)textLocation.getEndLine(), (int)cachedViolation.getEndLine());
        Assertions.assertEquals((int)textLocation.getEndColumn(), (int)cachedViolation.getEndColumn());
    }

    @Test
    void testStorePersistsFilesWithViolationsAndProcessingErrors() throws IOException {
        FileAnalysisCache cache = new FileAnalysisCache(this.newCacheFile);
        cache.checkValidity((RuleSets)Mockito.mock(RuleSets.class), (ClassLoader)Mockito.mock(ClassLoader.class), (Collection)CollectionUtil.setOf((Object)this.sourceFileBackend, (Object[])new TextFile[0]));
        FileAnalysisListener cacheListener = cache.startFileAnalysis(this.sourceFile);
        cache.isUpToDate(this.sourceFile);
        cacheListener.onError(new Report.ProcessingError((Throwable)new RuntimeException("some rule failed"), this.sourceFile.getFileId()));
        RuleViolation rv = (RuleViolation)Mockito.mock(RuleViolation.class);
        TextRange2d textLocation = TextRange2d.range2d((int)1, (int)2, (int)3, (int)4);
        Mockito.when((Object)rv.getLocation()).thenReturn((Object)FileLocation.range((FileId)this.sourceFile.getFileId(), (TextRange2d)textLocation));
        Rule rule = (Rule)Mockito.mock(Rule.class, (Answer)Mockito.RETURNS_SMART_NULLS);
        Mockito.when((Object)rule.getLanguage()).thenReturn((Object)((Language)Mockito.mock(Language.class)));
        Mockito.when((Object)rv.getRule()).thenReturn((Object)rule);
        cacheListener.onRuleViolation(rv);
        cache.persist();
        FileAnalysisCache reloadedCache = new FileAnalysisCache(this.newCacheFile);
        reloadedCache.checkValidity((RuleSets)Mockito.mock(RuleSets.class), (ClassLoader)Mockito.mock(ClassLoader.class), (Collection)CollectionUtil.setOf((Object)this.sourceFileBackend, (Object[])new TextFile[0]));
        Assertions.assertFalse((boolean)reloadedCache.isUpToDate(this.sourceFile), (String)"Cache believes file is up to date although processing errors happened earlier");
        List cachedViolations = reloadedCache.getCachedViolations(this.sourceFile);
        Assertions.assertTrue((boolean)cachedViolations.isEmpty(), (String)"There should be no cached rule violations");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testDisplayNameIsRespected() throws Exception {
        Rule rule = (Rule)Mockito.mock(Rule.class, (Answer)Mockito.RETURNS_SMART_NULLS);
        Mockito.when((Object)rule.getLanguage()).thenReturn((Object)((Language)Mockito.mock(Language.class)));
        TextRange2d textLocation = TextRange2d.range2d((int)1, (int)2, (int)3, (int)4);
        TextFile mockFile = (TextFile)Mockito.mock(TextFile.class);
        Mockito.when((Object)mockFile.getFileId()).thenReturn((Object)FileId.fromPathLikeString((String)"a/bc"));
        Mockito.when((Object)mockFile.getLanguageVersion()).thenReturn((Object)this.dummyVersion);
        Mockito.when((Object)mockFile.readContents()).thenReturn((Object)TextFileContent.fromCharSeq((CharSequence)"abc"));
        FileAnalysisCache cache = new FileAnalysisCache(this.newCacheFile);
        cache.checkValidity((RuleSets)Mockito.mock(RuleSets.class), (ClassLoader)Mockito.mock(ClassLoader.class), (Collection)CollectionUtil.setOf((Object)this.sourceFileBackend, (Object[])new TextFile[0]));
        try (TextDocument doc0 = TextDocument.create((TextFile)mockFile);){
            cache.isUpToDate(doc0);
            try (FileAnalysisListener listener = cache.startFileAnalysis(doc0);){
                listener.onRuleViolation((RuleViolation)InternalApiBridge.createRuleViolation((Rule)rule, (FileLocation)FileLocation.range((FileId)doc0.getFileId(), (TextRange2d)textLocation), (String)"message", Collections.emptyMap()));
            }
        }
        finally {
            cache.persist();
        }
        this.reloadWithOneViolation(mockFile);
    }

    private void reloadWithOneViolation(TextFile mockFile) throws IOException {
        FileAnalysisCache reloadedCache = new FileAnalysisCache(this.newCacheFile);
        reloadedCache.checkValidity((RuleSets)Mockito.mock(RuleSets.class), (ClassLoader)Mockito.mock(ClassLoader.class), (Collection)CollectionUtil.setOf((Object)mockFile, (Object[])new TextFile[0]));
        try (TextDocument doc1 = TextDocument.create((TextFile)mockFile);){
            Assertions.assertTrue((boolean)reloadedCache.isUpToDate(doc1), (String)"Cache believes unmodified file with violations is not up to date");
            List cachedViolations = reloadedCache.getCachedViolations(doc1);
            Assertions.assertEquals((int)1, (int)cachedViolations.size(), (String)"Cached rule violations count mismatch");
            RuleViolation cachedViolation = (RuleViolation)cachedViolations.get(0);
            Assertions.assertEquals((Object)mockFile.getFileId(), (Object)cachedViolation.getLocation().getFileId());
        }
    }

    @Test
    void testCacheValidityWithNoChanges() throws IOException {
        RuleSets rs = (RuleSets)Mockito.mock(RuleSets.class);
        ClassLoader cl = (ClassLoader)Mockito.mock(ClassLoader.class);
        this.setupCacheWithFiles(this.newCacheFile, rs, cl);
        FileAnalysisCache reloadedCache = new FileAnalysisCache(this.newCacheFile);
        reloadedCache.checkValidity(rs, cl, (Collection)CollectionUtil.setOf((Object)this.sourceFileBackend, (Object[])new TextFile[0]));
        Assertions.assertTrue((boolean)reloadedCache.isUpToDate(this.sourceFile), (String)"Cache believes unmodified file is not up to date without ruleset / classpath changes");
    }

    @Test
    void testCacheValidityWithIrrelevantChanges() throws IOException {
        RuleSets rs = (RuleSets)Mockito.mock(RuleSets.class);
        URLClassLoader cl = (URLClassLoader)Mockito.mock(URLClassLoader.class);
        Mockito.when((Object)cl.getURLs()).thenReturn((Object)new URL[0]);
        this.setupCacheWithFiles(this.newCacheFile, rs, cl);
        File classpathFile = Files.createTempFile(this.tempFolder, null, "foo.xml", new FileAttribute[0]).toFile();
        Mockito.when((Object)cl.getURLs()).thenReturn((Object)new URL[]{classpathFile.toURI().toURL()});
        FileAnalysisCache reloadedCache = new FileAnalysisCache(this.newCacheFile);
        reloadedCache.checkValidity(rs, (ClassLoader)cl, (Collection)CollectionUtil.setOf((Object)this.sourceFileBackend, (Object[])new TextFile[0]));
        Assertions.assertTrue((boolean)reloadedCache.isUpToDate(this.sourceFile), (String)"Cache believes unmodified file is not up to date without ruleset / classpath changes");
    }

    @Test
    void testRulesetChangeInvalidatesCache() throws IOException {
        RuleSets rs = (RuleSets)Mockito.mock(RuleSets.class);
        ClassLoader cl = (ClassLoader)Mockito.mock(ClassLoader.class);
        this.setupCacheWithFiles(this.newCacheFile, rs, cl);
        FileAnalysisCache reloadedCache = new FileAnalysisCache(this.newCacheFile);
        Mockito.when((Object)rs.getChecksum()).thenReturn((Object)1L);
        reloadedCache.checkValidity(rs, cl, Collections.emptySet());
        Assertions.assertFalse((boolean)reloadedCache.isUpToDate(this.sourceFile), (String)"Cache believes unmodified file is up to date after ruleset changed");
    }

    @Test
    void testAuxClasspathNonExistingAuxclasspathEntriesIgnored() throws MalformedURLException, IOException {
        RuleSets rs = (RuleSets)Mockito.mock(RuleSets.class);
        URLClassLoader cl = (URLClassLoader)Mockito.mock(URLClassLoader.class);
        Mockito.when((Object)cl.getURLs()).thenReturn((Object)new URL[]{this.tempFolder.resolve("non-existing-dir").toFile().toURI().toURL()});
        this.setupCacheWithFiles(this.newCacheFile, rs, cl);
        FileAnalysisCache analysisCache = new FileAnalysisCache(this.newCacheFile);
        Mockito.when((Object)cl.getURLs()).thenReturn((Object)new URL[0]);
        analysisCache.checkValidity(rs, (ClassLoader)cl, (Collection)CollectionUtil.setOf((Object)this.sourceFileBackend, (Object[])new TextFile[0]));
        Assertions.assertTrue((boolean)analysisCache.isUpToDate(this.sourceFile), (String)"Cache believes unmodified file is not up to date after non-existing auxclasspath entry removed");
    }

    @Test
    void testAuxClasspathChangeWithoutDFAorTypeResolutionDoesNotInvalidatesCache() throws MalformedURLException, IOException {
        RuleSets rs = (RuleSets)Mockito.mock(RuleSets.class);
        URLClassLoader cl = (URLClassLoader)Mockito.mock(URLClassLoader.class);
        Mockito.when((Object)cl.getURLs()).thenReturn((Object)new URL[0]);
        this.setupCacheWithFiles(this.newCacheFile, rs, cl);
        FileAnalysisCache reloadedCache = new FileAnalysisCache(this.newCacheFile);
        Mockito.when((Object)cl.getURLs()).thenReturn((Object)new URL[]{Files.createTempFile(this.tempFolder, null, null, new FileAttribute[0]).toFile().toURI().toURL()});
        reloadedCache.checkValidity(rs, (ClassLoader)cl, (Collection)CollectionUtil.setOf((Object)this.sourceFileBackend, (Object[])new TextFile[0]));
        Assertions.assertTrue((boolean)reloadedCache.isUpToDate(this.sourceFile), (String)"Cache believes unmodified file is not up to date after auxclasspath changed when no rule cares");
    }

    @Test
    void testAuxClasspathChangeInvalidatesCache() throws MalformedURLException, IOException {
        RuleSets rs = (RuleSets)Mockito.mock(RuleSets.class);
        URLClassLoader cl = (URLClassLoader)Mockito.mock(URLClassLoader.class);
        Mockito.when((Object)cl.getURLs()).thenReturn((Object)new URL[0]);
        this.setupCacheWithFiles(this.newCacheFile, rs, cl);
        FileAnalysisCache reloadedCache = new FileAnalysisCache(this.newCacheFile);
        File classpathFile = Files.createTempFile(this.tempFolder, null, "foo.class", new FileAttribute[0]).toFile();
        Mockito.when((Object)cl.getURLs()).thenReturn((Object)new URL[]{classpathFile.toURI().toURL()});
        Files.write(classpathFile.toPath(), "some text".getBytes(), new OpenOption[0]);
        Rule r = (Rule)Mockito.mock(Rule.class);
        Mockito.when((Object)r.getLanguage()).thenReturn((Object)((Language)Mockito.mock(Language.class)));
        Mockito.when((Object)rs.getAllRules()).thenReturn(Collections.singleton(r));
        reloadedCache.checkValidity(rs, (ClassLoader)cl, Collections.emptySet());
        Assertions.assertFalse((boolean)reloadedCache.isUpToDate(this.sourceFile), (String)"Cache believes unmodified file is up to date after auxclasspath changed");
    }

    @Test
    void testAuxClasspathJarContentsChangeInvalidatesCache() throws MalformedURLException, IOException {
        RuleSets rs = (RuleSets)Mockito.mock(RuleSets.class);
        URLClassLoader cl = (URLClassLoader)Mockito.mock(URLClassLoader.class);
        File classpathFile = Files.createTempFile(this.tempFolder, null, "foo.class", new FileAttribute[0]).toFile();
        Mockito.when((Object)cl.getURLs()).thenReturn((Object)new URL[]{classpathFile.toURI().toURL()});
        Rule r = (Rule)Mockito.mock(Rule.class);
        Mockito.when((Object)r.getLanguage()).thenReturn((Object)((Language)Mockito.mock(Language.class)));
        Mockito.when((Object)rs.getAllRules()).thenReturn(Collections.singleton(r));
        this.setupCacheWithFiles(this.newCacheFile, rs, cl);
        Files.write(classpathFile.toPath(), "some text".getBytes(), new OpenOption[0]);
        FileAnalysisCache reloadedCache = new FileAnalysisCache(this.newCacheFile);
        reloadedCache.checkValidity(rs, (ClassLoader)cl, Collections.emptySet());
        Assertions.assertFalse((boolean)reloadedCache.isUpToDate(this.sourceFile), (String)"Cache believes cache is up to date when a auxclasspath file changed");
    }

    @Test
    void testClasspathNonExistingEntryIsIgnored() throws Exception {
        SystemLambda.restoreSystemProperties(() -> {
            RuleSets rs = (RuleSets)Mockito.mock(RuleSets.class);
            ClassLoader cl = (ClassLoader)Mockito.mock(ClassLoader.class);
            System.setProperty("java.class.path", System.getProperty("java.class.path") + File.pathSeparator + this.tempFolder.toFile().getAbsolutePath() + File.separator + "non-existing-dir");
            FileAnalysisCache reloadedCache = new FileAnalysisCache(this.newCacheFile);
            try {
                reloadedCache.checkValidity(rs, cl, Collections.emptySet());
            }
            catch (Exception e) {
                Assertions.fail((String)"Validity check failed when classpath includes non-existing directories");
            }
        });
    }

    @Test
    void testClasspathChangeInvalidatesCache() throws Exception {
        SystemLambda.restoreSystemProperties(() -> {
            RuleSets rs = (RuleSets)Mockito.mock(RuleSets.class);
            ClassLoader cl = (ClassLoader)Mockito.mock(ClassLoader.class);
            File classpathFile = Files.createTempFile(this.tempFolder, null, "foo.class", new FileAttribute[0]).toFile();
            this.setupCacheWithFiles(this.newCacheFile, rs, cl);
            Files.write(classpathFile.toPath(), "some text".getBytes(), new OpenOption[0]);
            System.setProperty("java.class.path", System.getProperty("java.class.path") + File.pathSeparator + classpathFile.getAbsolutePath());
            FileAnalysisCache reloadedCache = new FileAnalysisCache(this.newCacheFile);
            reloadedCache.checkValidity(rs, cl, Collections.emptySet());
            Assertions.assertFalse((boolean)reloadedCache.isUpToDate(this.sourceFile), (String)"Cache believes cache is up to date when the classpath changed");
        });
    }

    @Test
    void testClasspathContentsChangeInvalidatesCache() throws Exception {
        SystemLambda.restoreSystemProperties(() -> {
            RuleSets rs = (RuleSets)Mockito.mock(RuleSets.class);
            ClassLoader cl = (ClassLoader)Mockito.mock(ClassLoader.class);
            File classpathFile = Files.createTempFile(this.tempFolder, null, "foo.class", new FileAttribute[0]).toFile();
            Files.write(classpathFile.toPath(), "some text".getBytes(), new OpenOption[0]);
            System.setProperty("java.class.path", System.getProperty("java.class.path") + File.pathSeparator + classpathFile.getAbsolutePath());
            this.setupCacheWithFiles(this.newCacheFile, rs, cl);
            Files.write(classpathFile.toPath(), "some other text".getBytes(), new OpenOption[0]);
            FileAnalysisCache reloadedCache = new FileAnalysisCache(this.newCacheFile);
            reloadedCache.checkValidity(rs, cl, Collections.emptySet());
            Assertions.assertFalse((boolean)reloadedCache.isUpToDate(this.sourceFile), (String)"Cache believes cache is up to date when a classpath file changed");
        });
    }

    @Test
    void testWildcardClasspath() throws Exception {
        SystemLambda.restoreSystemProperties(() -> {
            RuleSets rs = (RuleSets)Mockito.mock(RuleSets.class);
            ClassLoader cl = (ClassLoader)Mockito.mock(ClassLoader.class);
            this.setupCacheWithFiles(this.newCacheFile, rs, cl);
            this.createZipFile("mylib1.jar");
            this.createZipFile("mylib2.jar");
            System.setProperty("java.class.path", System.getProperty("java.class.path") + File.pathSeparator + this.tempFolder.toFile().getAbsolutePath() + "/*");
            FileAnalysisCache reloadedCache = new FileAnalysisCache(this.newCacheFile);
            Assertions.assertFalse((boolean)reloadedCache.isUpToDate(this.sourceFile), (String)"Cache believes cache is up to date when the classpath changed");
        });
    }

    @Test
    void testWildcardClasspathContentsChangeInvalidatesCache() throws Exception {
        SystemLambda.restoreSystemProperties(() -> {
            RuleSets rs = (RuleSets)Mockito.mock(RuleSets.class);
            ClassLoader cl = (ClassLoader)Mockito.mock(ClassLoader.class);
            File classpathJar1 = this.createZipFile("mylib1.jar");
            this.createZipFile("mylib2.jar");
            System.setProperty("java.class.path", System.getProperty("java.class.path") + File.pathSeparator + this.tempFolder.toFile().getAbsolutePath() + "/*");
            this.setupCacheWithFiles(this.newCacheFile, rs, cl);
            classpathJar1.delete();
            this.createZipFile(classpathJar1.getName(), 2);
            FileAnalysisCache reloadedCache = new FileAnalysisCache(this.newCacheFile);
            reloadedCache.checkValidity(rs, cl, Collections.emptySet());
            Assertions.assertFalse((boolean)reloadedCache.isUpToDate(this.sourceFile), (String)"Cache believes cache is up to date when the classpath changed");
        });
    }

    @Test
    void testUnknownFileIsNotUpToDate() throws IOException {
        FileAnalysisCache cache = new FileAnalysisCache(this.newCacheFile);
        Assertions.assertFalse((boolean)cache.isUpToDate(this.sourceFile), (String)"Cache believes an unknown file is up to date");
    }

    @Test
    void testFileIsUpToDate() throws IOException {
        this.setupCacheWithFiles(this.newCacheFile, (RuleSets)Mockito.mock(RuleSets.class), (ClassLoader)Mockito.mock(ClassLoader.class));
        FileAnalysisCache cache = new FileAnalysisCache(this.newCacheFile);
        cache.checkValidity((RuleSets)Mockito.mock(RuleSets.class), (ClassLoader)Mockito.mock(ClassLoader.class), (Collection)CollectionUtil.setOf((Object)this.sourceFileBackend, (Object[])new TextFile[0]));
        Assertions.assertTrue((boolean)cache.isUpToDate(this.sourceFile), (String)"Cache believes a known, unchanged file is not up to date");
    }

    @Test
    void testFileIsNotUpToDateWhenEdited() throws IOException {
        this.setupCacheWithFiles(this.newCacheFile, (RuleSets)Mockito.mock(RuleSets.class), (ClassLoader)Mockito.mock(ClassLoader.class));
        TextFileContent text = TextFileContent.fromCharSeq((CharSequence)"some text");
        Assertions.assertEquals((Object)System.lineSeparator(), (Object)text.getLineTerminator());
        this.sourceFileBackend.writeContents(text);
        this.sourceFile = TextDocument.create((TextFile)this.sourceFileBackend);
        FileAnalysisCache cache = new FileAnalysisCache(this.newCacheFile);
        Assertions.assertFalse((boolean)cache.isUpToDate(this.sourceFile), (String)"Cache believes a known, changed file is up to date");
    }

    private void setupCacheWithFiles(File cacheFile, RuleSets ruleSets, ClassLoader classLoader) throws IOException {
        FileAnalysisCache cache = new FileAnalysisCache(cacheFile);
        cache.checkValidity(ruleSets, classLoader, (Collection)CollectionUtil.setOf((Object)this.sourceFileBackend, (Object[])new TextFile[0]));
        cache.isUpToDate(this.sourceFile);
        cache.persist();
    }

    private File createZipFile(String fileName) throws IOException {
        return this.createZipFile(fileName, 1);
    }

    private File createZipFile(String fileName, int numEntries) throws IOException {
        File zipFile = Files.createTempFile(this.tempFolder, null, fileName, new FileAttribute[0]).toFile();
        try (ZipOutputStream zipOS = new ZipOutputStream(Files.newOutputStream(zipFile.toPath(), new OpenOption[0]));){
            for (int i = 0; i < numEntries; ++i) {
                zipOS.putNextEntry(new ZipEntry("lib/foo" + i + ".class"));
                zipOS.write(("content of " + fileName + " entry " + i).getBytes(StandardCharsets.UTF_8));
                zipOS.closeEntry();
            }
        }
        return zipFile;
    }
}

