/*
 * Decompiled with CFR 0.152.
 */
package net.sourceforge.pmd.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.CharArrayReader;
import java.io.Closeable;
import java.io.File;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import net.sourceforge.pmd.internal.util.IOUtil;
import org.apache.commons.lang3.SystemUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

class IOUtilTest {
    IOUtilTest() {
    }

    @Test
    void testReadAllBytes() throws IOException {
        byte[] data = "12345".getBytes(StandardCharsets.UTF_8);
        try (ByteArrayInputStream stream = new ByteArrayInputStream(data);){
            byte[] bytes = IOUtil.toByteArray((InputStream)stream);
            Assertions.assertEquals((int)5, (int)bytes.length);
            Assertions.assertArrayEquals((byte[])data, (byte[])bytes);
        }
    }

    @Test
    void testToByteArrayResize() throws IOException {
        int size = 16394;
        byte[] data = new byte[size];
        for (int i = 0; i < size; ++i) {
            data[i] = 65;
        }
        try (ByteArrayInputStream stream = new ByteArrayInputStream(data);){
            byte[] bytes = IOUtil.toByteArray((InputStream)stream);
            Assertions.assertEquals((int)size, (int)bytes.length);
            Assertions.assertArrayEquals((byte[])data, (byte[])bytes);
        }
    }

    @Test
    void testSkipFully() throws IOException {
        byte[] data = "12345".getBytes(StandardCharsets.UTF_8);
        try (ByteArrayInputStream stream = new ByteArrayInputStream(data);){
            Assertions.assertThrows(IllegalArgumentException.class, () -> IOUtil.skipFully((InputStream)stream, (long)-1L));
            Assertions.assertEquals((long)3L, (long)IOUtil.skipFully((InputStream)stream, (long)3L));
            byte[] bytes = IOUtil.toByteArray((InputStream)stream);
            Assertions.assertEquals((int)2, (int)bytes.length);
            Assertions.assertArrayEquals((byte[])"45".getBytes(StandardCharsets.UTF_8), (byte[])bytes);
        }
    }

    @Test
    void testSkipFully2() throws IOException {
        byte[] data = "12345".getBytes(StandardCharsets.UTF_8);
        try (ByteArrayInputStream stream = new ByteArrayInputStream(data);){
            Assertions.assertEquals((long)data.length, (long)IOUtil.skipFully((InputStream)stream, (long)(data.length + 1)));
            byte[] bytes = IOUtil.toByteArray((InputStream)stream);
            Assertions.assertEquals((int)0, (int)bytes.length);
        }
    }

    @Test
    void testNormalizePath() {
        if (SystemUtils.IS_OS_UNIX) {
            Assertions.assertEquals((Object)"ab/cd.txt", (Object)IOUtil.normalizePath((String)"ab/ef/../cd.txt"));
            Assertions.assertEquals((Object)"/a.txt", (Object)IOUtil.normalizePath((String)"/x/../../a.txt"));
            Assertions.assertEquals((Object)"/foo", (Object)IOUtil.normalizePath((String)"//../foo"));
            Assertions.assertEquals((Object)"/foo", (Object)IOUtil.normalizePath((String)"/foo//"));
            Assertions.assertEquals((Object)"/foo", (Object)IOUtil.normalizePath((String)"/foo/./"));
            Assertions.assertEquals((Object)"/bar", (Object)IOUtil.normalizePath((String)"/foo/../bar"));
            Assertions.assertEquals((Object)"/bar", (Object)IOUtil.normalizePath((String)"/foo/../bar/"));
            Assertions.assertEquals((Object)"/baz", (Object)IOUtil.normalizePath((String)"/foo/../bar/../baz"));
            Assertions.assertEquals((Object)"/foo/bar", (Object)IOUtil.normalizePath((String)"//foo//./bar"));
            Assertions.assertEquals((Object)"foo", (Object)IOUtil.normalizePath((String)"foo/bar/.."));
            Assertions.assertEquals((Object)"bar", (Object)IOUtil.normalizePath((String)"foo/../bar"));
            Assertions.assertEquals((Object)"/foo/baz", (Object)IOUtil.normalizePath((String)"//foo/bar/../baz"));
            Assertions.assertEquals((Object)"~/bar", (Object)IOUtil.normalizePath((String)"~/foo/../bar/"));
            Assertions.assertEquals((Object)"/", (Object)IOUtil.normalizePath((String)"/../"));
            Assertions.assertEquals((Object)"bar", (Object)IOUtil.normalizePath((String)"~/../bar"));
            Assertions.assertEquals((Object)"bar", (Object)IOUtil.normalizePath((String)"./bar"));
            Assertions.assertNull((Object)IOUtil.normalizePath((String)"../foo"));
            Assertions.assertNull((Object)IOUtil.normalizePath((String)"foo/../../bar"));
            Assertions.assertNull((Object)IOUtil.normalizePath((String)"."));
            Assertions.assertTrue((boolean)IOUtil.equalsNormalizedPaths((String)"foo/../bar", (String)"bar/./"));
        }
        if (SystemUtils.IS_OS_WINDOWS) {
            Assertions.assertEquals((Object)"ab\\cd.txt", (Object)IOUtil.normalizePath((String)"ab\\ef\\..\\cd.txt"));
            Assertions.assertEquals((Object)"\\a.txt", (Object)IOUtil.normalizePath((String)"\\x\\..\\..\\a.txt"));
            Assertions.assertEquals((Object)"\\foo", (Object)IOUtil.normalizePath((String)"\\foo\\\\"));
            Assertions.assertEquals((Object)"\\foo", (Object)IOUtil.normalizePath((String)"\\foo\\.\\"));
            Assertions.assertEquals((Object)"\\bar", (Object)IOUtil.normalizePath((String)"\\foo\\..\\bar"));
            Assertions.assertEquals((Object)"\\bar", (Object)IOUtil.normalizePath((String)"\\foo\\..\\bar\\"));
            Assertions.assertEquals((Object)"\\baz", (Object)IOUtil.normalizePath((String)"\\foo\\..\\bar\\..\\baz"));
            Assertions.assertEquals((Object)"\\\\foo\\bar\\", (Object)IOUtil.normalizePath((String)"\\\\foo\\bar"));
            Assertions.assertEquals((Object)"\\\\foo\\bar\\baz", (Object)IOUtil.normalizePath((String)"\\\\foo\\bar\\..\\baz"));
            Assertions.assertEquals((Object)"foo", (Object)IOUtil.normalizePath((String)"foo\\bar\\.."));
            Assertions.assertEquals((Object)"bar", (Object)IOUtil.normalizePath((String)"foo\\..\\bar"));
            Assertions.assertEquals((Object)"\\foo\\baz", (Object)IOUtil.normalizePath((String)"\\foo\\bar\\..\\baz"));
            Assertions.assertEquals((Object)"\\", (Object)IOUtil.normalizePath((String)"\\..\\"));
            Assertions.assertEquals((Object)"bar", (Object)IOUtil.normalizePath((String)".\\bar"));
            Assertions.assertNull((Object)IOUtil.normalizePath((String)"\\\\..\\foo"));
            Assertions.assertNull((Object)IOUtil.normalizePath((String)"..\\foo"));
            Assertions.assertNull((Object)IOUtil.normalizePath((String)"foo\\..\\..\\bar"));
            Assertions.assertNull((Object)IOUtil.normalizePath((String)"."));
            Assertions.assertNull((Object)IOUtil.normalizePath((String)"\\\\foo\\\\.\\bar"));
            Assertions.assertNull((Object)IOUtil.normalizePath((String)"\\\\foo\\.\\bar"));
            Assertions.assertTrue((boolean)IOUtil.equalsNormalizedPaths((String)"foo\\..\\bar", (String)"bar\\.\\"));
            Assertions.assertEquals((Object)"C:\\bar", (Object)IOUtil.normalizePath((String)"C:\\..\\bar"));
            Assertions.assertEquals((Object)"ab\\cd.txt", (Object)IOUtil.normalizePath((String)"ab\\ef\\..\\cd.txt"));
            Assertions.assertEquals((Object)"C:\\ab\\cd.txt", (Object)IOUtil.normalizePath((String)"C:\\ab\\ef\\..\\.\\cd.txt"));
            Assertions.assertNull((Object)IOUtil.normalizePath((String)"..\\foo"));
            Assertions.assertNull((Object)IOUtil.normalizePath((String)"foo\\..\\..\\bar"));
        }
    }

    @Test
    void testFilenameExtension() {
        Assertions.assertEquals((Object)"txt", (Object)IOUtil.getFilenameExtension((String)"ab/cd.txt"));
        Assertions.assertEquals((Object)"txt", (Object)IOUtil.getFilenameExtension((String)"ab.cd.txt"));
        Assertions.assertEquals((Object)"", (Object)IOUtil.getFilenameExtension((String)"ab/cd"));
        Assertions.assertEquals((Object)"html", (Object)IOUtil.getFilenameExtension((String)"cd.html"));
    }

    @Test
    void testFilenameBase() {
        Assertions.assertEquals((Object)"cd", (Object)IOUtil.getFilenameBase((String)"ab/cd.txt"));
        Assertions.assertEquals((Object)"ab.cd", (Object)IOUtil.getFilenameBase((String)"ab.cd.txt"));
        Assertions.assertEquals((Object)"cd", (Object)IOUtil.getFilenameBase((String)"ab/cd"));
    }

    @Test
    void testBomAwareStream() throws IOException {
        this.assertBomStream("No BOM".getBytes(StandardCharsets.UTF_8), "No BOM", null);
        this.assertBomStream("\ufeffBOM".getBytes(StandardCharsets.UTF_8), "BOM", StandardCharsets.UTF_8.name());
        this.assertBomStream("\ufeffBOM".getBytes(StandardCharsets.UTF_16LE), "BOM", StandardCharsets.UTF_16LE.name());
        this.assertBomStream("\ufeffBOM".getBytes(StandardCharsets.UTF_16BE), "BOM", StandardCharsets.UTF_16BE.name());
    }

    private void assertBomStream(byte[] data, String expectedData, String expectedCharset) throws IOException {
        try (IOUtil.BomAwareInputStream stream = new IOUtil.BomAwareInputStream((InputStream)new ByteArrayInputStream(data));){
            if (expectedCharset != null) {
                Assertions.assertTrue((boolean)stream.hasBom());
                Assertions.assertEquals((Object)expectedCharset, (Object)stream.getBomCharsetName());
                Assertions.assertEquals((Object)expectedData, (Object)new String(IOUtil.toByteArray((InputStream)stream), stream.getBomCharsetName()));
            } else {
                Assertions.assertFalse((boolean)stream.hasBom());
                Assertions.assertNull((Object)stream.getBomCharsetName());
                Assertions.assertEquals((Object)expectedData, (Object)new String(IOUtil.toByteArray((InputStream)stream), StandardCharsets.UTF_8));
            }
        }
    }

    @Test
    void testOutputStreamFromWriter() throws IOException {
        StringWriter writer = new StringWriter();
        try (OutputStream outputStream = IOUtil.fromWriter((Writer)writer, (String)"UTF-8");){
            outputStream.write("abc".getBytes(StandardCharsets.UTF_8));
        }
        Assertions.assertEquals((Object)"abc", (Object)writer.toString());
    }

    @Test
    void testInputStreamFromReader() throws IOException {
        try (InputStream inputStream = IOUtil.fromReader((Reader)new StringReader("abc"));){
            byte[] bytes = IOUtil.toByteArray((InputStream)inputStream);
            Assertions.assertEquals((Object)"abc", (Object)new String(bytes, StandardCharsets.UTF_8));
        }
    }

    @Test
    void testInputStreamFromReader2() throws IOException {
        int size = 16394;
        char[] data = new char[size];
        for (int i = 0; i < size; ++i) {
            data[i] = 65;
        }
        data[8192] = 228;
        try (InputStream inputStream = IOUtil.fromReader((Reader)new StringReader(new String(data)));){
            byte[] bytes = IOUtil.toByteArray((InputStream)inputStream);
            Assertions.assertEquals((Object)new String(data), (Object)new String(bytes, StandardCharsets.UTF_8));
        }
    }

    @Test
    void testCopyStream() throws IOException {
        int size = 16394;
        byte[] data = new byte[size];
        for (int i = 0; i < size; ++i) {
            data[i] = 65;
        }
        try (ByteArrayInputStream stream = new ByteArrayInputStream(data);
             ByteArrayOutputStream out = new ByteArrayOutputStream();){
            IOUtil.copy((InputStream)stream, (OutputStream)out);
            byte[] bytes = out.toByteArray();
            Assertions.assertEquals((int)size, (int)bytes.length);
            Assertions.assertArrayEquals((byte[])data, (byte[])bytes);
        }
    }

    @Test
    void testCopyReader() throws IOException {
        int size = 16394;
        char[] data = new char[size];
        for (int i = 0; i < size; ++i) {
            data[i] = 65;
        }
        try (CharArrayReader reader = new CharArrayReader(data);
             StringWriter writer = new StringWriter();){
            IOUtil.copy((Reader)reader, (Writer)writer);
            char[] chars = writer.toString().toCharArray();
            Assertions.assertEquals((int)size, (int)chars.length);
            Assertions.assertArrayEquals((char[])data, (char[])chars);
        }
    }

    @Test
    void testReadEmptyStream() throws IOException {
        try (ByteArrayInputStream in = new ByteArrayInputStream(new byte[0]);){
            byte[] bytes = IOUtil.toByteArray((InputStream)in);
            Assertions.assertNotNull((Object)bytes);
            Assertions.assertEquals((int)0, (int)bytes.length);
        }
    }

    @Test
    void testCloseQuietly() {
        class Stream
        extends InputStream {
            private boolean closed = false;

            Stream() {
            }

            @Override
            public int read() throws IOException {
                return 0;
            }

            @Override
            public void close() throws IOException {
                this.closed = true;
                throw new IOException("test");
            }

            public boolean isClosed() {
                return this.closed;
            }
        }
        Stream stream = new Stream();
        IOUtil.closeQuietly((Closeable)stream);
        Assertions.assertTrue((boolean)stream.isClosed());
    }

    @Test
    void testReadFileToString() throws IOException {
        String testString = "Test ABC";
        Path tempFile = Files.createTempFile("pmd", ".txt", new FileAttribute[0]);
        Files.write(tempFile, testString.getBytes(Charset.defaultCharset()), new OpenOption[0]);
        Assertions.assertEquals((Object)testString, (Object)IOUtil.readFileToString((File)tempFile.toFile()));
    }

    @Test
    void testReadToString() throws IOException {
        String testString = "testReadToString";
        StringReader reader = new StringReader(testString);
        Assertions.assertEquals((Object)testString, (Object)IOUtil.readToString((Reader)reader));
    }

    @Test
    void testReadStreamToString() throws IOException {
        String testString = "testReadStreamToString";
        ByteArrayInputStream stream = new ByteArrayInputStream(testString.getBytes(StandardCharsets.UTF_8));
        Assertions.assertEquals((Object)testString, (Object)IOUtil.readToString((InputStream)stream, (Charset)StandardCharsets.UTF_8));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testCreateWriterStdout() throws IOException {
        PrintStream originalOut = System.out;
        ByteArrayOutputStream data = new ByteArrayOutputStream();
        PrintStream out = new PrintStream(new FilterOutputStream(data){

            @Override
            public void close() {
                Assertions.fail((String)"Stream must not be closed");
            }
        });
        try {
            System.setOut(out);
            Writer writer = IOUtil.createWriter();
            writer.write("Test");
            writer.close();
            Assertions.assertEquals((Object)"Test", (Object)data.toString());
        }
        finally {
            System.setOut(originalOut);
        }
    }
}

