// THIS FILE HAS BEEN GENERATED BY A PREPROCESSOR.
/* +=======================================================================
 * |
 * |      PlantUML : a free UML diagram generator
 * |
 * +=======================================================================
 *
 * (C) Copyright 2009-2024, Arnaud Roques
 *
 * Project Info:  https://plantuml.com
 *
 * If you like this project or if you find it useful, you can support us at:
 *
 * https://plantuml.com/patreon (only 1$ per month!)
 * https://plantuml.com/liberapay (only 1€ per month!)
 * https://plantuml.com/paypal
 *
 *
 * PlantUML is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License V2.
 *
 * THE ACCOMPANYING PROGRAM IS PROVIDED UNDER THE TERMS OF THIS ECLIPSE PUBLIC
 * LICENSE ("AGREEMENT"). [GNU General Public License V2]
 *
 * ANY USE, REPRODUCTION OR DISTRIBUTION OF THE PROGRAM CONSTITUTES
 * RECIPIENT'S ACCEPTANCE OF THIS AGREEMENT.
 *
 * You may obtain a copy of the License at
 *
 * https://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * PlantUML can occasionally display sponsored or advertising messages. Those
 * messages are usually generated on welcome or error images and never on
 * functional diagrams.
 * See https://plantuml.com/professional if you want to remove them
 *
 * Images (whatever their format : PNG, SVG, EPS...) generated by running PlantUML
 * are owned by the author of their corresponding sources code (that is, their
 * textual description in PlantUML language). Those images are not covered by
 * this GPL v2 license.
 *
 * The generated images can then be used without any reference to the GPL v2 license.
 * It is not even necessary to stipulate that they have been generated with PlantUML,
 * although this will be appreciated by the PlantUML team.
 *
 * There is an exception : if the textual description in PlantUML language is also covered
 * by any license, then the generated images are logically covered
 * by the very same license.
 *
 * This is the IGY distribution (Install GraphViz by Yourself).
 * You have to install GraphViz and to setup the GRAPHVIZ_DOT environment variable
 * (see https://plantuml.com/graphviz-dot )
 *
 * Icons provided by OpenIconic :  https://useiconic.com/open
 * Archimate sprites provided by Archi :  http://www.archimatetool.com
 * Stdlib AWS provided by https://github.com/milo-minderbinder/AWS-PlantUML
 * Stdlib Icons provided https://github.com/tupadr3/plantuml-icon-font-sprites
 * ASCIIMathML (c) Peter Jipsen http://www.chapman.edu/~jipsen
 * ASCIIMathML (c) David Lippman http://www.pierce.ctc.edu/dlippman
 * CafeUndZopfli ported by Eugene Klyuchnikov https://github.com/eustas/CafeUndZopfli
 * Brotli (c) by the Brotli Authors https://github.com/google/brotli
 * Themes (c) by Brett Schwarz https://github.com/bschwarz/puml-themes
 * Twemoji (c) by Twitter at https://twemoji.twitter.com/
 *
 */

package zext.plantuml.com.google.zxing.datamatrix.decoder;

import zext.plantuml.com.google.zxing.FormatException;
import zext.plantuml.com.google.zxing.common.BitMatrix;

/**
 * @author bbrown@google.com (Brian Brown)
 */
final class BitMatrixParser {

  private final BitMatrix mappingBitMatrix;
  private final BitMatrix readMappingMatrix;
  private final Version version;

  /**
   * @param bitMatrix {@link BitMatrix} to parse
   * @throws FormatException if dimension is < 10 or > 144 or not 0 mod 2
   */
  BitMatrixParser(BitMatrix bitMatrix) throws FormatException {
    int dimension = bitMatrix.getHeight();
    if (dimension < 10 || dimension > 144 || (dimension & 0x01) != 0) {
      throw FormatException.getFormatInstance();
    }
    
    version = readVersion(bitMatrix);
    this.mappingBitMatrix = extractDataRegion(bitMatrix);
    // TODO(bbrown): Make this work for rectangular symbols
    this.readMappingMatrix = new BitMatrix(this.mappingBitMatrix.getHeight());
  }

  /**
   * <p>Creates the version object based on the dimension of the original bit matrix from 
   * the datamatrix code.</p>
   *
   * <p>See ISO 16022:2006 Table 7 - ECC 200 symbol attributes</p>
   * 
   * @param bitMatrix Original {@link BitMatrix} including alignment patterns
   * @return {@link Version} encapsulating the Data Matrix Code's "version"
   * @throws FormatException if the dimensions of the mapping matrix are not valid
   * Data Matrix dimensions.
   */
  Version readVersion(BitMatrix bitMatrix) throws FormatException {

    if (version != null) {
      return version;
    }

    // TODO(bbrown): make this work for rectangular dimensions as well.
    int numRows = bitMatrix.getHeight();
    int numColumns = numRows;
    
    return Version.getVersionForDimensions(numRows, numColumns);
  }

  /**
   * <p>Reads the bits in the {@link BitMatrix} representing the mapping matrix (No alignment patterns)
   * in the correct order in order to reconstitute the codewords bytes contained within the
   * Data Matrix Code.</p>
   *
   * @return bytes encoded within the Data Matrix Code
   * @throws FormatException if the exact number of bytes expected is not read
   */
  byte[] readCodewords() throws FormatException {

    byte[] result = new byte[version.getTotalCodewords()];
    int resultOffset = 0;
    
    int row = 4;
    int column = 0;
    // TODO(bbrown): Data Matrix can be rectangular, assuming square for now
    int numRows = mappingBitMatrix.getHeight();
    int numColumns = numRows;
    
    boolean corner1Read = false;
    boolean corner2Read = false;
    boolean corner3Read = false;
    boolean corner4Read = false;
    
    // Read all of the codewords
    do {
      // Check the four corner cases
      if ((row == numRows) && (column == 0) && !corner1Read) {
        result[resultOffset++] = (byte) readCorner1(numRows, numColumns);
        row -= 2;
        column +=2;
        corner1Read = true;
      } else if ((row == numRows-2) && (column == 0) && ((numColumns & 0x03) != 0) && !corner2Read) {
        result[resultOffset++] = (byte) readCorner2(numRows, numColumns);
        row -= 2;
        column +=2;
        corner2Read = true;
      } else if ((row == numRows+4) && (column == 2) && ((numColumns & 0x07) == 0) && !corner3Read) {
        result[resultOffset++] = (byte) readCorner3(numRows, numColumns);
        row -= 2;
        column +=2;
        corner3Read = true;
      } else if ((row == numRows-2) && (column == 0) && ((numColumns & 0x07) == 4) && !corner4Read) {
        result[resultOffset++] = (byte) readCorner4(numRows, numColumns);
        row -= 2;
        column +=2;
        corner4Read = true;
      } else {
        // Sweep upward diagonally to the right
        do {
          if ((row < numRows) && (column >= 0) && !readMappingMatrix.get(column, row)) {
            result[resultOffset++] = (byte) readUtah(row, column, numRows, numColumns);
          }
          row -= 2;
          column +=2;
        } while ((row >= 0) && (column < numColumns));
        row += 1;
        column +=3;
        
        // Sweep downward diagonally to the left
        do {
          if ((row >= 0) && (column < numColumns) && !readMappingMatrix.get(column, row)) {
             result[resultOffset++] = (byte) readUtah(row, column, numRows, numColumns);
          }
          row += 2;
          column -=2;
        } while ((row < numRows) && (column >= 0));
        row += 3;
        column +=1;
      }
    } while ((row < numRows) || (column < numColumns));

    if (resultOffset != version.getTotalCodewords()) {
      throw FormatException.getFormatInstance();
    }
    return result;
  }
  
  /**
   * <p>Reads a bit of the mapping matrix accounting for boundary wrapping.</p>
   * 
   * @param row Row to read in the mapping matrix
   * @param column Column to read in the mapping matrix
   * @param numRows Number of rows in the mapping matrix
   * @param numColumns Number of columns in the mapping matrix
   * @return value of the given bit in the mapping matrix
   */
  boolean readModule(int row, int column, int numRows, int numColumns) {
    // Adjust the row and column indices based on boundary wrapping
    if (row < 0) {
      row += numRows;
      column += 4 - ((numRows + 4) & 0x07);
    }
    if (column < 0) {
      column += numColumns;
      row += 4 - ((numColumns + 4) & 0x07);
    }
    readMappingMatrix.set(column, row);
    return mappingBitMatrix.get(column, row);
  }
  
  /**
   * <p>Reads the 8 bits of the standard Utah-shaped pattern.</p>
   * 
   * <p>See ISO 16022:2006, 5.8.1 Figure 6</p>
   * 
   * @param row Current row in the mapping matrix, anchored at the 8th bit (LSB) of the pattern
   * @param column Current column in the mapping matrix, anchored at the 8th bit (LSB) of the pattern
   * @param numRows Number of rows in the mapping matrix
   * @param numColumns Number of columns in the mapping matrix
   * @return byte from the utah shape
   */
  int readUtah(int row, int column, int numRows, int numColumns) {
    int currentByte = 0;
    if (readModule(row - 2, column - 2, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(row - 2, column - 1, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(row - 1, column - 2, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(row - 1, column - 1, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(row - 1, column, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(row, column - 2, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(row, column - 1, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(row, column, numRows, numColumns)) {
      currentByte |= 1;
    }
    return currentByte;
  }
  
  /**
   * <p>Reads the 8 bits of the special corner condition 1.</p>
   * 
   * <p>See ISO 16022:2006, Figure F.3</p>
   * 
   * @param numRows Number of rows in the mapping matrix
   * @param numColumns Number of columns in the mapping matrix
   * @return byte from the Corner condition 1
   */
  int readCorner1(int numRows, int numColumns) {
    int currentByte = 0;
    if (readModule(numRows - 1, 0, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(numRows - 1, 1, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(numRows - 1, 2, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(0, numColumns - 2, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(0, numColumns - 1, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(1, numColumns - 1, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(2, numColumns - 1, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(3, numColumns - 1, numRows, numColumns)) {
      currentByte |= 1;
    }
    return currentByte;
  }
  
  /**
   * <p>Reads the 8 bits of the special corner condition 2.</p>
   * 
   * <p>See ISO 16022:2006, Figure F.4</p>
   * 
   * @param numRows Number of rows in the mapping matrix
   * @param numColumns Number of columns in the mapping matrix
   * @return byte from the Corner condition 2
   */
  int readCorner2(int numRows, int numColumns) {
    int currentByte = 0;
    if (readModule(numRows - 3, 0, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(numRows - 2, 0, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(numRows - 1, 0, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(0, numColumns - 4, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(0, numColumns - 3, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(0, numColumns - 2, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(0, numColumns - 1, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(1, numColumns - 1, numRows, numColumns)) {
      currentByte |= 1;
    }
    return currentByte;
  }
  
  /**
   * <p>Reads the 8 bits of the special corner condition 3.</p>
   * 
   * <p>See ISO 16022:2006, Figure F.5</p>
   * 
   * @param numRows Number of rows in the mapping matrix
   * @param numColumns Number of columns in the mapping matrix
   * @return byte from the Corner condition 3
   */
  int readCorner3(int numRows, int numColumns) {
    int currentByte = 0;
    if (readModule(numRows - 1, 0, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(numRows - 1, numColumns - 1, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(0, numColumns - 3, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(0, numColumns - 2, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(0, numColumns - 1, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(1, numColumns - 3, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(1, numColumns - 2, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(1, numColumns - 1, numRows, numColumns)) {
      currentByte |= 1;
    }
    return currentByte;
  }
  
  /**
   * <p>Reads the 8 bits of the special corner condition 4.</p>
   * 
   * <p>See ISO 16022:2006, Figure F.6</p>
   * 
   * @param numRows Number of rows in the mapping matrix
   * @param numColumns Number of columns in the mapping matrix
   * @return byte from the Corner condition 4
   */
  int readCorner4(int numRows, int numColumns) {
    int currentByte = 0;
    if (readModule(numRows - 3, 0, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(numRows - 2, 0, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(numRows - 1, 0, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(0, numColumns - 2, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(0, numColumns - 1, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(1, numColumns - 1, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(2, numColumns - 1, numRows, numColumns)) {
      currentByte |= 1;
    }
    currentByte <<= 1;
    if (readModule(3, numColumns - 1, numRows, numColumns)) {
      currentByte |= 1;
    }
    return currentByte;
  }
  
  /**
   * <p>Extracts the data region from a {@link BitMatrix} that contains
   * alignment patterns.</p>
   * 
   * @param bitMatrix Original {@link BitMatrix} with alignment patterns
   * @return BitMatrix that has the alignment patterns removed
   */
  BitMatrix extractDataRegion(BitMatrix bitMatrix) {
    int symbolSizeRows = version.getSymbolSizeRows();
    int symbolSizeColumns = version.getSymbolSizeColumns();
    
    // TODO(bbrown): Make this work with rectangular codes
    if (bitMatrix.getHeight() != symbolSizeRows) {
      throw new IllegalArgumentException("Dimension of bitMarix must match the version size");
    }
    
    int dataRegionSizeRows = version.getDataRegionSizeRows();
    int dataRegionSizeColumns = version.getDataRegionSizeColumns();
    
    int numDataRegionsRow = symbolSizeRows / dataRegionSizeRows;
    int numDataRegionsColumn = symbolSizeColumns / dataRegionSizeColumns;
    
    int sizeDataRegionRow = numDataRegionsRow * dataRegionSizeRows;
    //int sizeDataRegionColumn = numDataRegionsColumn * dataRegionSizeColumns;
    
    // TODO(bbrown): Make this work with rectangular codes
    BitMatrix bitMatrixWithoutAlignment = new BitMatrix(sizeDataRegionRow);
    for (int dataRegionRow = 0; dataRegionRow < numDataRegionsRow; ++dataRegionRow) {
      int dataRegionRowOffset = dataRegionRow * dataRegionSizeRows;
      for (int dataRegionColumn = 0; dataRegionColumn < numDataRegionsColumn; ++dataRegionColumn) {
        int dataRegionColumnOffset = dataRegionColumn * dataRegionSizeColumns;
        for (int i = 0; i < dataRegionSizeRows; ++i) {
          int readRowOffset = dataRegionRow * (dataRegionSizeRows + 2) + 1 + i;
          int writeRowOffset = dataRegionRowOffset + i;
          for (int j = 0; j < dataRegionSizeColumns; ++j) {
            int readColumnOffset = dataRegionColumn * (dataRegionSizeColumns + 2) + 1 + j;
            if (bitMatrix.get(readColumnOffset, readRowOffset)) {
              int writeColumnOffset = dataRegionColumnOffset + j;
              bitMatrixWithoutAlignment.set(writeColumnOffset, writeRowOffset);
            }
          }
        }
      }
    }
    return bitMatrixWithoutAlignment;
  }

}
