/**
 * MCS Media Computer Software
 * Copyright 2012 by Wilfried Klaas
 * Project: MCSUtils
 * File: CallbackList.java
 * EMail: W.Klaas@gmx.de
 * Created: 25.03.2012 Willie
 */

package de.mcs.utils.event;

import java.lang.ref.WeakReference;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

/**
 * @author Willie
 *
 */
public class CallbackList<E> extends AbstractList<E> implements List<E> {
  public enum Operation {
    add, addall, remove, clear
  };

  public interface ListCallback {
    void handle(Operation operation);
  }

  private List<WeakReference<ListCallback>> callbackList = new ArrayList<WeakReference<ListCallback>>();

  private List<E> list = null;

  public CallbackList() {
    this.list = new ArrayList<E>();
  }

  public CallbackList(List<E> list) {
    this.list = list;
  }

  public void addCallback(final ListCallback listCallback) {
    for (Iterator<WeakReference<ListCallback>> iter = callbackList.iterator(); iter.hasNext();) {
      WeakReference<ListCallback> listener = iter.next();
      if (listener.isEnqueued()) {
        iter.remove();
      } else {
        if (listCallback.equals(listener.get())) {
          return;
        }
      }
    }
    WeakReference<ListCallback> weakEventListener = new WeakReference<ListCallback>(listCallback);
    callbackList.add(weakEventListener);
  }

  public void removeCallback(final ListCallback listCallback) {
    for (Iterator<WeakReference<ListCallback>> iter = callbackList.iterator(); iter.hasNext();) {
      WeakReference<ListCallback> listener = iter.next();
      if (listener.isEnqueued()) {
        iter.remove();
      } else {
        if (listCallback.equals(listener.get())) {
          iter.remove();
        }
      }
    }
  }

  @Override
  public E get(int index) {
    return list.get(index);
  }

  @Override
  public int size() {
    return list.size();
  }

  /* (non-Javadoc)
   * @see java.util.AbstractList#add(int, java.lang.Object)
   */
  @Override
  public void add(int index, E element) {
    list.add(index, element);
    fireCallback(Operation.add);
  }

  public void fireCallback(Operation operation) {
    for (Iterator<WeakReference<ListCallback>> iter = callbackList.iterator(); iter.hasNext();) {
      WeakReference<ListCallback> weakListener = iter.next();
      if (weakListener.isEnqueued()) {
        iter.remove();
      } else if (weakListener.get() == null) {
        iter.remove();
      } else {
        weakListener.get().handle(operation);
      }
    }
  }

  /* (non-Javadoc)
   * @see java.util.AbstractList#remove(int)
   */
  @Override
  public E remove(int index) {
    E remove = list.remove(index);
    fireCallback(Operation.remove);
    return remove;
  }

  /* (non-Javadoc)
   * @see java.util.AbstractList#clear()
   */
  @Override
  public void clear() {
    list.clear();
    fireCallback(Operation.clear);
  }

  /* (non-Javadoc)
   * @see java.util.AbstractList#addAll(int, java.util.Collection)
   */
  @Override
  public boolean addAll(int index, Collection<? extends E> c) {
    boolean addAll = list.addAll(index, c);
    fireCallback(Operation.addall);
    return addAll;
  }

  /* (non-Javadoc)
   * @see java.util.AbstractCollection#addAll(java.util.Collection)
   */
  @Override
  public boolean addAll(Collection<? extends E> c) {
    boolean addAll = list.addAll(c);
    fireCallback(Operation.addall);
    return addAll;
  }

}
