/*
 * MCS Media Computer Software Copyright (c) 2005 by MCS
 * -------------------------------------- Created on 16.01.2004 by w.klaas
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package de.mcs.utils.codecs;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * BASE64 encoder implementation. This object takes as parameter an input stream
 * and an output stream. It encodes the input stream, using the BASE64 ENCODING
 * rules, as defined in <a href="http://ds.internic.net/rfc/rfc1521.txt">MIME
 * specification </a> and emit the resulting data to the output stream. see
 * org.w3c.tools.codec.Base64Decoder This class is a modified version based on
 * code obtained from the w3 consortium website, which is subject to their
 * generic copyright notice:
 * <dl>
 * <dd><a href="http://www.w3.org/Consortium/Legal/">Copyright </a>
 * [$date-of-software] <a HREF="http://www.w3.org/">World Wide Web Consortium
 * </a>, ( <a HREF="http://www.lcs.mit.edu/">Massachusetts Institute of
 * Technology </a>, <a HREF="http://www.inria.fr/">Institut National de
 * Recherche en Informatique et en Automatique </a>, <a
 * HREF="http://www.keio.ac.jp/">Keio University </a>). All Rights Reserved.
 * This program is distributed under the <a
 * HREF="http://www.w3.org/Consortium/Legal/copyright-software-19980720.html">W3C's
 * Software Intellectual Property License </a>. This program is distributed in
 * the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * W3C License <a
 * href="http://www.w3.org/Consortium/Legal/">http://www.w3.org/Consortium/Legal/
 * </a> for more details.</dd>
 * </dl>
 * 
 * @version $Revision: 1.1 $
 * @deprecated please use the base64 class.
 */

public final class Base64Encoder {
  /** internal buffer size for ENCODING. */
  private static final int BUFFER_SIZE = 1024;

  /** encoding byte array. */
  private static final byte[] ENCODING = { (byte) 'A', (byte) 'B', (byte) 'C', (byte) 'D', (byte) 'E', (byte) 'F',
      (byte) 'G', (byte) 'H', // 0-7
      (byte) 'I', (byte) 'J', (byte) 'K', (byte) 'L', (byte) 'M', (byte) 'N', (byte) 'O', (byte) 'P', // 8-15
      (byte) 'Q', (byte) 'R', (byte) 'S', (byte) 'T', (byte) 'U', (byte) 'V', (byte) 'W', (byte) 'X', // 16-23
      (byte) 'Y', (byte) 'Z', (byte) 'a', (byte) 'b', (byte) 'c', (byte) 'd', (byte) 'e', (byte) 'f', // 24-31
      (byte) 'g', (byte) 'h', (byte) 'i', (byte) 'j', (byte) 'k', (byte) 'l', (byte) 'm', (byte) 'n', // 32-39
      (byte) 'o', (byte) 'p', (byte) 'q', (byte) 'r', (byte) 's', (byte) 't', (byte) 'u', (byte) 'v', // 40-47
      (byte) 'w', (byte) 'x', (byte) 'y', (byte) 'z', (byte) '0', (byte) '1', (byte) '2', (byte) '3', // 48-55
      (byte) '4', (byte) '5', (byte) '6', (byte) '7', (byte) '8', (byte) '9', (byte) '+', (byte) '/', // 56-63
      (byte) '=' // 64
  };

  /** to prevent instancing. */
  private Base64Encoder() {
  }

  /**
   * Encodes data from supplied input to output.
   * 
   * @param in
   *            The input stream to be encoded.
   * @param out
   *            The output stream to write encoded data to.
   * @throws IOException
   *             if something goes wrong
   */
  public static void encode(final InputStream in, final OutputStream out) throws IOException {
    process(in, out);
  }

  /**
   * Encodes from the supplied byte array and write the encoded data to the
   * OutputStream <i>out </i>.
   * 
   * @param input
   *            The byte array input to be encoded.
   * @param out
   *            The output stream to write encoded data to.
   * @throws IOException
   *             if something goes wrong
   */
  public static void encode(final byte[] input, final OutputStream out) throws IOException {
    ByteArrayInputStream in = new ByteArrayInputStream(input);
    process(in, out);
  }

  /**
   * Encode the given string and return the encoded version as a string.
   * 
   * @param input
   *            The string input to be encoded. It is assumed the string input
   *            uses characters from the ISO 8859-1 code page.
   * @return A String, representing the encoded content of the input String.
   *         The returned string uses charactes from 8859-1 code page.
   * @throws IOException
   *             if something goes wrong
   */
  public static String encode(final String input) throws IOException {
    byte[] bytes;
    bytes = input.getBytes("ISO-8859-1");
    return encode(bytes);
  }

  /**
   * Encode the given byte array and return the encoded version as a string.
   * 
   * @param bytes
   *            The byte array to be encoded.
   * @return A String, representing the encoded content of the input bytes.
   *         The returned string uses charactes from 8859-1 code page.
   * @throws IOException
   *             if something goes wrong
   */
  public static String encode(final byte[] bytes) throws IOException {
    ByteArrayInputStream in = new ByteArrayInputStream(bytes);
    ByteArrayOutputStream out = new ByteArrayOutputStream();
    process(in, out);
    return out.toString("ISO-8859-1");
  }

  /**
   * Run with one argument, prints the encoded version of it. With two, the
   * second is assumed to be the name of a MessageDigest to be applied to the
   * string before ENCODING (useful for generating password hashes).
   * <p>
   * Alternatively, use the openssl utility, for example:
   * <p>
   * echo -n "password" | openssl dgst -sha1 -binary | openssl base64
   * 
   * @param args
   *            String[]
   * @throws Exception
   *             if something goes wrong
   */
  public static void main(final String[] args) throws Exception {
    if (args.length == 1) {
      System.out.println("[" + Base64Encoder.encode(args[0]) + "]");
      // joe:eoj -> am9lOmVvag==
      // 12345678:87654321 -> MTIzNDU2Nzg6ODc2NTQzMjE=
    } else if (args.length == 2) {
      byte[] hash = java.security.MessageDigest.getInstance(args[1]).digest(args[0].getBytes());
      System.out.println("[" + Base64Encoder.encode(hash) + "]");
    } else {
      System.out.println("Usage: Base64Encoder <string> <optional hash algorithm>");
    }
  }

  // Private ----------------------------------------------------------------
  /**
   * getting the first integer.
   * 
   * @param buf
   *            byte array
   * @param off
   *            offset
   * @return int
   */
  private static int get1(final byte[] buf, final int off) {
    return (buf[off] & 0xfc) >> 2;
  }

  /**
   * getting the second integer.
   * 
   * @param buf
   *            byte array
   * @param off
   *            offset
   * @return int
   */
  private static int get2(final byte[] buf, final int off) {
    return ((buf[off] & 0x3) << 4) | ((buf[off + 1] & 0xf0) >>> 4);
  }

  /**
   * getting the third integer.
   * 
   * @param buf
   *            byte array
   * @param off
   *            offset
   * @return int
   */
  private static int get3(final byte[] buf, final int off) {
    return ((buf[off + 1] & 0x0f) << 2) | ((buf[off + 2] & 0xc0) >>> 6);
  }

  /**
   * getting the forth integer.
   * 
   * @param buf
   *            byte array
   * @param off
   *            offset
   * @return int
   */
  private static int get4(final byte[] buf, final int off) {
    return buf[off + 2] & 0x3f;
  }

  /**
   * Process the data: encode the input stream to the output stream. This
   * method runs through the input stream, ENCODING it to the output stream.
   * 
   * @param in
   *            input stream
   * @param out
   *            output stream
   * @exception IOException
   *                If we weren't able to access the input stream or the
   *                output stream.
   */
  private static void process(final InputStream in, final OutputStream out) throws IOException {
    byte[] buffer = new byte[BUFFER_SIZE];
    int got = -1;
    int off = 0;
    int count = 0;
    while ((got = in.read(buffer, off, BUFFER_SIZE - off)) > 0) {
      if (got >= 3) {
        got += off;
        off = 0;
        while (off + 3 <= got) {
          int c1 = get1(buffer, off);
          int c2 = get2(buffer, off);
          int c3 = get3(buffer, off);
          int c4 = get4(buffer, off);
          switch (count) {
          case 73:
            out.write(ENCODING[c1]);
            out.write(ENCODING[c2]);
            out.write(ENCODING[c3]);
            out.write('\n');
            out.write(ENCODING[c4]);
            count = 1;
            break;
          case 74:
            out.write(ENCODING[c1]);
            out.write(ENCODING[c2]);
            out.write('\n');
            out.write(ENCODING[c3]);
            out.write(ENCODING[c4]);
            count = 2;
            break;
          case 75:
            out.write(ENCODING[c1]);
            out.write('\n');
            out.write(ENCODING[c2]);
            out.write(ENCODING[c3]);
            out.write(ENCODING[c4]);
            count = 3;
            break;
          case 76:
            out.write('\n');
            out.write(ENCODING[c1]);
            out.write(ENCODING[c2]);
            out.write(ENCODING[c3]);
            out.write(ENCODING[c4]);
            count = 4;
            break;
          default:
            out.write(ENCODING[c1]);
            out.write(ENCODING[c2]);
            out.write(ENCODING[c3]);
            out.write(ENCODING[c4]);
            count += 4;
            break;
          }
          off += 3;
        }
        // Copy remaining bytes to beginning of buffer:
        for (int i = 0; i < 3; i++) {
          if (i < got - off) {
            buffer[i] = buffer[off + i];
          } else {
            buffer[i] = ((byte) 0);
          }
        }
        off = got - off;
      } else {
        // Total read amount is less then 3 bytes:
        off += got;
      }
    }
    // Manage the last bytes, from 0 to off:
    switch (off) {
    case 1:
      out.write(ENCODING[get1(buffer, 0)]);
      out.write(ENCODING[get2(buffer, 0)]);
      out.write('=');
      out.write('=');
      break;
    case 2:
      out.write(ENCODING[get1(buffer, 0)]);
      out.write(ENCODING[get2(buffer, 0)]);
      out.write(ENCODING[get3(buffer, 0)]);
      out.write('=');
    default:
    }
    return;
  }
}
