package de.mcs.utils.caches;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Timer;
import java.util.TimerTask;
import java.util.UUID;

/**
 * Diese Klasse dient dem automatischen Cachen von beliebigen Objekten.
 * 
 * @author w.klaas
 */
public class ObjectCache<E> {

  public interface ObjectListener<E> {
    void onDelete(E item);
  }

  // private List<CacheItem> objectList = null;

  /** internal map for the objects. */
  private Map<String, CacheItem<E>> objectMap = null;

  /** internal map for the blobs. */
  private Map<String, String> objectMapExternal = null;

  private int maxCount;

  private Timer timer = null;

  private ObjectListener<E> objectListener = null;

  private Comparator<? super CacheItem<E>> comperator = new Comparator<CacheItem<E>>() {

    @Override
    public int compare(CacheItem<E> o1, CacheItem<E> o2) {
      try {
        int cmp = 0;
        if ((o1 == null) && (o2 == null)) {
          cmp = 0;
        } else if (o1 == null) {
          cmp = -1;
        } else if (o2 == null) {
          cmp = 1;
        } else {
          Long d1 = o1.getAccess();
          Long d2 = o2.getAccess();
          cmp = d1.compareTo(d2);
        }
        return cmp;
      } catch (Exception e) {
        e.printStackTrace();
      }
      return 0;
    }

  };

  /**
   * Diese Klasse dient der Aufnahme eines einzelnen Blobs.
   * 
   * @author w.klaas
   */
  class CacheItem<E> implements Serializable {
    /**
           * 
           */
    private static final long serialVersionUID = -871711242126446479L;

    /** ID of this entry. */
    private String sID = UUID.randomUUID().toString();

    /** last access to this item. */
    private Date dAccess = new Date();

    private String externalKey;

    private E object;

    public CacheItem() {
    }

    public long getAccess() {
      synchronized (this) {
        if (dAccess != null) {
          return dAccess.getTime();
        }
        return 0;
      }
    }

    /**
     * Eine neue Datei registrieren, inkl. MD5 Hash bestimmen
     * 
     * @param filename
     *          Filename of the file
     * @param refid
     *          this is the registered id of the document
     * @param persistent
     *          Should this blob be persistent?
     */
    public CacheItem(final E object, final String externalKey) {
      this();
      this.object = object;
      this.externalKey = externalKey;
    }

    /**
     * Aufräumen
     */
    public void freeResources() {
      // nothing to do here;
    }

    /**
     * Vor dem abräumen, aufräumen.
     */
    protected void finalize() {
      freeResources();
    }

    public E getObject() {
      return object;
    }

    public String getExternalKey() {
      return externalKey;
    }

    public String toString() {
      StringBuffer buffer = new StringBuffer();
      buffer.append("object:");
      buffer.append(object.toString());
      buffer.append(",ID:");
      buffer.append(sID);
      return buffer.toString();
    }

    public void setAccess() {
      synchronized (this) {
        dAccess = new Date();
      }
    }
  }

  public ObjectCache() {
    this(-1);
  }

  /**
   * Constructs a new object cache.
   * 
   * @param maxCount
   *          maximal count of objects in this cache.
   */
  public ObjectCache(final int maxCount) {
    this.maxCount = maxCount;
    int size = maxCount > 0 ? maxCount : 1000;
    if (maxCount >= 0) {
      objectMap = new HashMap<String, CacheItem<E>>(size, 100);
      objectMapExternal = new HashMap<String, String>(size, 100);
      // objectList = new ArrayList<CacheItem>(size);
    } else {
      objectMap = new HashMap<String, CacheItem<E>>();
      objectMapExternal = new HashMap<String, String>();
    }
    initCache();
  }

  private void initCache() {
    // nothing to do here
  }

  /**
   * Registering an object in the cache.
   * 
   * @param object
   *          object to add
   * @return String the id of the object in the cache.
   */
  public final String addObject(final E object) {
    try {
      return addObject(null, object);
    } catch (KeyAlreadyExistsException e) {
      // can never occure
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Registering an object in the cache.
   * 
   * @param externalKey
   *          the external key to use with this object
   * @param object
   *          object to add
   * @return String the id of the object in the cache.
   * @throws KeyAlreadyExistsException
   */
  public final String addObject(final String externalKey, final E object) throws KeyAlreadyExistsException {
    if ((externalKey != null) && (hasExternalKey(externalKey))) {
      throw new KeyAlreadyExistsException("the external key is already in use.");
    }
    CacheItem<E> cacheItem = new CacheItem<E>(object, externalKey);
    if (!hasObject(cacheItem.sID)) {
      checkSize();
      synchronized (objectMap) {
        objectMap.put(cacheItem.sID, cacheItem);
        if (externalKey != null) {
          objectMapExternal.put(externalKey, cacheItem.sID);
        }
      }
    }
    return cacheItem.sID;
  }

  private void checkSize() {
    if (maxCount > 0) {
      while (objectMap.size() >= (maxCount)) {
        removeOldest();
      }
    }
  }

  private void removeOldest() {
    List<CacheItem<E>> list = null;
    synchronized (objectMap) {
      list = new ArrayList<CacheItem<E>>(objectMap.values());
      synchronized (comperator) {
        Collections.sort(list, comperator);
      }
      CacheItem<E> item = list.get(0);
      removeObject(item.sID);
    }
  }

  /**
   * Prüfen, ob ein bestimmtes Object auch zur Verfügung steht.
   * 
   * @param objectID
   *          die ID des Objektes
   * @return boolean
   */
  public final boolean hasObject(final String objectID) {
    boolean hasId = false;
    hasId = objectMap.containsKey(objectID);
    return hasId;
  }

  /**
   * Prüfen, ob ein bestimmtes Objekt mit exteranl Key auch zur Verfügung steht.
   * 
   * @param externalKey
   *          die ID des Blobs
   * @return boolean
   */
  public final boolean hasExternalKey(final String externalKey) {
    boolean hasId = false;
    synchronized (objectMapExternal) {
      hasId = objectMapExternal.containsKey(externalKey);
    }
    return hasId;
  }

  /**
   * Objektid eines bestimmtes Objektes mit external Key holen.
   * 
   * @param externalKey
   *          die ID des Objektes
   * @return string, id des Objektes, oder <code>null</code> wenn nicht
   *         vorhanden
   */
  public final String getIDfromExternalKey(final String externalKey) {
    String objectId = null;
    synchronized (objectMapExternal) {
      if (objectMapExternal.containsKey(externalKey)) {
        objectId = objectMapExternal.get(externalKey);
      }
    }
    return objectId;
  }

  private E getObjectInternal(String objectID) {
    E object = null;
    synchronized (objectMap) {
      if (hasObject(objectID)) {
        CacheItem<E> cacheItem = objectMap.get(objectID);
        object = cacheItem.object;
      }
    }
    return object;
  }

  /**
   * Das Objekt holen, gleichzeitig wird das Objekt für die nächste Zeitspanne
   * freigeschaltet.
   * 
   * @param objectID
   *          die ID des Objektes
   */
  public final E getObject(final String objectID) {
    E object = null;
    synchronized (objectMap) {
      if (hasObject(objectID)) {
        CacheItem<E> cacheItem = objectMap.get(objectID);
        cacheItem.setAccess();
        object = cacheItem.object;
      }
    }
    return object;
  }

  /**
   * Das älteste Objekt holen, gleichzeitig wird das Objekt für die nächste
   * Zeitspanne freigeschaltet.
   */
  public final E getOldestObject() {
    E object = null;
    synchronized (objectMap) {
      List<CacheItem<E>> list = new ArrayList<CacheItem<E>>(objectMap.values());
      synchronized (comperator) {
        Collections.sort(list, comperator);
      }
      if (list.size() == 0) {
        return null;
      }

      CacheItem<E> cacheItem = list.get(0);
      cacheItem.setAccess();
      object = cacheItem.object;

      return (E) object;
    }
  }

  /**
   * Das Objekt holen, gleichzeitig wird das Objekt für die nächste Zeitspanne
   * freigeschaltet.
   * 
   * @param externalKey
   *          die ID des Objektes
   */
  public final E getObjectFromExternalKey(final String externalKey) {
    E object = null;
    String objectID = getIDfromExternalKey(externalKey);
    if (objectID != null) {
      return getObject(objectID);
    }
    return object;
  }

  public void touchExternalKey(String externalKey) {
    String objectID = getIDfromExternalKey(externalKey);
    if (objectID != null) {
      getObject(objectID);
    }
  }

  public void touch(String objectID) {
    getObject(objectID);
  }

  /**
   * Das Objekt löschen.
   * 
   * @param objectID
   *          die ID des Objektes
   */
  public final E removeObject(final String objectID) {
    E object = null;
    synchronized (objectMap) {
      if (hasObject(objectID)) {
        CacheItem<E> cacheItem = objectMap.remove(objectID);
        if (cacheItem != null) {
          if (objectMapExternal.containsKey(cacheItem.getExternalKey())) {
            objectMapExternal.remove(cacheItem.getExternalKey());
          }
          object = cacheItem.object;
          if (objectListener != null) {
            objectListener.onDelete(cacheItem.getObject());
          }
        }
      }
    }
    return object;
  }

  /**
   * Das Objekt löschen.
   * 
   * @param externalKey
   *          die ID des Objektes
   */
  public final E removeObjectFromExternalKey(final String externalKey) {
    E object = null;
    String objectID = getIDfromExternalKey(externalKey);
    if (objectID != null) {
      return removeObject(objectID);
    }
    return object;
  }

  /**
   * Hier werden jetzt alle Objekte die älter sind als die Angabe,
   * deregistriert. Bei der nächsten GC werden diese dann gelöscht
   * 
   * @param sec
   *          Anzahl der Sekunden die die Blobs alt sein d�rfen
   */
  public final void freeObjects(final long sec) {
    Date myEqualDate = new Date(new Date().getTime() - (sec * 1000));
    CacheItem<E> cacheItem;
    ArrayList<String> vDelKeys = new ArrayList<String>(objectMap.size());
    String sKey = "";
    synchronized (objectMap) {
      Iterator<String> it = objectMap.keySet().iterator();
      while (it.hasNext()) {
        sKey = it.next();
        cacheItem = objectMap.get(sKey);
        if (myEqualDate.getTime() > (cacheItem.getAccess())) {
          cacheItem.freeResources();
          vDelKeys.add(sKey);
        }
      }
      it = null;
      for (int n = 0; n < vDelKeys.size(); n++) {
        CacheItem<E> item = objectMap.remove(vDelKeys.get(n));
        if (objectListener != null) {
          objectListener.onDelete(item.getObject());
        }
      }
      vDelKeys = null;
    }
    synchronized (objectMapExternal) {
      Iterator<Entry<String, String>> it = objectMapExternal.entrySet().iterator();
      while (it.hasNext()) {
        Entry<String, String> entry = it.next();
        if (!hasObject(entry.getValue())) {
          it.remove();
        }
      }
    }
  }

  /**
   * Holt die aktuelle Anzahl der im Cache liegenden Objekte.
   * 
   * @return int Anzahl der Objekte im Cache
   */
  public final int getCacheCount() {
    int size;
    synchronized (objectMap) {
      size = objectMap.size();
    }
    return size;
  }

  /**
   * Cache leeren
   */
  public final void freeResources() {
    freeObjects(0);
  }

  /**
   * Besort eine Liste aller Objekt ID's.
   * 
   * @return String[]
   */
  public final String[] getObjectIDs() {
    String[] keys;
    synchronized (objectMap) {
      keys = new String[objectMap.size()];
      Iterator<String> it = objectMap.keySet().iterator();
      int n = 0;
      while (it.hasNext()) {
        keys[n] = (String) it.next();
        n++;
      }
      it = null;
    }
    return keys;
  }

  /**
   * starting the cleanuptask for this object cache
   * 
   * @param cleanupTime
   *          time to cleanup in sec.
   * @param taskName
   *          name of the thread.
   */
  public void startCleanupTask(final int cleanupTime, final String taskName) {
    if (timer != null) {
      stopCleanupTask();
    }
    if (timer == null) {
      timer = new Timer(taskName, true);
      timer.scheduleAtFixedRate(new TimerTask() {

        @Override
        public void run() {
          freeObjects(cleanupTime);
        }
      }, cleanupTime * 1000, cleanupTime * 1000);
    }

  }

  /**
   * stopping the cleanup task.
   */
  public void stopCleanupTask() {
    if (timer != null) {
      timer.cancel();
      timer = null;
    }
  }

  public void registerObjectListener(ObjectListener<E> listener) {
    this.objectListener = listener;
  }

  public void removeObjectListener() {
    this.objectListener = null;
  }

  public Collection<E> values() {
    Collection<E> vs = new ArrayList<E>(objectMap.size());
    synchronized (objectMap) {
      for (CacheItem<E> cacheItem : objectMap.values()) {
        vs.add(cacheItem.getObject());
      }
    }
    return vs;
  }

  public static void main(String[] args) throws InterruptedException {
    ObjectCache<Integer> cache = new ObjectCache<Integer>(10);
    for (int i = 0; i < 20; i++) {
      System.out.print(i + ":" + cache.getCacheCount() + ":");
      for (String string : cache.getObjectIDs()) {
        System.out.print(cache.getObjectInternal(string));
        System.out.print(",");
      }
      System.out.println();
      try {
        cache.addObject(Integer.toString(i), new Integer(i));
      } catch (KeyAlreadyExistsException e) {
        e.printStackTrace();
      }
      Thread.sleep(1000);
    }
  }
}
