package de.mcs.utils.caches;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.Properties;

import org.omg.CORBA.SystemException;

import de.mcs.utils.Files;
import de.mcs.utils.StringUtils;

/**
 * Diese Klasse dient der Verwaltung von zwischengespeicherten Blobs auf dem EEX
 * XMLServer. This class is the blob server part. In this class all blobs will
 * be registered and administrated.
 * 
 * @author w.klaas
 */
public class BlobCache {

  class KeyAlreadyExistsException extends Exception {

    /**
         * 
         */
    private static final long serialVersionUID = 715224362894259926L;

    public KeyAlreadyExistsException(String string) {
      super(string);
    }

  }

  /** internal map for the blobs. */
  private HashMap<String, BlobCacheItem> hmBlobs = null;

  /** internal map for the blobs. */
  private HashMap<String, String> hmExternal = null;

  /** max files in a folder. */
  private final int MAXFILECOUNT = 2000;

  private ArrayList<CacheFolder> folders;

  private File blobPath;

  private CacheFolder actualFolder;

  private CacheFolder folder;

  private boolean subfolders;

  /**
   * Diese Klasse dient der Aufnahme eines einzelnen Blobs.
   * 
   * @author w.klaas
   */
  class BlobCacheItem implements Serializable {
    /**
         * 
         */
    private static final long serialVersionUID = -871711242126446479L;

    /** filename of this entry. */
    private String sFilename;

    /** ID of this entry. */
    private String sID;

    /** last access to this item. */
    private Date dAccess;

    private String fileMD5;

    private String externalKey;

    /**
     * Eine neue Datei registrieren, inkl. MD5 Hash bestimmen
     * 
     * @param filename
     *          Filename of the file
     * @param refid
     *          this is the registered id of the document
     * @param persistent
     *          Should this blob be persistent?
     */
    public BlobCacheItem(final String filename, final String externalKey) {
      sFilename = filename;
      fileMD5 = Files.computeMD5FromFile(new File(sFilename));
      // wenn wir keine refid haben ist der MD5 der Datei nicht sehr
      // eindeutig
      // selbst mit refid ist es nicht eindeutig genug
      // darum machen wir noch einen MD5 der MD5 + Timestamp.
      sID = StringUtils.md5String(fileMD5 + (new Date()).getTime());
      dAccess = new Date();
      this.externalKey = externalKey;
    }

    /**
     * Die Blobdatei wieder freigeben.
     */
    public void freeResources() {
      new File(sFilename).delete();
    }

    /**
     * Vor dem abräumen, wird die Datei gelöscht.
     */
    protected void finalize() {
      freeResources();
    }

    public String getFilename() {
      return sFilename;
    }

    public void recalcBlobID() {
      sID = StringUtils.md5String(fileMD5 + (new Date()).getTime());
    }

    public String getExternalKey() {
      return externalKey;
    }

    public String toString() {
      StringBuffer buffer = new StringBuffer();
      buffer.append("filename:");
      buffer.append(sFilename);
      buffer.append(",ID:");
      buffer.append(sID);
      return buffer.toString();
    }
  }

  /**
   * This class represent a cache folder in the file system.
   * 
   * @author w.klaas
   */
  class CacheFolder {
    private int count;

    private File path;

    private int folderNumber;

    private int maxCount;

    /**
     * constructor for a cache folder. Creates automatically the folder.
     * 
     * @param blobPath
     *          path where the folder lives.
     * @param folderCount
     *          number of the folder.
     * @throws IOException
     */
    public CacheFolder(final File blobPath, final int folderCount, final int aMaxCount) throws IOException {
      folderNumber = folderCount;
      maxCount = aMaxCount;
      path = new File(blobPath, Integer.toHexString(folderNumber));
      path.mkdirs();

      // now deleting all files in this folder
      File[] files = path.listFiles();
      for (int i = 0; i < files.length; i++) {
        File file = files[i];
        file.delete();
      }
      new FolderBlocker(path, true);
      initFolder();
    }

    /**
     * initialise the folder. (at this time only recount)
     */
    private void initFolder() {
      recount();
    }

    /**
     * counting all data files.
     */
    private void recount() {
      if (path != null) {
        synchronized (path) {
          File[] files = path.listFiles(new FilenameFilter() {
            public boolean accept(File dir, String name) {
              return name.endsWith(".bin");
            }
          });
          count = files.length;
        }
      }
    }

    /**
     * Anfordern eines neuen Blob Dateinamens.
     * 
     * @return String Name der Datei
     * @throws IOException
     *           Wenn was schiefgeht.
     */
    public final String getNewBlobName(final String prefix, final String suffix) throws IOException {
      String fileName = "";
      synchronized (path) {
        if (count > maxCount) {
          throw new IOException("to many files in this folder");
        }
        fileName = Files.getTempFileName(prefix, suffix, path.getAbsolutePath());
        count++;
      }
      return fileName;
    }

    /**
     * @return the count
     */
    public final int getCount() {
      return count;
    }

    /**
     * @return the folderNumber
     */
    public final int getFolderNumber() {
      return folderNumber;
    }

    /**
     * @return the path
     */
    public final File getPath() {
      return path;
    }

    /**
     * the folder can create a new file.
     * 
     * @return boolean.
     */
    public boolean canCreate() {
      return count < maxCount;
    }

  }

  /**
   * Constructs a new blob cache.
   * 
   * @param blobCachePath
   *          where to store the blob files.
   * @param aSecurity
   *          the security for this blob cache should be activated.
   * @param aServer
   *          the server
   * @param useSubFolders
   *          using subfolders for more than 2000 files.
   * @throws IOException
   */
  public BlobCache(final String blobCachePath, final boolean useSubFolders) throws IOException {
    blobPath = new File(blobCachePath);
    hmBlobs = new HashMap<String, BlobCacheItem>(1000, 100);
    hmExternal = new HashMap<String, String>(100, 100);
    this.subfolders = useSubFolders;
    initBlobCache();
  }

  public BlobCache(File cachePath, boolean useSubFolders) throws IOException {
    blobPath = cachePath;
    hmBlobs = new HashMap<String, BlobCacheItem>(1000, 100);
    hmExternal = new HashMap<String, String>(100, 100);
    this.subfolders = useSubFolders;
    initBlobCache();
  }

  private void initBlobCache() throws IOException {
    if (!blobPath.exists()) {
      blobPath.mkdirs();
    }
    new FolderBlocker(blobPath, true);

    if (subfolders) {
      folders = new ArrayList<CacheFolder>(10);
      addCacheFolder();
    }
  }

  /**
   * @param blobPath
   * @param folders
   * @param folderCount
   * @throws IOException
   */
  private CacheFolder addCacheFolder() throws IOException {
    CacheFolder folder = new CacheFolder(blobPath, folders.size(), MAXFILECOUNT);
    folders.add(folder);
    return folder;
  }

  /**
   * Anfordern eines neuen Blob Dateinamens.
   * 
   * @return String Name der Datei
   * @throws IOException
   *           Wenn was schiefgeht.
   */
  public final String getNewBlobName(final String prefix, final String suffix) throws IOException {
    String fileName = "";
    if (!subfolders) {
      fileName = Files.getTempFileName(prefix, suffix, blobPath.getCanonicalPath());
    } else {
      synchronized (folders) {
        if ((actualFolder == null) || !actualFolder.canCreate()) {
          actualFolder = null;
          for (Iterator<CacheFolder> iter = folders.iterator(); iter.hasNext();) {
            folder = iter.next();
            if (folder.canCreate()) {
              actualFolder = folder;
            }
          }
          if (actualFolder == null) {
            actualFolder = addCacheFolder();
          }
        }
        fileName = actualFolder.getNewBlobName(prefix, suffix);
      }
    }
    if ((fileName == null) || fileName.equals("")) {
      throw new IOException("can't create a new filename.");
    }
    return fileName;
  }

  /**
   * Registering the file in the blobcache. The file is only registered if the
   * id is not already in the cache.
   * 
   * @param blobName
   *          filename of the file to register
   * @param refid
   *          this is the registered id of the document
   * @param contextID
   *          the context id to set for this blob, if security is enabled.
   * @return String the id to find the file in the cache.
   */
  public final String registerBlob(final String blobName) {
    BlobCacheItem myBlob = new BlobCacheItem(blobName, null);
    if (!hasBlob(myBlob.sID)) {
      synchronized (hmBlobs) {
        hmBlobs.put(myBlob.sID, myBlob);
      }
    }
    return myBlob.sID;
  }

  /**
   * Registering the file in the blobcache. The file is only registered if the
   * id is not already in the cache.
   * 
   * @param blobName
   *          filename of the file to register
   * @param refid
   *          this is the registered id of the document
   * @param contextID
   *          the context id to set for this blob, if security is enabled.
   * @return String the id to find the file in the cache.
   * @throws KeyAlreadyExistsException
   */
  public final String registerBlob(final String blobName, final String externalKey) throws KeyAlreadyExistsException {
    if (hasExternalKey(externalKey)) {
      throw new KeyAlreadyExistsException("the extrnal key is already in use.");
    }
    BlobCacheItem myBlob = new BlobCacheItem(blobName, externalKey);
    if (!hasBlob(myBlob.sID)) {
      synchronized (hmBlobs) {
        hmBlobs.put(myBlob.sID, myBlob);
        if (externalKey != null) {
          hmExternal.put(externalKey, myBlob.sID);
        }
      }
    }
    return myBlob.sID;
  }

  /**
   * Pr�fen, ob ein bestimmtes Blob auch zur Verf�gung steht.
   * 
   * @param blobID
   *          die ID des Blobs
   * @return boolean
   */
  public final boolean hasBlob(final String blobID) {
    boolean hasId;
    synchronized (hmBlobs) {
      hasId = hmBlobs.containsKey(blobID);
    }
    return hasId;
  }

  /**
   * Pr�fen, ob ein bestimmtes Blob mit extranl KEy auch zur Verf�gung steht.
   * 
   * @param externalKey
   *          die ID des Blobs
   * @return boolean
   */
  public final boolean hasExternalKey(final String externalKey) {
    boolean hasId;
    synchronized (hmExternal) {
      hasId = hmExternal.containsKey(externalKey);
    }
    return hasId;
  }

  /**
   * Blobid eines bestimmtes Blob mit extranl KEy holen.
   * 
   * @param externalKey
   *          die ID des Blobs
   * @return string, blobid des blobs, oder <code>null</code> wenn nicht
   *         vorhanden
   */
  public final String getIDfromExternalKey(final String externalKey) {
    String blobId = null;
    synchronized (hmExternal) {
      if (hmExternal.containsKey(externalKey)) {
        blobId = hmExternal.get(externalKey);
      }
    }
    return blobId;
  }

  /**
   * Den Dateinamen eines Blobs holen, gleichzeitig wird das Blob für die
   * nächste zeitspanne freigeschaltet. with security.
   * 
   * @param blobID
   *          die ID des Blobs
   * @param contextID
   *          the contextid to use with this blob
   * @return String Dateiname des Blobs
   * @throws BusinessComponentsException
   * @throws SystemException
   * @throws IOException
   */
  public final String getBlobFileName(final String blobID) throws IOException {
    String sBlobFileName = null;
    synchronized (hmBlobs) {
      if (hasBlob(blobID)) {
        BlobCacheItem myBlob = (BlobCacheItem) hmBlobs.get(blobID);
        myBlob.dAccess = new Date();
        sBlobFileName = myBlob.sFilename;
      }
      return sBlobFileName;
    }
  }

  /**
   * Den Dateinamen eines Blobs holen, gleichzeitig wird das Blob für die
   * nächste zeitspanne freigeschaltet. with security.
   * 
   * @param blobID
   *          die ID des Blobs
   * @return String Dateiname des Blobs
   * @throws BusinessComponentsException
   * @throws SystemException
   * @throws IOException
   */
  public final String getBlobProperties(final String blobID) throws IOException {
    Properties props = new Properties();
    String sBlobFileName = "";
    synchronized (hmBlobs) {
      if (hasBlob(blobID)) {
        BlobCacheItem myBlob = (BlobCacheItem) hmBlobs.get(blobID);
        myBlob.dAccess = new Date();
        sBlobFileName = myBlob.sFilename;
        props.setProperty("filename", sBlobFileName);
        props.setProperty("length", Long.toString(new File(myBlob.sFilename).length()));
        props.setProperty("md5", myBlob.fileMD5);
        props.setProperty("id", myBlob.sID);
        sBlobFileName = props.toString();
      }
    }
    return sBlobFileName;
  }

  /**
   * Hier werden jetzt alle Blobs die älter sind als die Angabe, deregistriert.
   * Bei der n�chsten GC werden diese dann gel�scht
   * 
   * @param sec
   *          Anzahl der Sekunden die die Blobs alt sein d�rfen
   */
  public final void freeBlobs(final long sec) {
    Date myEqualDate = new Date(new Date().getTime() - (sec * 1000));
    BlobCacheItem myBlobItem;
    ArrayList<String> vDelKeys = new ArrayList<String>(hmBlobs.size());
    String sKey = "";
    List<String> filenames = new ArrayList<String>();
    synchronized (hmBlobs) {
      Iterator<String> it = hmBlobs.keySet().iterator();
      while (it.hasNext()) {
        sKey = (String) it.next();
        myBlobItem = (BlobCacheItem) hmBlobs.get(sKey);
        if (myBlobItem.getFilename() != null) {
          File file = new File(myBlobItem.getFilename());
          filenames.add(file.getName());
          if (!file.exists()) {
            vDelKeys.add(sKey);
          }
        }
        if (myEqualDate.after(myBlobItem.dAccess)) {
          // now deleting the file
          myBlobItem.freeResources();
          vDelKeys.add(sKey);
        }
      }
      it = null;
      for (int n = 0; n < vDelKeys.size(); n++) {
        hmBlobs.remove((String) vDelKeys.get(n));
      }
      vDelKeys = null;
    }
    synchronized (hmExternal) {
      Iterator<Entry<String, String>> it = hmExternal.entrySet().iterator();
      while (it.hasNext()) {
        Entry<String, String> entry = it.next();
        if (!hasBlob(entry.getValue())) {
          it.remove();
        }
      }
    }
    if (folders != null) {
      // jetzt alle cache folder aktualisieren
      for (Iterator<CacheFolder> iter = folders.iterator(); iter.hasNext();) {
        CacheFolder folder = iter.next();
        folder.recount();
      }
    }
  }

  /**
   * Holt die aktuelle Anzahl der im Cache liegenden Blobs.
   * 
   * @return int Anzahl der Blobs im Cache
   */
  public final int getBlobCacheCount() {
    int size;
    synchronized (hmBlobs) {
      size = hmBlobs.size();
    }
    return size;
  }

  /**
   * Vor dem abräumen, wird die Datei gelöscht.
   */
  public final void freeResources() {
    // alle Blobs löschen
    freeBlobs(0);
  }

  /**
   * Besort eine Liste aller BlobID's.
   * 
   * @return String[]
   */
  public final String[] getBlobIDs() {
    String[] keys;
    synchronized (hmBlobs) {
      keys = new String[hmBlobs.size()];
      Iterator<String> it = hmBlobs.keySet().iterator();
      int n = 0;
      while (it.hasNext()) {
        keys[n] = (String) it.next();
        n++;
      }
      it = null;
    }
    return keys;
  }

}
