/*
 * MCS Media Computer Software Copyright (c) 2005 by MCS
 * -------------------------------------- Created on 16.01.2004 by w.klaas
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package de.mcs.utils;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilePermission;
import java.io.IOException;
import java.security.AccessControlException;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map.Entry;
import java.util.jar.JarOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

/**
 * Modifies files in an .ear or .jar archive file.
 */
public class ZipUpdater {

  /** the archive file to work with. */
  private File archiveFile;

  /** the archive file to work with. */
  private ZipInputStream archive;

  /** the base directory. */
  private String baseDirPath; // OS dependent path

  /** temp name of a file. */
  private String zipname;

  /**
   * converting a filename to a zip name.
   * 
   * @param osFilename
   *          filename to convert
   * @return converted name.
   */
  private String toZipName(final String osFilename) {
    String zipName = osFilename;
    if (null != baseDirPath && baseDirPath.length() > 0 && baseDirPath.length() < osFilename.length()
        && osFilename.startsWith(baseDirPath)) {

      zipName = osFilename.substring(baseDirPath.length() + 1);
    }
    return zipName.replace(File.separatorChar, '/');
  }

  /**
   * Constructing the xar updater.
   * 
   * @param aArchiveFile
   *          The archive file to use.
   * @param baseDirFile
   *          the base file to work with.
   * @throws IOException
   *           if something goes wrong.
   */
  public ZipUpdater(final File aArchiveFile, final File baseDirFile) throws IOException {
    super();
    if (!aArchiveFile.exists()) {
      throw new FileNotFoundException(aArchiveFile.getAbsolutePath());
    }
    if (!aArchiveFile.canWrite()) {
      throw new AccessControlException("No write access", new FilePermission(aArchiveFile.getAbsolutePath(), "write"));
    }
    this.archiveFile = aArchiveFile;

    // File(".");
    if (null == baseDirFile) {
      throw new NullPointerException("baseDirFile should not be null.");
    }
    if (!baseDirFile.exists()) {
      throw new FileNotFoundException(baseDirFile.getCanonicalPath());
    }
    if (0 == baseDirFile.getName().length()) {
      baseDirPath = new File(System.getProperty("java.io.tmpdir")).getPath(); // new
    } else {
      baseDirPath = baseDirFile.getPath();
    }

    zipname = toZipName(aArchiveFile.getPath());
  }

  /**
   * Constructing the xar updater.
   * 
   * @param archiveFilename
   *          The archive file to use.
   * @param baseDir
   *          the base file to work with.
   * @throws IOException
   *           if something goes wrong.
   */
  public ZipUpdater(final String archiveFilename, final String baseDir) throws IOException {
    this(new File(archiveFilename), new File(baseDir));
  }

  /**
   * @return getting the archive file.
   */
  public final File getFile() {
    return archiveFile;
  }

  /** some special dir and file names. */
  static final String METAINF_NAME = "META-INF/";

  /** some special dir and file names. */
  static final String INDEX_NAME = METAINF_NAME + "INDEX.LIST";

  /** some special dir and file names. */
  static final String MANIFEST_NAME = METAINF_NAME + "MANIFEST.MF";

  /**
   * adding a file to the zip.
   * 
   * @param zipoutputstream
   *          the outputstream
   * @param aName
   *          name of the entry
   * @param file
   *          the file to add
   * @param buffer
   *          a buffer
   * @throws IOException
   *           if something goes wrong.
   */
  public final void addFile(final ZipOutputStream zipoutputstream, final String aName, final File file,
      final byte[] buffer) throws IOException {
    String name = aName;
    boolean isDir = file.isDirectory();
    if (isDir) {
      name = name.endsWith("/") ? name : name + "/";
    }
    if (name.startsWith("/")) {
      name = name.substring(1);
    } else if (name.startsWith("./")) {
      name = name.substring(2);
    }

    if (name.equals("") || name.equals(".") || name.equals(zipname)) {
      return;
    }

    long l = isDir ? 0L : file.length();

    // System.out.print("is file:" + name);
    // System.out.println(" length:" + Long.toString(l));

    ZipEntry zipentry = new ZipEntry(name);
    zipentry.setTime(file.lastModified());
    zipentry.setMethod(ZipEntry.DEFLATED);
    if (l == 0L) {
      zipentry.setMethod(0);
      zipentry.setSize(0L);
      zipentry.setCrc(0L);
    }
    zipoutputstream.putNextEntry(zipentry);
    if (!isDir) {
      BufferedInputStream bufferedinputstream = new BufferedInputStream(new FileInputStream(file));
      int i;
      while ((i = bufferedinputstream.read(buffer, 0, buffer.length)) > 0) {
        zipoutputstream.write(buffer, 0, i);
      }

      bufferedinputstream.close();
    }
    zipoutputstream.closeEntry();

  }

  /**
   * update an archive.
   * 
   * @param toArchiveFile
   *          the archive file to update
   * @param files
   *          the files to zip
   * @param asNames
   *          as which names
   * @throws IOException
   *           if something goes wrong.
   */
  public final void update(final File toArchiveFile, final File[] files, final String[] asNames) throws IOException {

    if (null == archive) {
      archive = new ZipInputStream(new FileInputStream(archiveFile));
    }

    if (null == files) {
      return;
    }
    Hashtable<String, File> hashtable = new Hashtable<String, File>();
    for (int index = 0; index < files.length; index++) {
      String name = toZipName(files[index].getPath());
      if (null != asNames && null != asNames[index]) {
        name = asNames[index];
      }
      hashtable.put(name, files[index]);
    }
    hashtable.remove(INDEX_NAME); // currently no META-INF/INDEX.LIST
    // support at all
    hashtable.remove(MANIFEST_NAME); // currently no META-INF/MANIFEST.MF
    // update support

    JarOutputStream jaroutputstream = new JarOutputStream(new FileOutputStream(toArchiveFile));
    jaroutputstream.setLevel(java.util.zip.Deflater.BEST_COMPRESSION);

    byte[] buffer = new byte[8 * 1024];
    ZipEntry zipentry;
    while (null != (zipentry = archive.getNextEntry())) {
      String name = zipentry.getName();
      if (!name.equalsIgnoreCase(INDEX_NAME)) {
        if (!hashtable.containsKey(name)) {
          ZipEntry newEntry = new ZipEntry(name);
          newEntry.setMethod(zipentry.getMethod());
          newEntry.setTime(zipentry.getTime());
          newEntry.setComment(zipentry.getComment());
          newEntry.setExtra(zipentry.getExtra());
          if (ZipEntry.STORED == zipentry.getMethod()) {
            newEntry.setSize(zipentry.getSize());
            newEntry.setCrc(zipentry.getCrc());
          }
          jaroutputstream.putNextEntry(newEntry);
          int len;
          while ((len = archive.read(buffer, 0, buffer.length)) != -1) {
            jaroutputstream.write(buffer, 0, len);
          }
        } else {
          addFile(jaroutputstream, name, (File) hashtable.get(name), buffer);
          hashtable.remove(name);
        }
      }
    }

    if (!hashtable.isEmpty()) {
      for (Iterator<Entry<String, File>> iter = hashtable.entrySet().iterator(); iter.hasNext();) {
        Entry<String, File> entry = iter.next();
        addFile(jaroutputstream, entry.getKey(), entry.getValue(), buffer);
      }
    }

    jaroutputstream.close();

  }

  /**
   * closing the zip file.
   * 
   * @throws IOException
   *           if something goes wrong.
   */

  public final void close() throws IOException {
    if (null != archive) {
      archive.close();
      archive = null;
    }
  }

  /**
   * @return string representing this object.
   * @see java.lang.Object#toString()
   */
  public final String toString() {
    return archiveFile.toString();
  }

}
