/*
 * MCS Media Computer Software Copyright (c) 2005 by MCS
 * -------------------------------------- Created on 16.01.2004 by w.klaas
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package de.mcs.utils;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilePermission;
import java.io.IOException;
import java.io.InputStream;
import java.security.AccessControlException;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/**
 * Extract files from a .ear, .war or .jar archive file.
 * 
 * @author j.froehlich
 */
public class ZipExtracter {

  /** the archive file to work with. */
  private File archiveFile;

  /** the archive file to work with. */
  private ZipFile archive;

  /** the base directory. */
  private String baseDirPath; // OS dependent path

  /** a recursion counter. */
  private static int recursioncnt = 0;

  /**
   * @return getting the archive file.
   */
  public final File getFile() {
    return archiveFile;
  }

  /**
   * @see java.util.zip.ZipFile#getName()
   */
  public final String getName() {
    return null == archive ? null : archive.getName();
  }

  /**
   * converting a filename to a zip name.
   * 
   * @param osFilename
   *          filename to convert
   * @return converted name.
   */
  private String toZipName(final String osFilename) {
    String zipName = osFilename;
    if (null != baseDirPath && baseDirPath.length() > 0 && baseDirPath.length() < osFilename.length()
        && osFilename.startsWith(baseDirPath)) {
      zipName = osFilename.substring(baseDirPath.length() + 1);
    }
    return zipName.replace(File.separatorChar, '/');
  }

  /**
   * constructor with the archive name and a working target.
   * 
   * @param aArchiveFile
   *          the archive name
   * @param aBaseDirFile
   *          where to extract.
   * @throws IOException
   *           if something goes wrong.
   */
  public ZipExtracter(final File aArchiveFile, final File aBaseDirFile) throws IOException {
    super();
    if (!aArchiveFile.exists()) {
      throw new FileNotFoundException(aArchiveFile.getAbsolutePath());
    }
    if (!aArchiveFile.canRead()) {
      throw new AccessControlException("No read access", new FilePermission(aArchiveFile.getAbsolutePath(), "read"));
    }
    this.archiveFile = aArchiveFile;

    if (null == aBaseDirFile || 0 == aBaseDirFile.getName().length()) {
      baseDirPath = new File(System.getProperty("java.io.tmpdir")).getPath();
    } else {
      baseDirPath = aBaseDirFile.getPath();
    }

    toZipName(aArchiveFile.getPath());
  }

  /**
   * constructor with the archive name and a working target.
   * 
   * @param archiveFilename
   *          the archive name
   * @param baseDir
   *          where to extract.
   * @throws IOException
   *           if something goes wrong.
   */
  public ZipExtracter(final String archiveFilename, final String baseDir) throws IOException {
    this(new File(archiveFilename), new File(baseDir));
  }

  /**
   * doing the extraction.
   * 
   * @param fromFilename
   *          filename in zip file
   * @param toLocalFilename
   *          local file name
   * @return File
   * @throws IOException
   *           if something goes wrong.
   */
  public final File extract(final String fromFilename, final String toLocalFilename) throws IOException {
    if (null == archive) {
      archive = new ZipFile(archiveFile);
    }

    File localFilename;
    if (null == toLocalFilename) {
      localFilename = new File(this.baseDirPath, fromFilename);
    } else {
      localFilename = new File(toLocalFilename);
    }
    File parent = localFilename.getParentFile();
    if (null != parent) {
      parent.mkdirs();
    }

    Enumeration<? extends ZipEntry> archiveEntries = archive.entries();
    byte[] buffer = new byte[8 * 1024];
    while (archiveEntries.hasMoreElements()) {
      ZipEntry entry = (ZipEntry) archiveEntries.nextElement();
      if (!entry.isDirectory() && entry.getName().equalsIgnoreCase(fromFilename) && entry.getSize() > 0) {
        // --- extracting ---
        InputStream input = archive.getInputStream(entry);
        FileOutputStream output = new FileOutputStream(localFilename);
        int len;
        while ((len = input.read(buffer, 0, buffer.length)) != -1) {
          output.write(buffer, 0, len);
        }
        output.close();
        // ------------------
        return localFilename;
      }
    }
    throw new FileNotFoundException("The file '" + fromFilename + "' is not contained in '" + archive.getName()
        + " or is a directory.");

  }

  /**
   * closing the zip file.
   * 
   * @throws IOException
   *           if something goes wrong.
   */
  public final void close() throws IOException {
    if (null != archive) {
      archive.close();
      archive = null;
    }
  }

  /**
   * @see java.lang.Object#toString()
   */
  public final String toString() {
    return archiveFile.toString();
  }

  /**
   * extract all files from the zip file.
   * 
   * @throws IOException
   *           if something goes wrong.
   */
  public final void extractAll() throws IOException {
    extractAll(false, 0);
  }

  /**
   * extract all files from the zip file, with all or none subfolder, or with
   * subfolder up to the defined depth..
   * 
   * @param recursive
   *          should all subfolder be extracted.
   * @param recursionDepth
   *          if recursive = true, how depth-
   * @throws IOException
   *           if something goes wrong.
   */
  public final void extractAll(final boolean recursive, final int recursionDepth) throws IOException {
    if (null == archive) {
      archive = new ZipFile(archiveFile);
    }

    Enumeration<? extends ZipEntry> archiveEntries = archive.entries();
    byte[] buffer = new byte[8 * 1024];
    while (archiveEntries.hasMoreElements()) {
      ZipEntry entry = (ZipEntry) archiveEntries.nextElement();
      if (!entry.isDirectory() && entry.getSize() > 0) {
        // --- extracting ---
        File outfile = new File(baseDirPath, entry.getName());
        outfile.getParentFile().mkdirs();
        InputStream input = archive.getInputStream(entry);
        FileOutputStream output = new FileOutputStream(outfile);
        int len;
        while ((len = input.read(buffer, 0, buffer.length)) != -1) {
          output.write(buffer, 0, len);
        }
        output.close();
        input.close();
        // --- recursion ---
        // in case of a .zip .war .ear or .jar file, we can unzip them
        // too! just if u want to...
        if (recursive && entry.getName().matches("(.*[(e)|(w)|(j)]ar)|(.*zip)") && recursioncnt < recursionDepth) {
          recursioncnt++;
          File subdir = new File(baseDirPath, entry.getName().substring(0, entry.getName().length() - 4));
          File subjar = new File(baseDirPath, entry.getName());
          ZipExtracter xarex = new ZipExtracter(subjar, subdir);
          xarex.extractAll(true, recursionDepth);
          xarex.close();
          recursioncnt--;
        }
      }
    }
  }
}
