/*
 * MCS Media Computer Software Copyright (c) 2005 by MCS
 * -------------------------------------- Created on 16.01.2004 by w.klaas
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package de.mcs.utils;

/**
 * this is a littel class to help versioning. Version could be build from a
 * string with the following syntax: major.minor.release.build
 * 
 * @author w.klaas
 */
public class Version {

  /** major version. */
  private int major;

  /** minor version. */
  private int minor;

  /** release version. */
  private int release;

  /** build version. */
  private int build;

  /**
   * constructor for empty version.
   */
  public Version() {
    major = -1;
    minor = -1;
    release = -1;
    build = -1;
  }

  /**
   * Constructor for creating a version object from an string.
   * 
   * @param version
   *            string to convert
   */
  public Version(final String version) {
    this();
    parseString(version);
  }

  /**
   * parsing a string into this version class.
   * 
   * @param version
   *            string to parse
   */
  public final void parseString(final String version) {
    String[] versionvalues = version.split("\\.");
    for (int i = 0; i < versionvalues.length; i++) {
      String string = versionvalues[i];
      switch (i) {
      case 0:
        major = Integer.parseInt(string);
        break;
      case 1:
        minor = Integer.parseInt(string);
        break;
      case 2:
        release = Integer.parseInt(string);
        break;
      case 3:
        build = Integer.parseInt(string);
        break;
      default:
        break;
      }
    }
  }

  /**
   * constructor from another version class.
   * 
   * @param version
   *            version to use
   */
  public Version(final Version version) {
    this();
    this.major = version.major;
    this.minor = version.minor;
    this.release = version.release;
    this.build = version.build;
  }

  /** increment the major field. */
  public final void incrementMajor() {
    major++;
  }

  /** increment the minor field. */
  public final void incrementMinor() {
    minor++;
  }

  /** incementing one the release field. */
  public final void incrementRelease() {
    release++;
  }

  /** increment the build field. */
  public final void incrementBuild() {
    build++;
  }

  /** decrement the major field. */
  public final void decrementMajor() {
    major--;
  }

  /** decrement the minor field. */
  public final void decrementMinor() {
    minor--;
  }

  /** decrement the release field. */
  public final void decrementRelease() {
    release--;
  }

  /** decrement the build field. */
  public final void decrementBuild() {
    build--;
  }

  /**
   * is this version geater than the destination version?
   * 
   * @param dest
   *            destination version
   * @return <code>true</code> if this version is greater than the
   *         destination version, else <code>false</code>
   */
  public final boolean greaterThan(final Version dest) {
    if (major == dest.major) {
      if (minor == dest.minor) {
        if (release == dest.release) {
          return build > dest.build;
        } else {
          return release > dest.release;
        }
      } else {
        return minor > dest.minor;
      }
    } else {
      return major > dest.major;
    }
  }

  /**
   * is this version lesser than the destination version?
   * 
   * @param dest
   *            destination version
   * @return <code>true</code> if this version is lesser than the
   *         destination version, else <code>false</code>
   */
  public final boolean lesserThan(final Version dest) {
    return !equals(dest) && !greaterThan(dest);
  }

  /**
   * is this version equal to the destination version?
   * 
   * @param arg0
   *            destination version
   * @return <code>true</code> if this version is eqaul than the destination
   *         version, else <code>false</code>
   */
  public final boolean equals(final Object arg0) {
    if (arg0 instanceof Version) {
      Version dest = (Version) arg0;
      if (major == dest.major) {
        if (minor == dest.minor) {
          if (release == dest.release) {
            return build == dest.build;
          } else {
            return false;
          }
        } else {
          return false;
        }
      } else {
        return false;
      }
    } else {
      return false;
    }
  }

  /**
   * converting this version to an version string.
   * 
   * @return String
   */
  public final String toString() {
    StringBuffer stringBuffer = new StringBuffer(20);
    stringBuffer.append(major);
    if (minor >= 0) {
      stringBuffer.append(".").append(minor);
      if (release >= 0) {
        stringBuffer.append(".").append(release);
        if (build >= 0) {
          stringBuffer.append(".").append(build);
        }
      }
    }
    return stringBuffer.toString();
  }

  /**
   * Main methode for testing purpose only.
   * 
   * @param args
   *            commandline arguments
   */
  public static void main(final String[] args) {
    Version version = new Version("2.1.2.97");
    Version version1 = new Version("2.1.2.96");
    System.out.print("Version " + version.toString() + " > " + version1.toString() + ": ");
    System.out.println(version.greaterThan(version1));

    System.out.print("Version " + version.toString() + " < " + version1.toString() + ": ");
    System.out.println(version.lesserThan(version1));

    System.out.print("Version " + version.toString() + " = " + version1.toString() + ": ");
    System.out.println(version.equals(version1));

    System.out.print("Version " + version.toString() + " = \"String\": ");
    System.out.println(version.equals("String"));

    version1 = new Version(version.toString());
    System.out.print("Version " + version.toString() + " > " + version1.toString() + ": ");
    System.out.println(version.greaterThan(version1));

    System.out.print("Version " + version.toString() + " < " + version1.toString() + ": ");
    System.out.println(version.lesserThan(version1));

    System.out.print("Version " + version.toString() + " = " + version1.toString() + ": ");
    System.out.println(version.equals(version1));

    version = new Version("2.1.97");
    version1 = new Version("2.1.96");
    System.out.print("Version " + version.toString() + " > " + version1.toString() + ": ");
    System.out.println(version.greaterThan(version1));
  }

  /**
   * getting the build field.
   * 
   * @return int
   */
  public final int getBuild() {
    return build;
  }

  /**
   * getting the major field.
   * 
   * @return int
   */
  public final int getMajor() {
    return major;
  }

  /**
   * getting the minor field.
   * 
   * @return int
   */
  public final int getMinor() {
    return minor;
  }

  /**
   * getting the release field.
   * 
   * @return int
   */
  public final int getRelease() {
    return release;
  }

  /**
   * getting the hashcode.
   * 
   * @return int
   */
  public final int hashCode() {
    int hash = major ^ minor ^ release ^ build;
    return hash;
  }

  /**
   * @param aBuild
   *            setting the build number.
   */
  public final void setBuild(final int aBuild) {
    this.build = aBuild;
  }

  /**
   * @param aMajor
   *            setting the major number.
   */
  public final void setMajor(final int aMajor) {
    this.major = aMajor;
  }

  /**
   * @param aMinor
   *            setting the minor number.
   */
  public final void setMinor(final int aMinor) {
    this.minor = aMinor;
  }

  /**
   * @param aRelease
   *            setting the release number.
   */
  public final void setRelease(final int aRelease) {
    this.release = aRelease;
  }

}
