/*
 * MCS Media Computer Software Copyright (c) 2006 by MCS
 * -------------------------------------- Created on 11.04.2006 by w.klaas
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package de.mcs.utils;

import java.util.ArrayList;
import java.util.Queue;
import java.util.concurrent.LinkedBlockingQueue;

/**
 * this class is intend to implement a thread pool with the possibility to start
 * and stop all threads and to interact with some other parts of the program.
 * 
 * @author w.klaas
 */
public class ThreadPool extends Thread {

    /** the thread pool. */
    private ArrayList<WorkingThread> pool;

    /** the working queue. */
    private Queue<Runnable> queue;

    /** maximal count of threads. */
    private int maxCount;

    /**
     * instancing the thread pool. With max count of parallel threads.
     * 
     * @param maxcount
     *            maximal count of threads
     */
    public ThreadPool(final int maxcount) {
        this.maxCount = maxcount;
        this.pool = new ArrayList<WorkingThread>(maxCount);
        this.queue = new LinkedBlockingQueue<Runnable>(100);
    }

    /**
     * Adding a new working part to the queue.
     * 
     * @param work
     *            the runnable to start.
     * @return <code>true</code> if the work could be added, otherwise
     *         <code>false</code>
     */
    public final boolean addWorkingPart(final Runnable work) {
        return queue.offer(work);
    }

    /**
     * removing a working part from the queue.
     * 
     * @param work
     *            the runnable to remove.
     * @return <code>true</code> if the work could be removed, otherwise
     *         <code>false</code>
     */
    public final boolean removeWorkingPart(final Runnable work) {
        return queue.remove(work);
    }

    /**
     * @see java.lang.Thread#run()
     */
    @Override
    public final void run() {
        while (!isInterrupted()) {
            if (!queue.isEmpty()) {
                Runnable run = queue.poll();
                try {
                    WorkingThread thread = getNextFreeThread();
                    thread.setRunnable(run);
                    thread.start();
                } catch (InterruptedException e) {
                    // e.printStackTrace();
                }
            } else {
                try {
                    sleep(100);
                } catch (InterruptedException e) {
                    // e.printStackTrace();
                }
            }
        }
    }

    /**
     * getting the next thread that should be startet.
     * 
     * @return WorkingThread
     * @throws InterruptedException
     *             if the process ahould be stoped.
     */
    private WorkingThread getNextFreeThread() throws InterruptedException {
        // zun�chst mal schauen ob die maxcount nicht geändert worden ist und
        // demnach dann die Anzahl der Working Threads anpassen
        while (maxCount < pool.size()) {
            // jetzt warten bis die Anzahl wieder stimmt.
            WorkingThread thread = getFreeThread();
            pool.remove(thread);
            thread = null;
        }
        while (maxCount > pool.size()) {
            WorkingThread thread = new WorkingThread();
            pool.add(thread);
        }
        return getFreeThread();
    }

    /**
     * getting the next thread that should be startet.
     * 
     * @return WorkingThread
     * @throws InterruptedException
     *             if the process ahould be stoped.
     */
    private WorkingThread getFreeThread() throws InterruptedException {
        while (true) {
            for (WorkingThread thread : pool) {
                if (thread.isFree()) {
                    thread.setBusy();
                    return thread;
                }
            }
            sleep(50);
        }
    }
}
