/*
 * MCS Media Computer Software Copyright (c) 2005 by MCS
 * -------------------------------------- Created on 16.01.2004 by w.klaas
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package de.mcs.utils;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.StringTokenizer;

import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;

import de.mcs.utils.codecs.Base64;

/**
 * This class provides static util methods for String manaipulation that aren't
 * part of the default JDK functionalities.
 * 
 * @author w.klaas
 */
public final class StringUtils {
  /** SPCAE Character. */
  private static final String SPACE = " ";

  /**
   * private constructor, to provide instancing.
   */
  private StringUtils() {
  }

  /**
   * Replaces all the occurences of a substring found within a StringBuffer by a
   * replacement string.
   * 
   * @param buffer
   *          the StringBuffer in where the replace will take place
   * @param find
   *          the substring to find and replace
   * @param replacement
   *          the replacement string for all occurences of find
   * @return the original StringBuffer where all the occurences of find are
   *         replaced by replacement
   */
  public static StringBuffer replaceAll(final StringBuffer buffer, final String find, final String replacement) {
    int bufidx = buffer.length() - 1;
    int offset = find.length();
    while (bufidx > -1) {
      int findidx = offset - 1;
      while (findidx > -1) {
        if (bufidx == -1) {
          // Done
          return buffer;
        }
        if (buffer.charAt(bufidx) == find.charAt(findidx)) {
          findidx--; // Look for next char
          bufidx--;
        } else {
          findidx = offset - 1; // Start looking again
          bufidx--;
          if (bufidx == -1) {
            // Done
            return buffer;
          }
          continue;
        }
      }
      buffer.replace(bufidx + 1, bufidx + 1 + offset, replacement);
      // start looking again
    }
    // No more matches
    return buffer;

  }

  /**
   * convinient methode for converting. seperator will be ',' delimiter will be
   * '"'.
   * 
   * @param List
   *          <String> The array of strings input.
   * @return String A string containing tokens separated by seperator.
   */
  public static String listToCSVString(List<String> list) {
    return listToCSVString(list, false);
  }

  public static String listToCSVString(List<String> list, boolean forceDelimiter) {
    // TODO Auto-generated method stub
    String[] array = new String[list.size()];
    for (int i = 0; i < array.length; i++) {
      array[i] = list.get(i).toString();
    }
    return arrayToCSVString(array, ',', '"', forceDelimiter);
  }

  /**
   * convinient methode for converting. seperator will be ',' delimiter will be
   * '"'.
   * 
   * @param array
   *          The array of strings input.
   * @return String A string containing tokens separated by seperator.
   */
  public static String arrayToCSVString(final String[] array) {
    return arrayToCSVString(array, ',', '"');
  }

  public static String arrayToCSVString(final String[] array, final char localseparators, final char localdelimiter) {
    return arrayToCSVString(array, localseparators, localdelimiter, false);
  }

  /**
   * Takes an array of tokens and converts into separator-separated string.
   * 
   * @param array
   *          The array of strings input.
   * @param localseparators
   *          The separator char.
   * @param localdelimiter
   *          The delimiter char.
   * @return String A string containing tokens separated by seperator.
   */
  public static String arrayToCSVString(final String[] array, final char localseparators, final char localdelimiter,
      boolean forceDelimiter) {
    String delimiter = Character.toString(localdelimiter);
    String separators = Character.toString(localseparators);
    StringBuffer sb = new StringBuffer("");
    String empty = "";

    if ((array == null) || (array.length == 0)) {
      return empty;
    }

    if (delimiter == null) {
      delimiter = "\"";
    }

    if (separators == null) {
      separators = ",";
    }

    for (int ix = 0; ix < array.length; ix++) {
      // if (array[ix] != null && !array[ix].equals("")) {
      if (array[ix] != null) {
        StringBuffer value = new StringBuffer(array[ix]);
        if (value.indexOf(delimiter) >= 0) {
          value = replaceAll(value, delimiter, delimiter + delimiter);
        }
        if (forceDelimiter || (array[ix].indexOf(separators) >= 0) || (array[ix].indexOf(delimiter) >= 0)
            || array[ix].startsWith(" ")) {
          sb.append(delimiter);
          sb.append(value);
          sb.append(delimiter);
          sb.append(separators);
        } else {
          sb.append(value);
          sb.append(separators);
        }
      }
    }
    String str = sb.toString();
    if (!str.equals("")) {
      str = str.substring(0, (str.length() - separators.length()));
    }
    return str;
  }

  /**
   * Converts a delimited string into an array of string tokens. Separator is
   * ',', delimiter is '"'
   * 
   * @param string
   *          The 'separator' separated string.
   * @return String[] A string array of the original tokens.
   */
  public static String[] csvStringToArray(final String string) {
    return csvStringToArray(string, ',', '"');
  }

  /**
   * Converts a delimited string into an array of string tokens.
   * 
   * @param str
   *          The 'separator' separated string.
   * @param localseparators
   *          The separator char.
   * @param localdelimiter
   *          The delimiter char.
   * @return String[] A string array of the original tokens.
   */
  public static String[] csvStringToArray(final String str, final char localseparators, final char localdelimiter) {
    /**
     * Parse comma-separated values (CSV), a common Windows file format. Sample
     * input: "LU",86.25,"11/4/1998","2:19PM",+4.0625
     * <p>
     * Inner logic adapted from a C++ original that was Copyright (C) 1999
     * Lucent Technologies Excerpted from 'The Practice of Programming' by Brian
     * W. Kernighan and Rob Pike.
     * <p>
     * Included by permission of the http://tpop.awl.com/ web site, which says:
     * "You may use this code for any purpose, as long as you leave the
     * copyright notice and book citation attached." I have done so.
     * 
     * @author Brian W. Kernighan and Rob Pike (C++ original)
     * @author Ian F. Darwin (translation into Java and removal of I/O)
     * @author Ben Ballard (rewrote advQuoted to handle '""' and for
     *         readability)
     */
    class CSV {

      // public static final char DEFAULT_SEP = ',';

      // public static final char DEFAULT_DELIM = '\"';

      /** Construct a CSV parser, with the default separator (`,'). */
      // private CSV() {
      // this(DEFAULT_SEP, DEFAULT_DELIM);
      // }

      /**
       * Construct a CSV parser with a given separator. Must be exactly the
       * string that is the separator, not a list of separator characters!
       * 
       * @param delim
       *          field delimiter
       * @param sep
       *          field separator
       */
      public CSV(final char sep, final char delim) {
        fieldSep = sep;
        fieldDelim = delim;
      }

      /** The fields in the current String */
      ArrayList<String> list = new ArrayList<String>();

      /** the separator char for this parser */
      private char fieldSep;

      /** the delimiter char for this parser */
      private char fieldDelim;

      /**
       * parse: break the input String into fields
       * 
       * @param line
       *          the line to parse
       * @return java.util.Iterator containing each field from the original as a
       *         String, in order.
       */
      public final Iterator<String> parse(final String line) {
        StringBuffer sb = new StringBuffer();
        list.clear(); // discard previous, if any
        int i = 0;

        if (line.length() == 0) {
          // list.add(line);
          return list.iterator();
        }
        do {
          sb.setLength(0);
          if (i < line.length() && line.charAt(i) == fieldDelim) {
            i = advQuoted(line, sb, ++i); // skip quote
          } else {
            i = advPlain(line, sb, i);
          }
          list.add(sb.toString());
          // Debug.println("csv", sb.toString());
          i++;
        } while (i < line.length());
        return list.iterator();
      }

      /**
       * advQuoted: quoted field; return index of next separator
       * 
       * @param s
       *          String
       * @param sb
       *          StringBuffer
       * @param i
       *          int
       * @return int
       */
      private int advQuoted(final String s, final StringBuffer sb, final int i) {
        int j;
        int len = s.length();
        for (j = i; j < len; j++) {
          if (s.charAt(j) == fieldDelim && j + 1 < len) {
            if (s.charAt(j + 1) == fieldDelim) {
              j++; // skip escape char
            } else if (s.charAt(j + 1) == fieldSep) { // next
              // delimeter
              j++; // skip end quotes
              break;
            }
          } else if (s.charAt(j) == fieldDelim && j + 1 == len) { // end
            // quotes at end of line
            break; // done
          }
          sb.append(s.charAt(j)); // regular character.
        }
        return j;
      }

      /**
       * advPlain: unquoted field; return index of next separator
       * 
       * @param s
       *          String
       * @param sb
       *          StringBuffer
       * @param i
       *          int
       * @return int
       */
      private int advPlain(final String s, final StringBuffer sb, final int i) {
        int j;

        j = s.indexOf(fieldSep, i); // look for separator
        // Debug.println("csv", "i = " + i + ", j = " + j);
        if (j == -1) { // none found
          sb.append(s.substring(i));
          return s.length();
        } else {
          sb.append(s.substring(i, j));
          return j;
        }
      }
    }
    CSV csv = new CSV(localseparators, localdelimiter);
    csv.parse(str);
    return (String[]) csv.list.toArray(new String[0]);
  }

  /**
   * Remove a given set of characters from a String.
   * 
   * @param data
   *          The input string to be cleansed of 'removeChars'.
   * @param removeChars
   *          The characters to be removed.
   * @return String The new string cleansed of 'removeChars'.
   */
  public static String removeChars(final String data, final String removeChars) {
    String temp = null;
    StringBuffer out = new StringBuffer();
    temp = data;

    StringTokenizer st = new StringTokenizer(temp, removeChars);
    while (st.hasMoreTokens()) {
      String element = (String) st.nextElement();
      out.append(element);
    }
    return out.toString();
  }

  /**
   * Given a filename, strips the .extension.
   * 
   * @param filename
   *          filename to strip
   * @return String filename without .extension
   */
  public static String stripExtension(final String filename) {
    int index = filename.lastIndexOf('.');
    if (index > -1) {
      return filename.substring(0, index);
    }
    return filename;
  }

  /**
   * Performs variable substitution for a string. String is scanned for
   * ${variable_name} and if one is found, it is replaced with corresponding
   * value from the vars hashtable.
   * 
   * @param origString
   *          unmodified string
   * @param vars
   *          Hashtable of replacement values
   * @return modified string
   * @exception Exception
   */
  public static String replaceVars(final String origString, final Map<String, String> vars) {

    StringBuffer finalString = new StringBuffer();
    int index = 0;
    int i = 0;
    String key = null;
    String value = null;
    while ((index = origString.indexOf("${", i)) > -1) {
      key = origString.substring(index + 2, origString.indexOf("}", index + 3));
      value = vars.get(key);
      finalString.append(origString.substring(i, index));
      if (value != null) {
        finalString.append(value);
      } else {
        finalString.append("${" + key + "}");
      }
      i = index + 3 + key.length();
    }
    finalString.append(origString.substring(i));

    return finalString.toString();
  }

  /**
   * Check strSearchValue is in Array.
   * 
   * @param fieldProps
   *          Array
   * @param strSearchValue
   *          SearchString
   * @return integer Value of Index in Array
   */
  public static int isInArray(final String[] fieldProps, final String strSearchValue) {
    for (int i = 0; i < fieldProps.length; i++) {
      if (fieldProps[i].equals(strSearchValue)) {
        return i;
      }
    }
    return -1;
  }

  /**
   * Convert byte array to a mime coded string.
   * 
   * @param array
   *          the array to convert
   * @return mime coded String
   * @throws IOException
   *           if something goes wrong
   */
  public static String byteArrayToMimeString(final byte[] array) throws IOException {
    return Base64.encodeBytes(array);
  }

  /**
   * Convert a mimecoded string to a array of byte.
   * 
   * @param string
   *          the mime coded string
   * @return an array of byte
   * @throws Exception
   *           if something goes wrong
   */
  public static byte[] mimeStringTobyteArray(final String string) throws Exception {
    return Base64.decode(string);
  }

  /**
   * building the md5 hash of an String.
   * 
   * @param s
   *          String
   * @return String
   */
  public static String md5String(final String s) {
    // Build MD5-hashcode
    MessageDigest md;
    try {
      md = MessageDigest.getInstance("MD5");
      md.update(s.getBytes());
      byte[] digest = md.digest();

      return StringFormat.toHexString(digest);
    } catch (NoSuchAlgorithmException e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Method md5String.
   * 
   * @param s
   *          String
   * @return byte[[]
   */
  public static byte[] md5StringBytes(final String s) {
    // Build MD5-hashcode
    MessageDigest md;
    try {
      md = MessageDigest.getInstance("MD5");
      md.update(s.getBytes());
      byte[] digest = md.digest();

      return digest;
    } catch (NoSuchAlgorithmException e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * building the md5 hash of an String.
   * 
   * @param s
   *          String
   * @return String
   */
  public static String md5UTF8String(final String s) {
    // Build MD5-hashcode
    MessageDigest md;
    try {
      md = MessageDigest.getInstance("MD5");
      md.update(s.getBytes("UTF-8"));
      byte[] digest = md.digest();

      return StringFormat.toHexString(digest);
    } catch (NoSuchAlgorithmException e) {
      e.printStackTrace();
    } catch (UnsupportedEncodingException e) {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
    return null;
  }

  /**
   * xorArrays.
   * 
   * @param arg0
   *          byte[]
   * @param arg1
   *          byte[]
   * @return byte[]
   */
  public static byte[] xorArrays(final byte[] arg0, final byte[] arg1) {
    byte[] xor = new byte[arg0.length];
    int keylength = arg1.length;
    for (int i = 0; i < xor.length; i++) {
      xor[i] = (byte) (arg0[i] ^ arg1[i % keylength]);
    }
    return xor;
  }

  /**
   * Getting the value formatted to the actual fieldsize.
   * 
   * @param aValue
   *          value to format
   * @param size
   *          array index of this field
   * @return String
   */
  public static String formatFieldData(final String aValue, final int size) {
    StringBuffer value = new StringBuffer(aValue);
    while (value.length() < size) {
      value.append(SPACE);
    }
    if (value.length() > size) {
      value = new StringBuffer(value.substring(0, size));
    }
    return value.toString();
  }

  /**
   * converts a string array into a list of strings.
   * 
   * @param strings
   *          string array to convert to
   * @return a list.
   */
  public static List<String> toList(final String[] strings) {
    List<String> list = new ArrayList<String>(strings.length);
    for (int i = 0; i < strings.length; i++) {
      list.add(strings[i]);
    }
    return list;
  }

  /**
   * Converts a delimited string into a list of string tokens. Separator is ',',
   * delimiter is '"'
   * 
   * @param string
   *          The 'separator' separated string.
   * @return String[] A string array of the original tokens.
   */
  public static List<String> csvStringToList(final String string) {
    String[] strings = csvStringToArray(string, ',', '"');
    return toList(strings);
  }

  /**
   * Converts a delimited string into a list of string tokens. Separator and
   * delimiter could be used.
   * 
   * @param string
   *          The 'separator' separated string.
   * @param localseparator
   *          The separator char.
   * @param localdelimiter
   *          The delimiter char.
   * @return String[] A string array of the original tokens.
   */
  public static List<String> csvStringToList(final String string, final char localseparator, final char localdelimiter) {
    String[] strings = csvStringToArray(string, localseparator, localdelimiter);
    return toList(strings);
  }

  /**
   * converting a csvString in a property map.
   * 
   * @param csvString
   * @param csvSeparators
   * @param csvDelimiter
   * @return
   */
  public static Map<String, String> csvStringToPropMap(String csvString, char csvSeparators, char csvDelimiter,
      char propSeparator) {
    Map<String, String> map = new HashMap<String, String>();
    String[] propStrings = csvStringToArray(csvString, csvSeparators, csvDelimiter);
    for (String propString : propStrings) {
      int pos = propString.indexOf(propSeparator);
      if (pos >= 0) {
        String key = propString.substring(0, pos);
        String value = propString.substring(pos + 1);
        map.put(key, value);
      }
    }
    return map;
  }

  /**
   * converting a property map in a csvString .
   * 
   * @param map
   * @param csvSeparators
   * @param csvDelimiter
   * @param propSeparator
   * @return String
   */
  public static String propMapToCsvString(Map<String, String> map, char csvSeparators, char csvDelimiter,
      char propSeparator) {
    StringBuilder b = new StringBuilder();
    for (Iterator<Entry<String, String>> iterator = map.entrySet().iterator(); iterator.hasNext();) {
      Entry<String, String> entry = iterator.next();
      b.append(csvDelimiter);
      b.append(entry.getKey());
      b.append(propSeparator);
      b.append(entry.getValue());
      b.append(csvDelimiter);
      if (iterator.hasNext()) {
        b.append(csvSeparators);
      }
    }
    return b.toString();
  }

  static byte[] raw = new byte[] { 'T', 'h', 'i', 's', 'I', 's', 'A', 'S', 'e', 'c', 'r', 'e', 't', 'K', 'e', 'y' };

  public static String encrypt(String value, byte[] key) {
    try {
      SecretKeySpec skeySpec = new SecretKeySpec(key, "AES");
      Cipher cipher = Cipher.getInstance("AES/ECB/PKCS5PADDING");
      cipher.init(Cipher.ENCRYPT_MODE, skeySpec);
      byte[] encrypted = cipher.doFinal(value.getBytes());
      return Base64.encodeBytes(encrypted);
    } catch (Exception ex) {
      ex.printStackTrace();
    }
    return null;
  }

  public static String decrypt(String encrypted, byte[] key) {
    try {
      SecretKeySpec skeySpec = new SecretKeySpec(key, "AES");
      Cipher cipher = Cipher.getInstance("AES/ECB/PKCS5PADDING");
      cipher.init(Cipher.DECRYPT_MODE, skeySpec);
      byte[] original = cipher.doFinal(Base64.decode(encrypted));
      return new String(original);
    } catch (Exception ex) {
      ex.printStackTrace();
    }
    return null;
  }
}
