/*
 * MCS Media Computer Software Copyright (c) 2005 by MCS
 * -------------------------------------- Created on 16.01.2004 by w.klaas
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package de.mcs.utils;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * Implements missing String format functions as a function library.
 * 
 * @version $Revision: 1.1 $
 */
public final class StringFormat {

  /** Der MessageDigest. */
  private static MessageDigest standardMd;
  static {
    try {
      standardMd = MessageDigest.getInstance("MD5");
    } catch (NoSuchAlgorithmException e) {
      e.printStackTrace();
    }
  }

  /** i don't know. */
  private static final String SPECIALSAVECHARS = "\\";

  /** Die hexadezimalen Ziffern. */
  private static final char[] HEX_CHARS = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E',
      'F' };

  /** und nochmal als ein String. */
  private static final String HEX_STRING = "0123456789ABCDEF";

  /**
   * Hide the Constructor.
   */
  private StringFormat() {
  }

  /**
   * Converts a byte to hex digit and writes to the supplied buffer.
   * 
   * @param b
   *          byte to convert
   * @param buf
   *          StringBuffer to append to
   */
  public static void byte2hex(final byte b, final StringBuffer buf) {
    int high = ((b & 0xf0) >> 4);
    int low = (b & 0x0f);
    buf.append(HEX_CHARS[high]);
    buf.append(HEX_CHARS[low]);
  }

  /**
   * Converts a byte array to hex string.
   * 
   * @param block
   *          bytearray to convert
   * @return String Hexstring representing the block, constisting only from
   *         uppercase letters [0-9A-F]*
   */
  public static String toHexString(final byte[] block) {
    StringBuffer buf = new StringBuffer();

    int len = block.length;

    for (int i = 0; i < len; i++) {
      byte2hex(block[i], buf);

    }
    return buf.toString();
  }

  /**
   * Converts the given hex string to a byte array. The hex string is
   * interpreted as a sequence of hex digit pairs in high, low order. For odd
   * length strings the last digit is interpreted as low value.
   * 
   * @param str
   *          The hex string to convert to a byte array.
   * @return A byte array containing the byte values of the hex string. throws
   *         NumberFormatException if the given string isn't a hex string
   */
  public static byte[] fromHexString(final String str) {
    String hexStr = str.toUpperCase();
    final int len = hexStr.length();
    final int l = len / 2;
    byte[] bytes = new byte[l + (len % 2)];
    int i, n;
    for (i = 0, n = 0; i + 1 < len; i += 2, n++) {

      int low = HEX_STRING.indexOf(hexStr.charAt(i + 1));
      int high = HEX_STRING.indexOf(hexStr.charAt(i));
      if (-1 == low || -1 == high) {
        throw new NumberFormatException(hexStr);
      }
      bytes[n] = (byte) ((high << 4) | low);
    }
    if (len % 2 > 0) {
      int low = HEX_STRING.indexOf(hexStr.charAt(i));
      if (-1 == low) {
        throw new NumberFormatException(hexStr);
      }
      bytes[n] = (byte) low;
    }
    return bytes;
  }

  /**
   * Converts a string to its int equivalent.
   * 
   * @param strNum
   *          The string that should be converted
   * @return int The int version of the string value throws
   *         NumberFormatException if the given String isn't a textual
   *         representation of an int value.
   */
  public static int strToInt(final String strNum) {
    return Integer.parseInt(strNum);
  }

  /**
   * Returns a each 2 characters separated PathString from a MD5 HexString The
   * used separator is System.getProperty("file.separator").
   * 
   * @param string
   *          der String der gewandelt werden soll
   * @param directoryDepth
   *          Die Anzahl der Unterverzeichisse
   * @return String
   */
  public static String getPathFromString(final String string, final int directoryDepth) {
    int maxdirectoryDepth = (string.length() / 2);
    StringBuffer buf = new StringBuffer();
    if ((directoryDepth <= maxdirectoryDepth) && (directoryDepth >= 0)) {
      String filedelim = System.getProperty("file.separator");
      // Path
      for (int i = 0; i < (directoryDepth * 2); i++) {
        // File-Delimiter?
        if ((i > 0) && ((i % 2) == 0)) {
          buf.append(filedelim);
        }
        buf.append(string.charAt(i));
      }
      buf.append(filedelim);
    }
    return buf.toString();
  }

  /**
   * Format the String representation of the given long to the given length and
   * fills up at the beginning with the given fill value. l=654, length=5,
   * fill='0' will lead to '00654'. If the result of l is longer than length,
   * the full length of l is return
   * 
   * @param l
   *          the long to be converted
   * @param length
   *          the length of the result
   * @param fill
   *          the fill value of the result
   * @return the formatted string
   */
  public static String formatLong(final long l, final int length, final char fill) {
    String x = String.valueOf(l);
    if (x.length() < length) {
      StringBuffer z = new StringBuffer();
      for (int i = 0, s = length - x.length(); i < s; i++) {
        z.append(fill);
      }
      z.append(x);
      return z.toString();
    } else {
      return x;
    }
  }

  /**
   * Format the String representation of the given long to the given length and
   * fills up at the beginning with the given fill value. l=654, length=5,
   * fill='0' will lead to '00654'. If the result of l is longer than length,
   * the full length of l is return
   * 
   * @param x
   *          the long to be converted
   * @param length
   *          the length of the result
   * @param fill
   *          the fill value of the result
   * @return the formatted string
   */
  public static String formatStringLeading(final String x, final int length, final char fill) {

    if (x.length() < length) {
      StringBuffer z = new StringBuffer();
      for (int i = 0, s = length - x.length(); i < s; i++) {
        z.append(fill);
      }
      z.append(x);
      return z.toString();
    } else {
      if (x.length() == length) {
        return x;
      } else {
        return x.substring(0, length + 1);
      }
    }
  }

  /**
   * Format the String representation of the given long to the given length and
   * fills up at the beginning with the given fill value. l=654, length=5,
   * fill='0' will lead to '00654'. If the result of l is longer than length,
   * the full length of l is return
   * 
   * @param x
   *          the long to be converted
   * @param length
   *          the length of the result
   * @param fill
   *          the fill value of the result
   * @return the formatted string
   */
  public static String formatString(final String x, final int length, final char fill) {

    if (x.length() < length) {
      StringBuffer z = new StringBuffer(x);
      for (int i = 0, s = length - x.length(); i < s; i++) {
        z.append(fill);
      }
      return z.toString();
    } else {
      if (x.length() == length) {
        return x;
      } else {
        return x.substring(0, length + 1);
      }
    }
  }

  /**
   * Method md5String.
   * 
   * @param s
   *          String
   * @return String
   */
  public static String md5String(final String s) {
    // Build MD5-hashcode
    MessageDigest md;
    try {
      md = (MessageDigest) standardMd.clone();
      md.update(s.getBytes());
      byte[] digest = md.digest();

      return StringFormat.toHexString(digest);

    } catch (CloneNotSupportedException e) {
      // ignore, because cloning works
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Creates a well formed XML string.
   * 
   * @param g
   *          original string
   * @return a well formed string
   */
  public static String wellForm(final String g) {
    StringBuffer newString = new StringBuffer();
    for (int i = 0, s = g.length(); i < s; i++) {
      char t = g.charAt(i);
      switch (t) {
      case '&':
        newString.append("&amp;");
        break;
      case '>':
        newString.append("&gt;");
        break;
      case '<':
        newString.append("&lt;");
        break;
      case '"':
        newString.append("&quot;");
        break;
      case '\'':
        newString.append("&apos;");
        break;
      default:
        newString.append(t);
      }
    }
    return newString.toString();
  }

  /**
   * Creates a String that contains all element of the given collection,
   * seperated by the given seperator.
   * 
   * @param col
   *          a collection of objects
   * @param sep
   *          the seperator
   * @return result String
   */
  public static String getList(final Collection<Object> col, final char sep) {
    StringBuffer buf = new StringBuffer();
    boolean first = true;
    for (Iterator<Object> iter = col.iterator(); iter.hasNext();) {
      Object element = iter.next();
      if (!first) {
        buf.append(sep);
      }
      first = false;
      if (element.toString().indexOf(sep) != -1) {
        buf.append('"');
        buf.append(element.toString());
        buf.append('"');
      } else {
        buf.append(element.toString());
      }

    }
    return buf.toString();

  }

  /**
   * Parses the given String and returns a collection of String elements. If the
   * separator must be part of the String, the single string element must be
   * included in '"'
   * 
   * @param col
   *          a string with separated strings
   * @param sep
   *          a seperator
   * @return a collection of strings
   */
  public static List<String> parseList(final String col, final char sep) {
    ArrayList<String> q = new ArrayList<String>();
    StringBuffer element = new StringBuffer();
    boolean ignore = false;
    for (int i = 0; i < col.length(); i++) {
      char v = col.charAt(i);
      if (v == sep) {
        if (!ignore) {
          q.add(element.toString());
          element = new StringBuffer();
        } else {
          element.append(v);
        }
      } else {
        if (v == '"') {
          if (element.length() == 0) {
            ignore = true;
          } else {
            if (ignore) {
              ignore = false;
            } else {
              element.append(v);
            }
          }
        } else {
          element.append(v);
        }
      }
    }
    if (element.length() != 0) {
      q.add(element.toString());
    }

    return q;
  }

  /**
   * Reads a string and searches for every occurence of one of the characters in
   * <code>chars</code>. If the character occurs in <code>chars</code> it will
   * be added to the result. If none of the characters in <code>chars</code>
   * occurs in <code>source</code>, the method will return <code>null</code>.
   * 
   * @param source
   *          the source to check
   * @param chars
   *          a string that contains every character to check for.
   * @return the result of occurences of characters from <code>chars</code> in
   *         <code>source</code> or <code>null</code>.
   */
  public static String containsChars(final String source, final String chars) {
    StringBuffer errorChars = new StringBuffer();
    for (int i = 0, length = source.length(); i < length; i++) {
      char c = source.charAt(i);
      if (chars.indexOf(c) != -1) {
        if (errorChars.toString().indexOf(c) == -1) {
          errorChars.append(c);
        }
      }
    }
    if (errorChars.length() > 0) {
      return errorChars.toString();
    } else {
      return null;
    }
  }

  /**
   * Anzahl derr zeichen z�hlen.
   * 
   * @param searchString
   *          the string to count chars
   * @param character
   *          the character to count
   * @return the count of chars
   */
  public static int countsChars(final String searchString, final char character) {
    int count = 0;
    for (int i = 0, s = searchString.length(); i < s; i++) {
      if (searchString.charAt(i) == character) {
        count++;
      }
    }
    return count;
  }

  /**
   * @param r
   *          a colletion that contains strings
   * @return a String array
   */
  public static String[] fromCollection(final Collection<Object> r) {
    String[] values = new String[r.size()];
    int i = 0;
    for (Iterator<Object> iter = r.iterator(); iter.hasNext();) {
      String element = iter.next().toString();
      values[i++] = element;
    }
    return values;
  }

  /**
   * Removes white spaces (>=ascii 32) from a string and all spaces at the end.
   * 
   * @param s
   *          the input string
   * @return a string without whitespaces
   */
  public static String filterWhiteSpace(final String s) {
    StringBuffer result = new StringBuffer();
    String sResult = "";
    for (int i = 0; i < s.length(); i++) {
      char x = s.charAt(i);
      if (x > 31) {
        result.append(x);
      } else {
        result.append(' ');
      }
    }
    sResult = result.toString();
    while (sResult.endsWith(" ")) {
      sResult = sResult.substring(0, sResult.length() - 1);
    }
    return sResult;
  }

  /**
   * Method replaceString.
   * 
   * @param newString
   *          String
   * @param source
   *          String
   * @param searchStr
   *          String
   * @return String
   */
  public static String replaceString(final String newString, final String source, final String searchStr) {
    int position;
    String sResult = source;
    while ((position = source.indexOf(searchStr)) != -1) {
      String start;
      String end;
      if (position > 0) {
        start = source.substring(0, position);
      } else {
        start = "";
      }
      if (position + searchStr.length() - 1 < source.length() - 1) {
        end = source.substring(position + searchStr.length(), source.length());
      } else {
        end = "";
      }
      sResult = start + newString + end;
    }
    return sResult;
  }

  /**
   * Returns the value of a property. The given collection should contain
   * strings with "name=value" entries. The method looks for the name and
   * returns the value, else returns the default value
   * 
   * @param property
   *          ???
   * @param defaultValue
   *          ???
   * @param col
   *          ???
   * @return the value of the property or the default
   */
  public static String getProperty(final String property, final String defaultValue, final Collection<String> col) {
    String value = defaultValue;
    for (Iterator<String> iter = col.iterator(); iter.hasNext();) {
      String element = iter.next();
      int i = element.indexOf('=');
      if (i != -1) {
        String prop = element.substring(0, i).trim();
        if (prop.equalsIgnoreCase(property)) {
          value = element.substring(i + 1).trim();
          if (value.startsWith("\"") && value.endsWith("\"")) {
            value = value.substring(1, value.length() - 1);
          }
          break;
        }
      }
    }
    return value;
  }

  /**
   * Gets a Map of properties from a collection that contains strings like
   * "prop=value".
   * 
   * @param col
   *          the properties
   * @return a map
   */
  public static Map<String, String> getProperties(final Collection<String> col) {
    HashMap<String, String> props = new HashMap<String, String>();
    for (Iterator<String> iter = col.iterator(); iter.hasNext();) {
      String element = (String) iter.next();
      String prop;
      String value;
      int i = element.indexOf('=');
      if (i != -1) {
        prop = element.substring(0, i).trim();
        value = element.substring(i + 1);
        props.put(prop, value);
      }
    }
    return props;
  }

  /**
   * Creates a property string ( a list of comma separated key=value items) from
   * a HashMap.
   * 
   * @param props
   *          the properties
   * @return a string that contains the key=value list
   */
  public static String getPropertyString(final Map<String, String> props) {
    StringBuffer result = new StringBuffer();
    for (Iterator<String> iter = props.keySet().iterator(); iter.hasNext();) {
      String element = iter.next();
      String value = element + "=" + props.get(element);
      if (value.indexOf(",") != -1) {
        value = '"' + value + '"';
      }
      if (result.length() > 0) {
        result.append(',');
      }
      result.append(value);
    }
    return result.toString();
  }

  /**
   * Sets a property in a property string list.
   * 
   * @param property
   *          the name of the property
   * @param value
   *          the value
   * @param propertyString
   *          the property string list
   * @return the result
   */
  public static String setProperty(final String property, final String value, final String propertyString) {
    Map<String, String> m = getProperties(StringFormat.parseList(propertyString, ','));
    m.put(property, value);
    return getPropertyString(m);
  }

  /**
   * Holt aus einer Stringliste (mit CRLF getrennt) eine Collection von Strings.
   * 
   * @param input
   *          EingangsString
   * @return Collection
   */
  public static Collection<String> getMultiLineString(final String input) {
    ArrayList<String> lines = new ArrayList<String>();
    StringBuffer actual = new StringBuffer();
    for (int i = 0, length = input.length(); i < length; i++) {

      char ch = input.charAt(i);
      if (ch == 0x0a || ch == 0x0d) {
        if (actual.length() != 0) {
          lines.add(actual.toString());
        }
        actual = new StringBuffer();
      } else {
        actual.append(ch);
      }
    }
    if (actual.length() != 0) {
      lines.add(actual.toString());
    }
    return lines;
  }

  /**
   * filters a string from the given characters. Example: filterString
   * ("{blabla}", " {", "}") returns blabla
   * 
   * @param input
   *          input String
   * @param left
   *          linke Begrenzung
   * @param right
   *          rechte Begrenzung
   * @return the_filtered_string
   */
  public static String filterString(final String input, final char left, final char right) {
    String sReturn = input;
    int iLeft = input.indexOf(left);
    if (iLeft > -1) {
      sReturn = input.substring(iLeft + 1);
    }
    int iRight = sReturn.indexOf(right);
    if (iRight > -1) {
      sReturn = sReturn.substring(0, iRight);
    }
    return sReturn;
  }

  /**
   * filters a string from the given characters. Example: filterString
   * ("'blabla'", " '") returns blabla
   * 
   * @param input
   *          zu filternder String
   * @param filterchar
   *          begrenzungszeichen
   * @return the_filtered_string
   */

  public static String filterString(final String input, final char filterchar) {
    String sReturn = input;

    while (sReturn.startsWith("" + filterchar)) {
      sReturn = sReturn.substring(1);
    }
    while (sReturn.endsWith("" + filterchar)) {
      sReturn = sReturn.substring(0, sReturn.length() - 1);
    }
    return sReturn;
  }

  /**
   * rightrim.
   * 
   * @param s
   *          inputstring
   * @return string
   */
  public static String rtrim(final String s) {
    int firstx = 0;
    for (int i = s.length() - 1; i >= 0; i--) {
      if (s.charAt(i) > 32) {
        break;
      } else {
        firstx = i;

      }
    }
    if (firstx == 0) {
      return "";
    } else {
      return s.substring(0, firstx);
    }
  }

  /**
   * Converts some special characters (tab, CR, LF, 0 etc) from java escape
   * sequences to characters.
   * 
   * @param s
   *          the string to convert.
   * @return the converted string
   */
  public static String convertEscapes(final String s) {
    StringBuffer r = new StringBuffer();
    StringBuffer hexBuffer = null;
    boolean code = false;
    int hexCount = 0;
    for (int i = 0, length = s.length(); i < length; i++) {
      char c = s.charAt(i);
      if (c == '\\') {
        code = true;
      } else {
        if (code) {
          switch (c) {
          case '0':
            hexBuffer = new StringBuffer();
            hexCount = -1;
            code = false;
            break;
          case 'r':
            r.append('\r');
            code = false;
            break;
          case 'f':
            r.append('\f');
            code = false;
            break;
          case 'n':
            r.append('\n');
            code = false;
            break;
          case 't':
            r.append('\t');
            code = false;
            break;
          case '\\':
            r.append('\\');
            code = false;
            break;
          default:
          }
        } else {
          if (hexCount != 0) {
            if (hexCount == -1) {
              if (c == 'x') {
                hexCount = 2;
              } else if (c == 'u') {
                hexCount = 4;
              } else {
                hexCount = 0;
              }
            } else {
              hexBuffer.append(c);
              hexCount--;
              if (hexCount == 0) {
                r.append((char) Integer.parseInt(hexBuffer.toString(), 16));
              }
            }

          } else {
            r.append(c);
          }
        }
      }
    }
    return r.toString();

  }

  /**
   * converting a string so that it can be saved into a human readable file.
   * 
   * @param theString
   *          the string to convert
   * @param escapeSpace
   *          spaces must be escaped.
   * @return the converted String.
   */
  public static String saveConvert(final String theString, final boolean escapeSpace) {
    int len = theString.length();
    StringBuffer outBuffer = new StringBuffer(len * 2);

    for (int x = 0; x < len; x++) {
      char aChar = theString.charAt(x);
      switch (aChar) {
      case ' ':
        if (x == 0 || escapeSpace) {
          outBuffer.append('\\');
        }

        outBuffer.append(' ');
        break;
      case '\\':
        outBuffer.append('\\');
        outBuffer.append('\\');
        break;
      case '\t':
        outBuffer.append('\\');
        outBuffer.append('t');
        break;
      case '\n':
        outBuffer.append('\\');
        outBuffer.append('n');
        break;
      case '\r':
        outBuffer.append('\\');
        outBuffer.append('r');
        break;
      case '\f':
        outBuffer.append('\\');
        outBuffer.append('f');
        break;
      default:
        // if ((aChar < 0x0020) || (aChar > 0x007e)) {
        // outBuffer.append('\\');
        // outBuffer.append('u');
        // outBuffer.append(toHex((aChar >> 12) & 0xF));
        // outBuffer.append(toHex((aChar >> 8) & 0xF));
        // outBuffer.append(toHex((aChar >> 4) & 0xF));
        // outBuffer.append(toHex(aChar & 0xF));
        // } else {
        if (SPECIALSAVECHARS.indexOf(aChar) != -1) {
          outBuffer.append('\\');
        }
        outBuffer.append(aChar);
        // }
      }
    }
    return outBuffer.toString();
  }

  /**
   * Convert a nibble to a hex character.
   * 
   * @param nibble
   *          the nibble to convert.
   * @return char
   */
  @SuppressWarnings("unused")
  private static char toHex(final int nibble) {
    return HEXDIGIT[(nibble & 0xF)];
  }

  /** A table of hex digits. */
  private static final char[] HEXDIGIT = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E',
      'F' };

  /**
   * getting a stack trace as a string like in exception.printStackTrace.
   * 
   * @param cause
   *          the exception to convert.
   * @return String
   */
  public static String getStackTrace(final Throwable cause) {
    StackTraceElement[] elements = cause.getStackTrace();
    StringBuffer buffer = new StringBuffer(1024);
    buffer.append(cause);
    for (int i = 0; i < elements.length; i++) {
      StackTraceElement element = elements[i];
      buffer.append("\r\n\tat ");
      buffer.append(element.toString());
    }
    return buffer.toString();
  }

  public static String left(final String string, final int pos) {
    return string.substring(0, pos);
  }

  public static String right(final String string, final int pos) {
    return string.substring(pos);
  }

  public static String[] splitFirst(final String string, final String search) {
    String[] result = new String[2];
    int pos = string.indexOf(search);
    if (pos == -1) {
      result[0] = string;
    } else {
      result[0] = left(string, pos);
      if (pos < string.length()) {
        result[1] = right(string, pos + search.length());
      }
    }
    return result;
  }

  public static boolean isUppercase(final String string) {
    /* Now check if there are any characters that need to be changed. */
    for (int index = 0; index < string.length(); index++) {
      if (!Character.isUpperCase(string.codePointAt(index))) {
        return false;
      }
    }
    return true;
  }
}
