/*
 * MCS Media Computer Software Copyright (c) 2005 by MCS
 * -------------------------------------- Created on 23.04.2005 by w.klaas
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package de.mcs.utils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.util.Arrays;

/**
 * This class contains some helper emthode for streaming operations.
 * 
 * @author W.Klaas
 */
public final class StreamHelper {

  /** size of the internal buffer. */
  private static final int BUFFER_SIZE = 32768;

  /** no construction . */
  private StreamHelper() {
  }

  /**
   * copy all data from in to out.
   * 
   * @param out
   *          outputstream to copy to
   * @param in
   *          input stream to copy from
   * @throws IOException
   *           if something goes wrong
   */
  public static long copyStream(final InputStream in, final OutputStream out) throws IOException {
    long written = 0;
    byte[] buffer = new byte[BUFFER_SIZE];
    int k = 0;
    do {
      k = in.read(buffer);
      if (k > 0) {
        out.write(buffer, 0, k);
        written += k;
      }
    } while (k >= 0);
    return written;
  }

  /**
   * copy all data from in to out.
   * 
   * @param out
   *          outputstream to copy to
   * @param in
   *          input stream to copy from
   * @param bufSize
   *          the size of the buffer to use.
   * @throws IOException
   *           if something goes wrong
   */
  public static long copyStream(final InputStream in, final OutputStream out, final int bufSize) throws IOException {
    long written = 0;
    byte[] buffer = new byte[bufSize];
    int k = 0;
    do {
      k = in.read(buffer);
      if (k > 0) {
        out.write(buffer, 0, k);
        written += k;
      }
    } while (k >= 0);
    return written;
  }

  /**
   * compares to stream
   * 
   * @param in
   * 
   * @param in2
   * 
   * @throws IOException
   *           if something goes wrong
   */
  public static boolean equalStreams(final InputStream in, final InputStream in2) throws IOException {
    boolean ok = true;
    byte[] buffer = new byte[BUFFER_SIZE];
    byte[] buffer2 = new byte[BUFFER_SIZE];
    int k = 0;
    int k2 = 0;
    do {
      k = in.read(buffer);
      k2 = in2.read(buffer2);
      ok = ok && (k == k2);
      if (k > 0) {
        ok = ok && Arrays.equals(buffer, buffer2);
      }
    } while (k >= 0);
    return ok;
  }

  /**
   * printing the file to System.out.
   * 
   * @param rcPropFile
   *          file to print
   * @throws IOException
   *           if something goes wrong
   */
  public static void printOutFile(final File rcPropFile) throws IOException {
    FileInputStream fileInputStream = new FileInputStream(rcPropFile);
    Reader reader = new InputStreamReader(fileInputStream);
    BufferedReader bufferedReader = new BufferedReader(reader);
    try {
      String line = null;
      do {
        line = bufferedReader.readLine();
        System.out.println(line);
      } while (null != line);
    } finally {
      bufferedReader.close();
    }
  }

  /**
   * Convert input stream to string.
   *
   * @param inputStream
   *          the input stream
   * @return the string
   * @throws IOException
   *           Signals that an I/O exception has occurred.
   */
  public static String convertInputStreamToString(InputStream inputStream) throws IOException {
    if (inputStream != null) {
      StringBuilder stringBuilder = new StringBuilder();
      String line;

      try {
        BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream, "UTF-8"));
        while ((line = reader.readLine()) != null) {
          stringBuilder.append(line).append("\n");
        }
      } finally {
        inputStream.close();
      }

      return stringBuilder.toString();
    } else {
      return null;
    }
  }
}
