/*
 * MCS Media Computer Software Copyright (c) 2005 by MCS
 * -------------------------------------- Created on 23.04.2005 by w.klaas
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package de.mcs.utils;

import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.SoftReference;
import java.util.AbstractMap;
import java.util.AbstractSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;

/**
 * Implements a Map with only soft references backuped by a hashmap.
 * 
 * @author w.klaas
 */

public class SoftHashMap<K, V> extends AbstractMap<K, V> {
  /** the map to store. */
  private Map<K, Reference<V>> m = null;

  /** the reference queue. */
  private ReferenceQueue q = new ReferenceQueue();

  /**
   * Constructs an empty <tt>HashMap</tt> with the default initial capacity (16)
   * and the default load factor (0.75).
   */
  public SoftHashMap() {
    m = new HashMap<K, Reference<V>>();
  }

  /**
   * Constructs an empty <tt>HashMap</tt> with the specified initial capacity
   * and the default load factor (0.75).
   * 
   * @param initialCapacity
   *          the initial capacity.
   */
  public SoftHashMap(final int initialCapacity) {
    m = new HashMap<K, Reference<V>>(initialCapacity);
  }

  /**
   * Constructs an empty <tt>HashMap</tt> with the specified initial capacity
   * and load factor.
   * 
   * @param initialCapacity
   *          The initial capacity.
   * @param loadFactor
   *          The load factor.
   */
  public SoftHashMap(final int initialCapacity, final float loadFactor) {
    m = new HashMap<K, Reference<V>>(initialCapacity, loadFactor);
  }

  /**
   * Returns the value to which this map maps the specified key. Returns
   * <tt>null</tt> if the map contains no mapping for this key. A return value
   * of <tt>null</tt> does not <i>necessarily</i> indicate that the map contains
   * no mapping for the key; it's also possible that the map explicitly maps the
   * key to <tt>null</tt>. The <tt>containsKey</tt> operation may be used to
   * distinguish these two cases.
   * 
   * <p>
   * More formally, if this map contains a mapping from a key <tt>k</tt> to a
   * value <tt>v</tt> such that <tt>(key==null ? k==null :
   * key.equals(k))</tt>, then this method returns <tt>v</tt>; otherwise it
   * returns <tt>null</tt>. (There can be at most one such mapping.)
   * 
   * @param key
   *          key whose associated value is to be returned.
   * @return the value to which this map maps the specified key, or
   *         <tt>null</tt> if the map contains no mapping for this key.
   * 
   * @see #containsKey(Object)
   */
  public final V get(final Object key) {
    Reference<V> res = m.get(key);
    return (res == null ? null : res.get());
  }

  /**
   * @see Map#put(K, V)
   */
  public final V put(final K key, final V value) {
    processQueue();
    Reference<V> ref = new SoftEntry(key, value, q);
    Reference<V> res = m.put(key, ref);
    return res == null ? null : res.get();
  }

  /** the entry set. */
  private Set entrySet = null;

  /**
   * getting the entry set.
   * 
   * @return Set
   */
  public final Set entrySet() {
    if (entrySet == null) {
      entrySet = new EntrySet();
    }
    return entrySet;
  }

  /**
   * processing the softentry queue.
   * 
   */
  private void processQueue() {
    Reference r;
    while ((r = q.poll()) != null) {
      SoftEntry e = (SoftEntry) r;
      m.remove(e.key);
    }
  }

  /**
   * @see java.util.Set#size()
   */
  public final int size() {
    return entrySet().size();
  }

  /**
   * @see java.util.Set#remove(java.lang.Object)
   */
  public final V remove(final Object key) {
    processQueue();
    Reference<V> res = m.remove(key);
    return res == null ? null : res.get();
  }

  /**
   * @see java.util.Map#clear()
   */
  public final void clear() {
    processQueue();
    m.clear();
  }

  /**
   * this is the soft reference entry in the map.
   * 
   * @author w.klaas
   * 
   */
  private static final class SoftEntry<K, T> extends SoftReference<T> {
    /** neccessary so that freed objects can be removed. */
    private K key;

    /**
     * private contructor to this entry.
     * 
     * @param aKey
     *          the key for this entry
     * @param value
     *          the value
     * @param q
     *          the reference queue
     */
    private SoftEntry(final K aKey, final T value, final ReferenceQueue q) {
      super(value, q);
      this.key = aKey;
    }
  }

  /**
   * My set of entries.
   * 
   * @author w.klaas
   * 
   */
  private class EntrySet<E> extends AbstractSet<E> {
    /** the entry set to use. */
    private Set entrySet = m.entrySet();

    /**
     * @see java.util.Set#size()
     */
    public int size() {
      int s = 0;
      for (Iterator<E> i = iterator(); i.hasNext(); i.next()) {
        s++;
      }
      return s;
    }

    /**
     * @see java.util.Set#isEmpty()
     */
    public boolean isEmpty() {
      return !(iterator().hasNext());
    }

    /**
     * @see java.util.Set#remove(java.lang.Object)
     */
    public boolean remove(final Object o) {
      processQueue();
      return super.remove(o);
    }

    /**
     * @see java.util.Set#iterator()
     */
    public Iterator<E> iterator() {

      return new Iterator() {
        private Iterator it = entrySet.iterator();

        private Entry next = null;

        private Object value = null;

        /*
         * Strong reference to key, so that the GC will leave it alone as long
         * as this Entry exists
         */

        public boolean hasNext() {
          while (it.hasNext()) {
            final Entry e = (Entry) it.next();
            SoftEntry se = (SoftEntry) e.getValue();
            value = null;
            if ((se != null) && ((value = se.get()) == null)) {
              /* Weak key has been cleared by GC */
              continue;
            }
            next = new Map.Entry() {
              public Object getKey() {
                return e.getKey();
              }

              public Object getValue() {
                return value;
              }

              public Object setValue(final Object v) {
                Object res = value;
                value = v;
                e.setValue(new SoftEntry(e.getKey(), value, q));
                return res;
              }

              public boolean equals(final Object x) {
                if (!(x instanceof Map.Entry)) {
                  return false;
                }
                Map.Entry e = (Map.Entry) x;
                Object key = getKey();
                return key == null ? e.getKey() == null
                    : key.equals(e.getKey()) && value == null ? e.getValue() == null : value.equals(e.getValue());
              }

              public int hashCode() {
                Object key = getKey();
                return (((key == null) ? 0 : key.hashCode()) ^ ((value == null) ? 0 : value.hashCode()));
              }

            };
            return true;
          }
          return false;
        }

        public Object next() {
          if ((next == null) && !hasNext()) {
            throw new NoSuchElementException();
          }
          Entry e = next;
          next = null;
          return e;
        }

        public void remove() {
          it.remove();
        }

      };
    }
  }
}
