/*
 * MCS Media Computer Software Copyright (c) 2005 by MCS
 * -------------------------------------- Created on 16.01.2004 by w.klaas
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package de.mcs.utils;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.Properties;
import java.util.SortedSet;
import java.util.TreeSet;

/**
 * Some helper functions for properties like getting all keys with a regex.
 * 
 * @author w.klaas
 * 
 */
public class PropertiesHelper extends Properties {

  /**
   * 
   */
  private static final long serialVersionUID = -3239128704806211452L;

  /**
   * Searching for all Keys with the regex string.
   * 
   * @param regex
   *            regular expression for the keys to find
   * @return String[] array with all found keys
   */
  @SuppressWarnings("rawtypes")
  public final String[] getKeys(final String regex) {
    ArrayList<String> list = new ArrayList<String>();
    for (Iterator iter = keySet().iterator(); iter.hasNext();) {
      String key = (String) iter.next();
      if (key.matches(regex)) {
        list.add(key);
      }
    }
    return (String[]) list.toArray(new String[0]);
  }

  /**
   * Extracting all properties into a new Propertie class with begin with the
   * prefix keyPrefix.
   * 
   * @param keyPrefix
   *            the prefix to use.
   * @return Properties
   */
  public final Properties extractProperties(final String keyPrefix) {
    Properties mediaProps = new Properties();
    String[] mediaKeyNames = getKeys(keyPrefix + ".*");
    for (int j = 0; j < mediaKeyNames.length; j++) {
      String key = mediaKeyNames[j];
      String value = getProperty(key);
      key = key.substring(key.indexOf(keyPrefix + ".") + (keyPrefix + ".").length());
      mediaProps.setProperty(key, value);
    }
    return mediaProps;
  }

  /**
   * Storing a properties class in a Stream in sorted manner.
   * 
   * @param out
   *            the output stream
   * @param comments
   *            the comments header.
   * @throws IOException
   *             if something goes wrong.
   */
  @SuppressWarnings("rawtypes")
  public final void storeSorted(final OutputStream out, final String comments) throws IOException {

    BufferedWriter awriter;
    awriter = new BufferedWriter(new OutputStreamWriter(out, "8859_1"));
    if (comments != null) {
      writeln(awriter, "#" + comments);
    }
    writeln(awriter, "#" + new Date().toString());
    SortedSet<Object> mySet = new TreeSet<Object>(new Comparator<Object>() {

      public int compare(final Object o1, final Object o2) {
        if ((o1 instanceof String) && (o2 instanceof String)) {
          String s1 = (String) o1;
          String s2 = (String) o2;
          return s1.compareTo(s2);
        }
        return 0;
      }

    });
    mySet.addAll(keySet());
    for (Iterator iter = mySet.iterator(); iter.hasNext();) {
      String key = (String) iter.next();
      String val = (String) get(key);
      key = saveConvert(key, true);

      /*
       * No need to escape embedded and trailing spaces for value, hence
       * pass false to flag.
       */
      val = saveConvert(val, false);
      writeln(awriter, key + "=" + val);
    }
    awriter.flush();
  }

  /**
   * write aa string and a newLine after.
   * 
   * @param bw
   *            where to write.
   * @param s
   *            what to write.
   * @throws IOException
   *             if something goes wrong.
   */
  private static void writeln(final BufferedWriter bw, final String s) throws IOException {
    bw.write(s);
    bw.newLine();
  }

  /**
   * Converts unicodes to encoded &#92;uxxxx and escapes special characters
   * with a preceding slash.
   * 
   * @param theString
   *            the string to convert.
   * @param escapeSpace
   *            escape the spaces too.
   * @return the converted String.
   */
  private String saveConvert(final String theString, final boolean escapeSpace) {
    int len = theString.length();
    int bufLen = len * 2;
    if (bufLen < 0) {
      bufLen = Integer.MAX_VALUE;
    }
    StringBuffer outBuffer = new StringBuffer(bufLen);

    for (int x = 0; x < len; x++) {
      char aChar = theString.charAt(x);
      // Handle common case first, selecting largest block that
      // avoids the specials below
      if ((aChar > 61) && (aChar < 127)) {
        if (aChar == '\\') {
          outBuffer.append('\\');
          outBuffer.append('\\');
          continue;
        }
        outBuffer.append(aChar);
        continue;
      }
      switch (aChar) {
      case ' ':
        if (x == 0 || escapeSpace) {
          outBuffer.append('\\');
        }
        outBuffer.append(' ');
        break;
      case '\t':
        outBuffer.append('\\');
        outBuffer.append('t');
        break;
      case '\n':
        outBuffer.append('\\');
        outBuffer.append('n');
        break;
      case '\r':
        outBuffer.append('\\');
        outBuffer.append('r');
        break;
      case '\f':
        outBuffer.append('\\');
        outBuffer.append('f');
        break;
      case '=': // Fall through
      case ':': // Fall through
      case '#': // Fall through
      case '!':
        outBuffer.append('\\');
        outBuffer.append(aChar);
        break;
      default:
        if ((aChar < 0x0020) || (aChar > 0x007e)) {
          outBuffer.append('\\');
          outBuffer.append('u');
          outBuffer.append(toHex((aChar >> 12) & 0xF));
          outBuffer.append(toHex((aChar >> 8) & 0xF));
          outBuffer.append(toHex((aChar >> 4) & 0xF));
          outBuffer.append(toHex(aChar & 0xF));
        } else {
          outBuffer.append(aChar);
        }
      }
    }
    return outBuffer.toString();
  }

  /**
   * Convert a nibble to a hex character.
   * 
   * @param nibble
   *            the nibble to convert.
   * @return char
   */
  private static char toHex(final int nibble) {
    return HEXDIGIT[(nibble & 0xF)];
  }

  /** A table of hex digits. */
  private static final char[] HEXDIGIT = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E',
      'F' };

}
