/*
 * MCS Media Computer Software Copyright (c) 2005 by MCS
 * -------------------------------------- Created on 16.01.2004 by w.klaas
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package de.mcs.utils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * FolderZiper provide a static method to zip a folder.
 * 
 * @author pitchoun
 */
public final class FolderZipper {

  /** prevent instancing. */
  private FolderZipper() {
  }

  /**
   * Zip the srcFolder into the destFileZipFile. All the folder subtree of the
   * src folder is added to the destZipFile archive. TODO handle the usecase of
   * srcFolder being en file.
   * 
   * @param srcFolder
   *          String, the path of the srcFolder
   * @param destZipFile
   *          String, the path of the destination zipFile. This file will be
   *          created or erased.
   */
  public static void zipFolder(final String srcFolder, final String destZipFile) {
    ZipOutputStream zip = null;
    FileOutputStream fileWriter = null;
    try {
      File file = new File(destZipFile);
      if (!file.getParentFile().exists()) {
        file.getParentFile().mkdirs();
      }
      fileWriter = new FileOutputStream(destZipFile);
      zip = new ZipOutputStream(fileWriter);
    } catch (Exception ex) {
      ex.printStackTrace();
    }

    addFolderToZip("", srcFolder, zip);
    try {
      zip.flush();
      zip.close();
    } catch (Exception ex) {
      ex.printStackTrace();
    }
  }

  /**
   * Write the content of srcFile in a new ZipEntry, named path+srcFile, of the
   * zip stream. The result is that the srcFile will be in the path folder in
   * the generated archive.
   * 
   * @param path
   *          String, the relatif path with the root archive.
   * @param srcName
   *          String, the absolute path of the file to add
   * @param zip
   *          ZipOutputStram, the stream to use to write the given file.
   */
  private static void addToZip(final String path, final String srcName, final ZipOutputStream zip) {

    File folder = new File(srcName);
    if (folder.isDirectory()) {
      if (path.equals("")) {
        addFolderToZip(folder.getName(), srcName, zip);
      } else {
        addFolderToZip(path + "/" + folder.getName(), srcName, zip);
      }
    } else {
      // Transfer bytes from in to out
      byte[] buf = new byte[1024];
      int len;
      try {
        File srcFile = new File(srcName);
        FileInputStream in = new FileInputStream(srcName);
        try {
          String zipEntryPrefix = "";
          if (!path.equals("")) {
            zipEntryPrefix = path + "/";
          }
          ZipEntry zipEntry = new ZipEntry(zipEntryPrefix + folder.getName());
          zipEntry.setTime(srcFile.lastModified());
          zip.putNextEntry(zipEntry);
          while ((len = in.read(buf)) > 0) {
            zip.write(buf, 0, len);
          }
        } finally {
          in.close();
        }
      } catch (Exception ex) {
        ex.printStackTrace();
      }
    }
  }

  /**
   * add the srcFolder to the zip stream.
   * 
   * @param path
   *          String, the relatif path with the root archive.
   * @param srcFolder
   *          String, the absolute path of the file to add
   * @param zip
   *          ZipOutputStram, the stream to use to write the given file.
   */
  private static void addFolderToZip(final String path, final String srcFolder, final ZipOutputStream zip) {
    File folder = new File(srcFolder);
    String[] fileListe = folder.list();
    for (String string : fileListe) {
      addToZip(path, srcFolder + "/" + string, zip);
    }
  }
}
