/*
 * MCS Media Computer Software Copyright (c) 2005 by MCS
 * -------------------------------------- Created on 16.01.2004 by w.klaas
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package de.mcs.utils;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.nio.channels.FileChannel;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

/**
 * Die Klasse Files definiert verschiedene statische Hilfsmethoden.
 * 
 * @author w.klaas
 */
public final class Files {
  /**
   * Damit man diese Klasse nicht instanzieren kann.
   */
  private Files() {
    // nothing to do here
  }

  /**
   * Diese Methode dient der Erzeugung einer tempor�ren Datei in einem
   * definierten Verzeichnis.
   * 
   * @param pre
   *          Prefix der Datei (unter WIN nur 3 Buchstaben m�glich)
   * @param suf
   *          Suffix der Datei (Dateiendung)
   * @param dir
   *          Verzeichniss in dem die Datei erzeugt werden soll
   * @return Verzeichnis und Dateiname der erzeugten Datei
   * @throws IOException
   *           wenn mal was schief geht
   */
  public static String getTempFileName(final String pre, final String suf, final String dir) throws IOException {
    String filename;

    File f = File.createTempFile(pre, suf, new File(dir));

    filename = f.getAbsolutePath();

    return filename;
  }

  /**
   * @return String getting the actual temp path
   * @throws IOException
   *           if something goes wrong
   */
  public static String getTempPath() {
    String strTempPath = null;
    strTempPath = System.getProperty("java.io.tmpdir");
    if (!strTempPath.endsWith(File.separator)) {
      return strTempPath + File.separator;
    } else {
      return strTempPath;
    }
  }

  public static File createTempDirectory(File parentPath) throws IOException {
    int count = 0;
    File folder = null;
    do {
      String foldername = new DecimalFormat("0000").format(count);
      folder = new File(parentPath, foldername);
      count++;
    } while (folder.exists());

    if (!(folder.mkdir())) {
      throw new IOException("Could not create temp directory: " + folder.getAbsolutePath());
    }

    return (folder);
  }

  /**
   * Ersetzung von ung�ltigen zeichen f�r den Blobnamen.
   * 
   * @param filename
   *          Eingangsname
   * @return String
   */
  public static String formatFileNameToBlobName(final String filename) {
    String sFilename = filename;
    sFilename = sFilename.replaceAll("[^0-9a-zA-Z]", "_");
    sFilename = sFilename.replace('.', '_');
    return sFilename;
  }

  /**
   * Von der (vorhandenen) Datei den MD5 berechnen.
   * 
   * @param file
   *          Dateiname
   * @return MD5 String
   */
  @Deprecated
  public static String computeMD5FromFile(final String file) {
    return computeMD5FromFile(new File(file));
  }

  public static byte[] computeMD5BytesFromFile(final File file) {
    InputStream in = null;
    MessageDigest digest = null;
    // Stream oeffnen
    try {
      in = new BufferedInputStream(new FileInputStream(file));

      // MessageDigest-Objekt fuer MD5 erzeugen
      try {
        digest = MessageDigest.getInstance("MD5");

        // Puffer
        byte[] array = new byte[8 * 4096];

        int length = 0;

        // Blockweise lesen und Pruefsumme auffrischen
        try {
          while (length >= 0) {
            length = in.read(array);
            if (length > 0) {
              digest.update(array, 0, length);
            }
          }
          in.close();

          // Pruefsumme berechnen
          return digest.digest();
        } catch (IOException e) {
          System.err.println("error reading " + file + "(" + e.getMessage() + ")");
        }
      } catch (NoSuchAlgorithmException e) {
        System.err.println("md5 not implemented");
      }
    } catch (FileNotFoundException e) {
      System.err.println("File " + file + " not found");
    }
    return null;
  }

  /**
   * Von der (vorhandenen) Datei den MD5 berechnen.
   * 
   * @param file
   *          Dateiname
   * @return MD5 String
   */
  public static String computeMD5FromFile(final File file) {
    String s;
    String sMD5 = "";
    byte[] md5 = computeMD5BytesFromFile(file);
    // Pruefsumme als Hex-Zahl ausgeben
    for (int l = 0; l < md5.length; l++) {
      // System.out.println(md5[l]);
      int i;
      if (md5[l] >= 0) {
        i = md5[l];
      } else {
        i = 256 + md5[l];
      }
      s = Integer.toHexString(i);
      while (s.length() < 2) {
        s = "0" + s;
      }
      sMD5 += s;
    }
    return sMD5;
  }

  /**
   * A File copy which should be quite fast. If the destination file does not
   * exist, it is created.
   * 
   * @param src
   *          Source file
   * @param dest
   *          Destination file
   * @param keepDate
   *          If <code>true</code> the destination obtains the soruce file
   *          modification date (like a real file copy does). If If
   *          <code>true</code> the destination will have the date at the time
   *          is is copied.
   * @throws IOException
   *           sonst was
   */
  public static void fileCopy(final File src, final File destIn, final boolean keepDate) throws IOException {
    File dest = destIn;
    if (destIn.isDirectory()) {
      dest = new File(destIn, src.getName());
    }
    if (!dest.exists()) {
      dest.createNewFile();
    }
    FileInputStream fin = null;
    FileOutputStream fout = null;
    FileChannel fchIn = null;
    FileChannel fchOut = null;

    try {
      fin = new FileInputStream(src);
      fout = new FileOutputStream(dest);
      fchIn = fin.getChannel();
      fchOut = fout.getChannel();

      long position = 0;
      long transferred;

      long remaining = fchIn.size();
      // it seems, that transferTo "may or may not transfer all of the
      // requested bytes",
      // so do a loop.

      fchOut.position(0);
      fchIn.position(0);

      while (remaining > 0) {
        // transferred = fchIn.transferTo(position, remaining, fchOut);
        transferred = fchOut.transferFrom(fchIn, position, remaining);
        remaining -= transferred;
        position += transferred;
      }
    } finally {
      if (null != fchOut) {
        fchOut.close();
      }
      if (null != fchIn) {
        fchIn.close();
      }
      if (null != fout) {
        fout.close();
      }
      if (null != fin) {
        fin.close();
      }
    }
    if (keepDate) {
      dest.setLastModified(src.lastModified());
    }
  }

  /**
   * A File copy which should be quite fast. If the destination file does not
   * exist, it is created.
   * 
   * @param src
   *          Source file
   * @param dest
   *          Destination file
   * @throws IOException
   *           geht nicht
   */
  public static void fileCopy(final File src, final File dest) throws IOException {

    fileCopy(src, dest, false);

  }

  /**
   * getting all files recursivly from a directory.
   * 
   * @param baseDir
   *          where to search from.
   * @param recursive
   *          search the file structure recursivly.
   * @return File[] list of all files
   */
  public static File[] getFiles(final File baseDir, final boolean recursive) {
    return getFiles(baseDir, recursive, false);
  }

  public static File[] getFiles(final File baseDir, final boolean recursive, final boolean excludeDotFiles) {
    ArrayList<File> fileList = new ArrayList<File>();
    File[] files = baseDir.listFiles();
    if (files != null) {
      for (int i = 0; i < files.length; i++) {
        File file = files[i];
        if (file.isDirectory()) {
          if (!excludeDotFiles || !file.getName().startsWith(".")) {
            File[] newFiles = getFiles(file, recursive, excludeDotFiles);
            for (int j = 0; j < newFiles.length; j++) {
              fileList.add(newFiles[j]);
            }
          }
        } else if (file.isFile()) {
          if (!excludeDotFiles || !file.getName().startsWith(".")) {
            fileList.add(file);
          }
        }
      }
    }
    return (File[]) fileList.toArray(new File[0]);
  }

  /**
   * copy a list of list into a directory.
   * 
   * @param files
   *          list with all files to copy.
   * @param installSiteDir
   *          where to copy to.
   * @throws IOException
   *           if something goes wrong.
   */
  public static void fileCopy(final File[] files, final File installSiteDir) throws IOException {
    for (int i = 0; i < files.length; i++) {
      File srcFile = files[i];
      File destFile = new File(installSiteDir, srcFile.getName());
      fileCopy(srcFile, destFile);
    }
  }

  // copied from FileTool class.
  /**
   * Retrieves the existance of a file or directory (to be or not to be....).
   * 
   * @param fileName
   *          name of the file or directory.
   * @return success
   */
  public static boolean fileExists(final String fileName) {
    return new File(fileName).exists();
  }

  /**
   * Create a directory; all ancestor directories must exist.
   * 
   * @param directory
   *          name of the directory
   * @return success
   */
  public static boolean createDirectory(final String directory) {
    boolean success = (new File(directory)).mkdir();
    if (!success) {
      System.err.println("Directory [" + directory + "] creation failed !");
    }
    return success;
  }

  /**
   * Create a directory; all non-existent ancestor directories are.
   * automatically created
   * 
   * @param directory
   *          the directory to create.
   * @return success
   */
  public static boolean createDirectories(final String directory) {
    boolean success = (new File(directory)).mkdirs();
    if (!success) {
      System.err.println("Directory [" + directory + "] creation failed !");
    }
    return success;
  }

  /**
   * Create a directory and a number of subdirectories specified in the String
   * array automatically created.
   * 
   * @param aDirectory
   *          the directory to create.
   * @param subdirectories
   *          array with the subdirectories to create.
   * @return success
   */
  public static boolean createDirectoryStructure(final String aDirectory, final String[] subdirectories) {
    String directory = aDirectory;
    boolean success = true;
    directory = eliminateDoubleSlashes(directory);
    if (!fileExists(directory)) {
      if (!createDirectories(directory)) {
        return false;
      }
    }
    if (!directory.endsWith("/")) {
      directory += '/';
    }
    for (int iLoop = 0; iLoop < subdirectories.length; iLoop++) {
      File subDir = new File(directory + subdirectories[iLoop]);
      if (!subDir.exists()) {
        if (!createDirectory(directory + subdirectories[iLoop])) {
          success = false;
          break;
        }
      }
      // success=(iLoop==subdirectories.length-1);
    }

    if (!success) {
      System.err.println("Directory [" + directory + "] creation failed !");
    }
    return success;
  }

  /**
   * Move / Rename source location to destination. String array automatically
   * created
   * 
   * @param source
   *          source file
   * @param destination
   *          destination file
   * @return success
   * 
   */
  @Deprecated
  public static boolean move(final String source, final String destination) {
    try {
      return (new File(source)).renameTo(new File(destination));
    } catch (SecurityException e) {
      System.err.println("The file " + source + " could not moved to " + destination + " : " + e.getLocalizedMessage());
      return false;
    }
  }

  /**
   * Move / Rename source location to destination. String array automatically
   * created
   * 
   * @param source
   *          source file
   * @param destination
   *          destination file
   * @return success
   */
  public static boolean move(final File source, final File destination) {
    try {
      return (source.renameTo(destination));
    } catch (SecurityException e) {
      System.err.println("The file " + source + " could not moved to " + destination + " : " + e.getLocalizedMessage());
      return false;
    }
  }

  /**
   * Copy source location to destination.
   * 
   * @param source
   *          source file
   * @param destination
   *          destination file
   * @return success
   */
  @Deprecated
  public static boolean copy(final String source, final String destination) {
    searchDepth = 0;
    return filecopy(source, destination, true);
  }

  /**
   * Copy source location to destination.
   * 
   * @param source
   *          source file
   * @param destination
   *          destination file
   * @param subdirectories
   *          <code>true</code> if subdirectories should be copied too,
   *          otherwise <code>false</code>
   * @return success
   */
  @Deprecated
  public static boolean copy(final String source, final String destination, final boolean subdirectories) {
    searchDepth = 0;
    return filecopy(source, destination, subdirectories);
  }

  /**
   * Filename filter class. This class will determine files with dos wildcards.
   * 
   * @author w.klaas
   */
  static class FileIncludes implements FilenameFilter {

    /** directory to eveluate. */
    private File dir;

    /** the regex. */
    private Pattern mask;

    /**
     * is the mask a mask?
     * 
     * @param mask
     *          the mask
     * @return <code>true</code> if the mask containing a * or ?
     */
    private static boolean isMask(final String mask) {
      int pos = mask.indexOf('*');
      if (pos < 0) {
        pos = mask.indexOf('?');
      }
      return pos >= 0;
    }

    /**
     * special regexp chars : '.', '\\', '$', '^', '(' , ')', '[', ']' ,'+', .
     * (must be masked if in mask) special mask chars to convert for regexp :
     * '*' -> '.*', '?' -> '.+'
     */
    private static String regexpSpecChars = ".\\$^()[]+";

    /**
     * convert the mask to a regualr expression.
     * 
     * @param aMask
     *          the mask
     * @return String regular expression.
     */
    String toRegExp(final String aMask) {
      StringBuffer sb = new StringBuffer();
      int length = aMask.length();
      for (int index = 0; index < length; index++) {
        char ch = aMask.charAt(index);
        if (ch == '*') {
          sb.append(".*");
        } else if (ch == '?') {
          sb.append(".+");
        } else if (regexpSpecChars.indexOf(ch) >= 0) {
          sb.append('\\').append(ch);
        } else {
          sb.append(ch);
        }
      }
      return sb.toString();
    }

    /**
     * Constructs a FilenameFilter instance to filter files, which base name
     * (without the directoy path) that match the given regular expression.
     * 
     * @param aDir
     *          the directory
     * @param aMask
     *          the mask
     * @throws java.util.regex.PatternSyntaxException
     */
    public FileIncludes(final File aDir, final String aMask) {
      this.dir = aDir;
      this.mask = Pattern.compile(toRegExp(aMask));
      // System.out.println("FileIncludes: dir=" + dir.toString() + ",
      // mask=" + mask);

    }

    /**
     * Tests if a specified file should be included in a file list.
     * 
     * @param aDir
     *          the directory in which the file was found.
     * @param name
     *          the name of the file.
     * @return <code>true</code> if and only if the name should be included in
     *         the file list; <code>false</code> otherwise.
     * @see java.io.FilenameFilter#accept(java.io.File, java.lang.String)
     */
    public boolean accept(final File aDir, final String name) {
      // System.out.println("accept: dir=" + dir.toString() + ", name=" +
      // name);
      if (aDir.equals(this.dir)) {
        boolean matches = mask.matcher(name).matches();
        // System.out.println("accept: result=" + matches);
        return matches;
      } else {
        // System.out.println("accept: result=" + false);
        return false;
      }
    }

  }

  /** default subdirectory depth to copy. */
  private static int searchDepth = 0;

  /**
   * Copy source location to destination.
   * 
   * @param source
   *          The pathname of a single directory or a single file or multiple
   *          files through a filemask.
   * @param destination
   *          destination path
   * @param subdirectories
   *          allow subdirectories to be copied
   * @return <code>true</code> if this operation succseeded
   */
  protected static boolean filecopy(final String source, final String destination, final boolean subdirectories) {

    boolean success = true; // just be optimistic
    File sourceFile = new File(source);
    if (sourceFile.isDirectory()) {
      if (searchDepth++ > 0 && !subdirectories) {
        return true;
      }
      if (!fileExists(destination)) {
        createDirectories(destination);
      }
      String[] subFiles = sourceFile.list();
      for (int iLoop = 0; iLoop < subFiles.length; iLoop++) {
        filecopy(source + File.separator + subFiles[iLoop], destination + File.separator + subFiles[iLoop],
            subdirectories);
      }
    } else {
      // create destination directory if it does not exist
      int fSlash = destination.lastIndexOf("/"), bSlash = destination.lastIndexOf("\\");
      int slash;
      if (fSlash > bSlash) {
        slash = fSlash;
      } else {
        slash = bSlash;
      }
      if (slash > -1) {
        String path = destination.substring(0, slash);
        File dest = new File(path);
        if (!dest.exists()) {
          createDirectories(path);
        }
      }

      File[] inputFiles = new File[] { sourceFile };
      String mask = sourceFile.getName();
      if (FileIncludes.isMask(mask)) {
        File sourceDir = sourceFile.getParentFile();
        if (null != sourceDir && mask.length() > 0) {
          try {
            FilenameFilter filter = new FileIncludes(sourceDir, mask);
            inputFiles = sourceDir.listFiles(filter);
          } catch (java.util.regex.PatternSyntaxException e) {
            // nothing to do here
          }
        }
      }

      File destFile = new File(destination);
      if (null != inputFiles) {
        if (destFile.isFile() && inputFiles.length > 1) {
          System.err.println("For copying multiple files to the destination \"" + destination
              + "\" it must be a directory and NOT A SINGLE file.");
          success = false;
        } else {
          for (int idx = 0; idx < inputFiles.length; idx++) {
            File inputFile = inputFiles[idx];

            File destinationFile;
            if (destFile.isDirectory()) {
              destinationFile = new File(destFile, inputFile.getName());
            } else {
              destinationFile = destFile;
            }

            try {
              // calling filecopy using FileChannel
              Files.fileCopy(inputFile, destinationFile, true);
              success = true;
            } catch (FileNotFoundException eFNF) {
              System.err.println("The source file could not be found. " + eFNF);
              System.err.println("Tried to copy " + inputFile + " to " + destinationFile + " : " + eFNF);
              success = false;
              break;
            } catch (IOException eIO) {
              System.err.println("IOException occured while copying file.");
              System.err.println("Tried to copy " + inputFile + " to " + destinationFile + " : " + eIO);
              success = false;
              break;
            }
          }
        }
      }
    }
    return success;
  }

  /**
   * @param filename
   *          A filename that may be a relative filename like "." or "..".
   * @return The absolute <code>File</code> name instance fo the given
   *         (relative) filename. Return <code>null</code> if ".." is given and
   *         the current or parent directory is /
   */
  public static File getAbsoluteFile(final String filename) {
    File file = new File(filename).getAbsoluteFile();
    if (".".equals(file.getName())) {
      file = file.getParentFile();
    } else if ("..".equals(file.getName())) {
      file = file.getParentFile();
      if (null != file) {
        file = file.getParentFile();
      }
    }
    return file;
  }

  /**
   * making a string to a directory with ending on File.separator.
   * 
   * @param aPath
   *          path to convert
   * @return string
   */
  public static String makeDirectory(final String aPath) {
    String path = aPath;
    String separatorChar = System.getProperty("file.separator");
    if (path != null && !path.endsWith(separatorChar)) {
      path += separatorChar;
    }
    return path;
  }

  /**
   * removes a directory recursively. All Files and Subdirectories are removed.
   * If bRecursive is false and the directory is not empty, an IOException is
   * thrown.
   * 
   * @param filename
   *          path or file to remove
   * @param bRecursive
   *          delete all subfolders too
   * @return success <code>true</code> if all files could be removed, else
   *         return <code>false</code>
   * @throws IOException
   *           if something goes wrong
   */
  @Deprecated
  public static boolean remove(final String filename, final boolean bRecursive) throws IOException {
    return remove(filename, bRecursive, null);
  }

  /**
   * removes a directory recursively. All Files and Subdirectories are removed.
   * If bRecursive is false and the directory is not empty, an IOException is
   * thrown.
   * 
   * @param filename
   *          path or file to remove
   * @param bRecursive
   *          delete all subfolders too
   * @return success <code>true</code> if all files could be removed, else
   *         return <code>false</code>
   * @throws IOException
   *           if something goes wrong
   */
  public static boolean remove(final File file, final boolean bRecursive) throws IOException {
    return remove(file, bRecursive, null);
  }

  /**
   * removes a directory recursively. All Files and Subdirectories are removed.
   * If bRecursive is false and the directory is not empty, an IOException is
   * thrown.
   * 
   * @param filename
   *          path or file to remove
   * @param bRecursive
   *          delete all subfolders too
   * @param excludes
   *          list with files to exclude from deleting
   * @return success <code>true</code> if all files could be removed, else
   *         return <code>false</code>
   * @throws IOException
   *           if something goes wrong
   */
  @Deprecated
  public static boolean remove(final String filename, final boolean bRecursive, final List<String> excludes)
      throws IOException {
    boolean bSuccess = true;
    if (filename != null && 0 != filename.length()) {
      File file = new File(filename);
      if (file.isDirectory()) {
        String[] files = file.list();

        if (!bRecursive && files.length > 0) {
          System.err.println();
          throw new IOException("Directory " + file.getAbsolutePath() + " is not Empty");
        } else {
          for (int iLoop = 0; iLoop < files.length; iLoop++) {
            String toRemove = convertSlashes(file.getAbsolutePath() + "/" + files[iLoop]);
            if (!isExcluded(toRemove, excludes)) {
              if (!remove(toRemove, true)) {
                bSuccess = false;
              }
            }
          }
          if (bSuccess) {
            if (!isExcluded(filename, excludes)) {
              bSuccess = remove(filename);
            }
          }
        }
      } else {
        if (!isExcluded(filename, excludes)) {
          bSuccess = remove(filename);
        }
      }
    }
    return bSuccess;
  }

  /**
   * removes a directory recursively. All Files and Subdirectories are removed.
   * If bRecursive is false and the directory is not empty, an IOException is
   * thrown.
   * 
   * @param filename
   *          path or file to remove
   * @param bRecursive
   *          delete all subfolders too
   * @param excludes
   *          list with files to exclude from deleting
   * @return success <code>true</code> if all files could be removed, else
   *         return <code>false</code>
   * @throws IOException
   *           if something goes wrong
   */
  public static boolean remove(final File source, final boolean bRecursive, final List<String> excludes)
      throws IOException {
    boolean bSuccess = true;
    if ((source != null) && source.exists()) {
      if (source.isDirectory()) {
        File[] files = source.listFiles();

        if (!bRecursive && files.length > 0) {
          System.err.println();
          throw new IOException("Directory " + source.getAbsolutePath() + " is not Empty");
        } else {
          for (int iLoop = 0; iLoop < files.length; iLoop++) {
            File file = files[iLoop];
            if (!isExcluded(file, excludes)) {
              if (!remove(file, true)) {
                bSuccess = false;
              }
            }
          }
          if (bSuccess) {
            if (!isExcluded(source, excludes)) {
              bSuccess = source.delete();
            }
          }
        }
      } else {
        if (!isExcluded(source, excludes)) {
          bSuccess = source.delete();
        }
      }
    }
    return bSuccess;
  }

  /**
   * checks if a file is in the exclude list.
   * 
   * @param filename
   *          file to test
   * @param excludes
   *          list with the excluded files
   * @return boolean <code>true</code> if the file is in the excludelist, else
   *         <code>false</code>
   */
  @Deprecated
  private static boolean isExcluded(final String filename, final List<String> excludes) {
    boolean bDelete = true;
    if (excludes != null && excludes.size() > 0) {
      for (int iFileLoop = 0; iFileLoop < excludes.size(); iFileLoop++) {
        String exclude = convertSlashes((String) excludes.get(iFileLoop));
        if (filename.indexOf(exclude) > -1 || exclude.startsWith(filename)) {
          bDelete = false;
          break;
        }
      }
    }
    return !bDelete;
  }

  /**
   * checks if a file is in the exclude list.
   * 
   * @param filename
   *          file to test
   * @param excludes
   *          list with the excluded files
   * @return boolean <code>true</code> if the file is in the excludelist, else
   *         <code>false</code>
   */
  private static boolean isExcluded(final File file, final List<String> excludes) {
    boolean bDelete = true;
    if (excludes != null && excludes.size() > 0) {
      for (int i = 0; i < excludes.size(); i++) {
        String exclude = convertSlashes((String) excludes.get(i));
        try {
          if (file.getCanonicalPath().indexOf(exclude) > -1 || exclude.startsWith(file.getCanonicalPath())) {
            bDelete = false;
            break;
          }
        } catch (IOException e) {
          bDelete = false;
        }
      }
    }
    return !bDelete;
  }

  /**
   * Delete the file.
   * 
   * @param filename
   *          file to delete
   * @return <code>true</code> if the file could be delete, <code>false</code>
   *         if not.
   */
  @Deprecated
  public static boolean remove(final String filename) {
    try {
      return new File(filename).delete();
    } catch (SecurityException e) {
      System.err.println("The file " + filename + " could not be removed: " + e.getLocalizedMessage());
      return false;
    }
  }

  public static List<String> readFileToList(final String fileName) throws IOException {
    return readFileToList(new File(fileName));
  }

  /**
   * reading a file into an arraylist. Every line is one entry.
   * 
   * @param file
   *          file to read
   * @return ArrayList with all lines
   * @throws IOException
   */
  public static List<String> readFileToList(final File file) throws IOException {
    return readFileToList(file, true);
  }

  public static List<String> readFileToList(final File file, boolean comments) throws IOException {
    ArrayList<String> list = new ArrayList<String>();
    BufferedReader br = null;
    br = new BufferedReader(new InputStreamReader(new FileInputStream(file)));
    String line = "";
    try {
      while ((line = br.readLine()) != null) {
        if (line.trim().length() > 0 && ((line.trim().charAt(0) != '#') || !comments)) {
          list.add(line);
        }
      }
    } finally {
      try {
        br.close();
      } catch (IOException e) {
        e.printStackTrace();
      }
    }
    return list;
  }

  @Deprecated
  public static void writeListToFile(final String fileName, List<String> list) throws IOException {
    writeListToFile(new File(fileName), list);
  }

  /**
   * reading a file into an arraylist. Every line is one entry.
   * 
   * @param file
   *          file to read
   * @return ArrayList with all lines
   * @throws IOException
   */
  public static void writeListToFile(final File file, List<String> list) throws IOException {
    PrintWriter br = null;
    br = new PrintWriter(new BufferedWriter(new FileWriter(file)));
    for (Object entry : list) {
      br.println(entry.toString());
    }
    br.close();
  }

  /**
   * reading a file into an string.
   * 
   * @param fileName
   *          file to read
   * @return String with the content
   */
  @Deprecated
  public static String readFileToString(final String fileName) {
    StringBuilder buffer = new StringBuilder();
    BufferedReader br = null;
    try {
      br = new BufferedReader(new InputStreamReader(new FileInputStream(fileName)));
    } catch (FileNotFoundException e) {
      System.err.println("The File " + fileName + " could not be found !!");
      e.printStackTrace();
      System.exit(3);
    }
    String line = "";
    try {
      while ((line = br.readLine()) != null) {
        if (line.trim().length() > 0) {
          buffer.append(line);
          buffer.append("\r\n");
        }
      }
    } catch (IOException e1) {
      System.err.println("IOException occured while reading " + fileName + "");
      e1.printStackTrace();
    } finally {
      try {
        br.close();
      } catch (IOException e) {
        e.printStackTrace();
      }
    }
    return buffer.toString();
  }

  /**
   * reading a file into an string. checking and converting the lineendings.
   * 
   * @param file
   *          file to read
   * @return String with the content
   */
  public static String readFileToString(final File file) {
    StringBuilder buffer = new StringBuilder();
    BufferedReader br = null;
    try {
      br = new BufferedReader(new InputStreamReader(new FileInputStream(file)));
    } catch (FileNotFoundException e) {
      System.err.println("The File " + file + " could not be found !!");
      e.printStackTrace();
      System.exit(3);
    }
    String line = "";
    try {
      while ((line = br.readLine()) != null) {
        if (line.trim().length() > 0) {
          buffer.append(line);
          buffer.append("\r\n");
        }
      }
    } catch (IOException e1) {
      System.err.println("IOException occured while reading " + file + "");
      e1.printStackTrace();
    } finally {
      try {
        br.close();
      } catch (IOException e) {
        e.printStackTrace();
      }
    }
    return buffer.toString();
  }

  /**
   * reading a file into an string. no checking and converting.
   * 
   * @param file
   *          file to read
   * @return String with the content
   */
  public static String readFile(final File file) throws IOException {
    BufferedInputStream br = null;
    br = new BufferedInputStream(new FileInputStream(file));

    ByteArrayOutputStream out = new ByteArrayOutputStream();
    StreamHelper.copyStream(br, out);
    br.close();

    return new String(out.toByteArray());
  }

  /**
   * simple writing a string into a file.
   * 
   * @param filename
   *          of the file to create
   * @param content
   *          string to write into the file.
   */
  @Deprecated
  public static void writeStringToFile(final String filename, final String content) {
    File file = new File(filename);
    try {
      writeStringToFile(file, content);
    } catch (IOException e) {
      e.printStackTrace();
    }
  }

  /**
   * simple writing a string into a file.
   * 
   * @param file
   *          of the file to create
   * @param content
   *          string to write into the file.
   * @throws IOException
   */
  public static void writeStringToFile(final File file, final String content) throws IOException {
    BufferedWriter bufWriter = new BufferedWriter(new FileWriter(file));
    bufWriter.write(content);
    bufWriter.close();
  }

  /**
   * Eleminates // in a filename.
   * 
   * @param filename
   *          to convert
   * @return String
   */
  public static String eliminateDoubleSlashes(final String filename) {
    String result = "";
    if (filename.indexOf(":") == -1 && filename.startsWith("//")) {
      result = filename.substring(0, 2) + filename.substring(2).replaceAll("//", "/");
    } else {
      result = filename.replaceAll("//", "/");
    }
    return result;
  }

  /**
   * converts 2 backslashes into one slash.
   * 
   * @param path
   *          to convert
   * @return string
   */
  public static String convertSlashes(final String path) {
    return path.replaceAll("\\\\", "/");
  }

  /**
   * converts one slash into 2 backslashes.
   * 
   * @param path
   *          to convert
   * @return string
   */
  public static String convertBackSlashes(final String path) {
    return path.replaceAll("/", "\\\\");
  }

  /**
   * converts only if backslash is not proceeded by a blank.
   * 
   * @param aPath
   *          to convert
   * @return string
   */
  public static String convertBackSlashesNoParams(final String aPath) {
    String path = aPath;
    int slash = -1;
    while ((slash = path.indexOf("/", slash + 1)) > -1) {
      if (slash > -1 && path.charAt(slash - 1) != ' ') {
        String sLineStart = path.substring(0, slash);
        String sLineEnd = path.substring(slash + 1);
        path = sLineStart + "\\" + sLineEnd;
      }
    }
    return path;
  }

  /**
   * converting the given path with an File.separator at the end.
   * 
   * @param aPath
   *          the path to convert
   * @return the converted path
   */
  public static String normPath(final String aPath) {
    if (!aPath.endsWith("/") && !aPath.endsWith("\\")) {
      return aPath + File.separator;
    } else {
      return aPath;
    }
  }

  /**
   * getting the file extension.
   * 
   * @param file
   *          the file to get the extension from.
   * @return the extension with "." or <code>null</code> if the file is not a
   *         file.
   */
  public static String getExtension(final File file) {
    if (file.isFile()) {
      String name = file.getName();
      if (name.indexOf((int) '.') >= 0) {
        return name.substring(name.lastIndexOf("."));
      } else {
        return "";
      }
    }
    return null;
  }

  /**
   * getting the file extension.
   * 
   * @param file
   *          the file to get the extension from.
   * @return the extension with "." or <code>null</code> if the file is not a
   *         file.
   */
  public static String getExtension(final String name) {
    return name.substring(name.lastIndexOf("."));
  }

  /**
   * getting the file name without extension.
   * 
   * @param file
   *          the file to get the extension from.
   * @return the name without "." or <code>null</code> if the file is not a
   *         file.
   */
  public static String extractName(final File file) {
    if (file.isFile()) {
      String name = file.getName();
      if (name.lastIndexOf(".") >= 0) {
        return name.substring(0, name.lastIndexOf("."));
      } else {
        return name;
      }
    } else {
      String name = file.getName();
      if (name.indexOf(".") >= 0) {
        return name.substring(0, name.lastIndexOf("."));
      } else {
        return name;
      }
    }
  }

  /**
   * changing the file extension.
   * 
   * @param file
   * @return
   */
  public static File changeExtension(final File file, final String newExtension) {
    return new File(file.getParentFile(), extractName(file) + newExtension);
  }

  public static String readableFileSize(long size) {
    if (size <= 0)
      return "0";
    final String[] units = new String[] { "B", "KB", "MB", "GB", "TB" };
    int digitGroups = (int) (Math.log10(size) / Math.log10(1024));
    return new DecimalFormat("#,##0.#").format(size / Math.pow(1024, digitGroups)) + " " + units[digitGroups];
  }

  public static String getDriveLetter(File file) throws IOException {
    String canonicalPath = file.getCanonicalPath();
    canonicalPath = canonicalPath.substring(0, 2);
    return canonicalPath;
  }

  public static File getAppData() {
    String appDataStr = System.getenv("APPDATA");
    if ((appDataStr == null) || appDataStr.equals("")) {
      appDataStr = System.getenv("LOCALAPPDATA");
      if ((appDataStr == null) || appDataStr.equals("")) {
        appDataStr = Files.getTempPath();
      }
    }
    return new File(appDataStr);
  }
}
