/*
 * MCS Media Computer Software Copyright (c) 2005 by MCS
 * -------------------------------------- Created on 16.01.2004 by w.klaas
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package de.mcs.utils;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

/**
 * filetool class.
 * 
 * @author w.klaas
 * @deprecated please use the Files class.
 */
public final class FileTool {

  /** prevent instancing. */
  private FileTool() {
  }

  /**
   * Retrieves the existance of a file or directory (to be or not to be....).
   * 
   * @param fileName
   *          name of the file or directory.
   * @return success
   */
  public static boolean fileExists(final String fileName) {
    return new File(fileName).exists();
  }

  /**
   * Create a directory; all ancestor directories must exist.
   * 
   * @param directory
   *          name of the directory
   * @return success
   */
  public static boolean createDirectory(final String directory) {
    boolean success = (new File(directory)).mkdir();
    if (!success) {
      System.err.println("Directory [" + directory + "] creation failed !");
    }
    return success;
  }

  /**
   * Create a directory; all non-existent ancestor directories are.
   * automatically created
   * 
   * @param directory
   *          the directory
   * @return success
   */
  public static boolean createDirectories(final String directory) {
    boolean success = (new File(directory)).mkdirs();
    if (!success) {
      System.err.println("Directory [" + directory + "] creation failed !");
    }
    return success;
  }

  /**
   * Create a directory and a number of subdirectories specified in the String
   * array automatically created.
   * 
   * @param aDirectory
   *          the directory
   * @param subdirectories
   *          the subdirectories
   * @return success
   */
  public static boolean createDirectoryStructure(final String aDirectory, final String[] subdirectories) {
    String directory = aDirectory;
    boolean success = true;
    directory = eliminateDoubleSlashes(directory);
    if (!fileExists(directory)) {
      if (!createDirectories(directory)) {
        return false;
      }
    }
    if (!directory.endsWith("/")) {
      directory += '/';
    }
    for (int iLoop = 0; iLoop < subdirectories.length; iLoop++) {
      File subDir = new File(directory + subdirectories[iLoop]);
      if (!subDir.exists()) {
        if (!createDirectory(directory + subdirectories[iLoop])) {
          success = false;
          break;
        }
      }
      // success=(iLoop==subdirectories.length-1);
    }

    if (!success) {
      System.err.println("Directory [" + directory + "] creation failed !");
    }
    return success;
  }

  /**
   * Move / Rename source location to destination. String array automatically
   * created
   * 
   * @param source
   *          source
   * @param destination
   *          destination
   * @return success
   */
  public static boolean move(final String source, final String destination) {
    try {
      return (new File(source)).renameTo(new File(destination));
    } catch (SecurityException e) {
      System.err.println("The file " + source + " could not moved to " + destination + " : " + e.getLocalizedMessage());
      return false;
    }
  }

  /**
   * Copy source location to destination.
   * 
   * @param source
   *          source
   * @param destination
   *          destination
   * @return success
   */
  public static boolean copy(final String source, final String destination) {
    searchDepth = 0;
    return filecopy(source, destination, true);
  }

  /**
   * Copy source location to destination.
   * 
   * @param source
   *          source
   * @param destination
   *          destination
   * @param subdirectories
   *          subdirectories
   * @return success
   */
  public static boolean copy(final String source, final String destination, final boolean subdirectories) {
    searchDepth = 0;
    return filecopy(source, destination, subdirectories);
  }

  /**
   * Filename filter class.
   * 
   * @author w.klaas
   */
  static class FileIncludes implements FilenameFilter {

    /** directory to eveluate. */
    private File dir;

    /** the regex. */
    private Pattern mask;

    /** prevent instancing with the default constructor. */
    // private FileIncludes() {
    // throw new UnsupportedOperationException();
    // }

    /**
     * is the mask a mask?
     * 
     * @param mask
     *          the mask
     * 
     * @return <code>true</code> if the mask containing a * or ?
     */
    private static boolean isMask(final String mask) {
      int pos = mask.indexOf('*');
      if (pos < 0) {
        pos = mask.indexOf('?');
      }
      return pos >= 0;
    }

    /**
     * special regexp chars : '.', '\\', '$', '^', '(' , ')', '[', ']' ,'+', .
     * (must be masked if in mask) special mask chars to convert for regexp :
     * '*' -> '.*', '?' -> '.+'
     */
    private static String regexpSpecChars = ".\\$^()[]+";

    /**
     * convert the mask to a regualr expression.
     * 
     * @param aMask
     *          the mask
     * @return String regular expression.
     */
    String toRegExp(final String aMask) {
      StringBuffer sb = new StringBuffer();
      int length = aMask.length();
      for (int index = 0; index < length; index++) {
        char ch = aMask.charAt(index);
        if (ch == '*') {
          sb.append(".*");
        } else if (ch == '?') {
          sb.append(".+");
        } else if (regexpSpecChars.indexOf(ch) >= 0) {
          sb.append('\\').append(ch);
        } else {
          sb.append(ch);
        }
      }
      return sb.toString();
    }

    /**
     * Constructs a FilenameFilter instance to filter files, which base name
     * (without the directoy path) that match the given regular expression.
     * 
     * @param aDir
     *          the directory
     * @param aMask
     *          the mask
     * @throws java.util.regex.PatternSyntaxException
     */
    public FileIncludes(final File aDir, final String aMask) {
      this.dir = aDir;
      this.mask = Pattern.compile(toRegExp(aMask));
      // System.out.println("FileIncludes: dir=" + dir.toString() + ",
      // mask=" + mask);

    }

    /**
     * Tests if a specified file should be included in a file list.
     * 
     * @param aDir
     *          the directory in which the file was found.
     * @param name
     *          the name of the file.
     * @return <code>true</code> if and only if the name should be included in
     *         the file list; <code>false</code> otherwise.
     * @see java.io.FilenameFilter#accept(java.io.File, java.lang.String)
     */
    public boolean accept(final File aDir, final String name) {
      // System.out.println("accept: dir=" + dir.toString() + ", name=" +
      // name);
      if (aDir.equals(this.dir)) {
        boolean matches = mask.matcher(name).matches();
        // System.out.println("accept: result=" + matches);
        return matches;
      } else {
        // System.out.println("accept: result=" + false);
        return false;
      }
    }

  }

  /** default subdirectory depth to copy. */
  private static int searchDepth = 0;

  /**
   * Copy source location to destination.
   * 
   * @param source
   *          The pathname of a single directory or a single file or multiple
   *          files through a filemask.
   * @param destination
   *          destination path
   * @param subdirectories
   *          allow subdirectories to be copied
   * @return <code>true</code> if this operation succseeded
   */
  protected static boolean filecopy(final String source, final String destination, final boolean subdirectories) {

    boolean success = true; // just be optimistic
    File sourceFile = new File(source);
    if (sourceFile.isDirectory()) {
      if (searchDepth++ > 0 && !subdirectories) {
        return true;
      }
      if (!fileExists(destination)) {
        createDirectories(destination);
      }
      String[] subFiles = sourceFile.list();
      for (int iLoop = 0; iLoop < subFiles.length; iLoop++) {
        filecopy(source + File.separator + subFiles[iLoop], destination + File.separator + subFiles[iLoop],
            subdirectories);
      }
    } else {
      // create destination directory if it does not exist
      int fSlash = destination.lastIndexOf("/"), bSlash = destination.lastIndexOf("\\");
      int slash;
      if (fSlash > bSlash) {
        slash = fSlash;
      } else {
        slash = bSlash;
      }
      if (slash > -1) {
        String path = destination.substring(0, slash);
        File dest = new File(path);
        if (!dest.exists()) {
          createDirectories(path);
        }
      }

      File[] inputFiles = new File[] { sourceFile };
      String mask = sourceFile.getName();
      if (FileIncludes.isMask(mask)) {
        File sourceDir = sourceFile.getParentFile();
        if (null != sourceDir && mask.length() > 0) {
          try {
            FilenameFilter filter = new FileIncludes(sourceDir, mask);
            inputFiles = sourceDir.listFiles(filter);
          } catch (java.util.regex.PatternSyntaxException e) {
            // nothing to do here
          }
        }
      }

      File destFile = new File(destination);
      if (null != inputFiles) {
        if (destFile.isFile() && inputFiles.length > 1) {
          System.err.println("For copying multiple files to the destination \"" + destination
              + "\" it must be a directory and NOT A SINGLE file.");
          success = false;
        } else {
          for (int idx = 0; idx < inputFiles.length; idx++) {
            File inputFile = inputFiles[idx];

            File destinationFile;
            if (destFile.isDirectory()) {
              destinationFile = new File(destFile, inputFile.getName());
            } else {
              destinationFile = destFile;
            }

            try {
              // calling filecopy using FileChannel
              Files.fileCopy(inputFile, destinationFile, true);
              success = true;
            } catch (FileNotFoundException eFNF) {
              System.err.println("The source file could not be found. " + eFNF);
              System.err.println("Tried to copy " + inputFile + " to " + destinationFile + " : " + eFNF);
              success = false;
              break;
            } catch (IOException eIO) {
              System.err.println("IOException occured while copying file.");
              System.err.println("Tried to copy " + inputFile + " to " + destinationFile + " : " + eIO);
              success = false;
              break;
            }
          }
        }
      }
    }
    return success;
  }

  /**
   * @param filename
   *          A filename that may be a relative filename like "." or "..".
   * @return The absolute <code>File</code> name instance fo the given
   *         (relative) filename. Return <code>null</code> if ".." is given and
   *         the current or parent directory is /
   */
  public static File getAbsoluteFile(final String filename) {
    File file = new File(filename).getAbsoluteFile();
    if (".".equals(file.getName())) {
      file = file.getParentFile();
    } else if ("..".equals(file.getName())) {
      file = file.getParentFile();
      if (null != file) {
        file = file.getParentFile();
      }
    }
    return file;
  }

  /**
   * making a string to a directory with ending on File.separator.
   * 
   * @param aPath
   *          path to convert
   * @return string
   */
  public static String makeDirectory(final String aPath) {
    String path = aPath;
    String separatorChar = System.getProperty("file.separator");
    if (path != null && !path.endsWith(separatorChar)) {
      path += separatorChar;
    }
    return path;
  }

  /**
   * removes a directory recursively. All Files and Subdirectories are removed.
   * If bRecursive is false and the directory is not empty, an IOException is
   * thrown.
   * 
   * @param filename
   *          path or file to remove
   * @param bRecursive
   *          delete all subfolders too
   * @return success <code>true</code> if all files could be removed, else
   *         return <code>false</code>
   * @throws IOException
   *           if something goes wrong
   */
  public static boolean remove(final String filename, final boolean bRecursive) throws IOException {
    return remove(filename, bRecursive, null);
  }

  /**
   * removes a directory recursively. All Files and Subdirectories are removed.
   * If bRecursive is false and the directory is not empty, an IOException is
   * thrown.
   * 
   * @param filename
   *          path or file to remove
   * @param bRecursive
   *          delete all subfolders too
   * @param excludes
   *          list with files to exclude from deleting
   * @return success <code>true</code> if all files could be removed, else
   *         return <code>false</code>
   * @throws IOException
   *           if something goes wrong
   */
  public static boolean remove(final String filename, final boolean bRecursive, final List<String> excludes)
      throws IOException {
    boolean bSuccess = true;
    if (filename != null && 0 != filename.length()) {
      File file = new File(filename);
      if (file.isDirectory()) {
        String[] files = file.list();

        if (!bRecursive && files.length > 0) {
          System.err.println();
          throw new IOException("Directory " + file.getAbsolutePath() + " is not Empty");
        } else {
          for (int iLoop = 0; iLoop < files.length; iLoop++) {
            String toRemove = FileTool.convertSlashes(file.getAbsolutePath() + "/" + files[iLoop]);
            if (!isExcluded(toRemove, excludes)) {
              if (!remove(toRemove, true)) {
                bSuccess = false;
              }
            }
          }
          if (bSuccess) {
            if (!isExcluded(filename, excludes)) {
              bSuccess = remove(filename);
            }
          }
        }
      } else {
        if (!isExcluded(filename, excludes)) {
          bSuccess = remove(filename);
        }
      }
    }
    return bSuccess;
  }

  /**
   * checks if a file is in the exclude list.
   * 
   * @param filename
   *          file to test
   * @param excludes
   *          list with the excluded files
   * @return boolean <code>true</code> if the file is in the excludelist, else
   *         <code>false</code>
   */
  private static boolean isExcluded(final String filename, final List<String> excludes) {
    boolean bDelete = true;
    if (excludes != null && excludes.size() > 0) {
      for (int iFileLoop = 0; iFileLoop < excludes.size(); iFileLoop++) {
        String exclude = FileTool.convertSlashes((String) excludes.get(iFileLoop));
        if (filename.indexOf(exclude) > -1 || exclude.startsWith(filename)) {
          bDelete = false;
          break;
        }
      }
    }
    return !bDelete;
  }

  /**
   * Delete the file.
   * 
   * @param filename
   *          file to delete
   * @return <code>true</code> if the file could be delete, <code>false</code>
   *         if not.
   */
  public static boolean remove(final String filename) {
    try {
      return new File(filename).delete();
    } catch (SecurityException e) {
      System.err.println("The file " + filename + " could not be removed: " + e.getLocalizedMessage());
      return false;
    }
  }

  /**
   * reading a file into an arraylist. Every line is one entry.
   * 
   * @param fileName
   *          file to read
   * @return ArrayList with all lines
   */
  public static List<String> readFileToList(final String fileName) {
    ArrayList<String> list = new ArrayList<String>();
    BufferedReader br = null;
    try {
      br = new BufferedReader(new InputStreamReader(new FileInputStream(fileName)));
    } catch (FileNotFoundException e) {
      System.err.println("The File " + fileName + " could not be found !!");
      e.printStackTrace();
      System.exit(3);
    }
    String line = "";
    try {
      while ((line = br.readLine()) != null) {
        if (line.trim().length() > 0 && line.trim().charAt(0) != '#') {
          list.add(line);
        }
      }
    } catch (IOException e1) {
      System.err.println("IOException occured while reading " + fileName + "");
      e1.printStackTrace();
    } finally {
      try {
        br.close();
      } catch (IOException e) {
        e.printStackTrace();
      }
    }
    return list;
  }

  /**
   * reading a file into an arraylist. Every line is one entry.
   * 
   * @param fileName
   *          file to read
   * @return ArrayList with all lines
   */
  public static String readFileToString(final String fileName) {
    StringBuffer buffer = new StringBuffer();
    BufferedReader br = null;
    try {
      br = new BufferedReader(new InputStreamReader(new FileInputStream(fileName)));
    } catch (FileNotFoundException e) {
      System.err.println("The File " + fileName + " could not be found !!");
      e.printStackTrace();
      System.exit(3);
    }
    String line = "";
    try {
      while ((line = br.readLine()) != null) {
        if (line.trim().length() > 0 && line.trim().charAt(0) != '#') {
          buffer.append(line);
        }
      }

    } catch (IOException e1) {
      System.err.println("IOException occured while reading " + fileName + "");
      e1.printStackTrace();
    } finally {
      try {
        br.close();
      } catch (IOException e) {
        e.printStackTrace();
      }
    }

    return buffer.toString();
  }

  /**
   * simple writing a string into a file.
   * 
   * @param filename
   *          of the file to create
   * @param content
   *          string to write into the file.
   */
  public static void writeStringToFile(final String filename, final String content) {
    File file = new File(filename);
    try {
      BufferedWriter bufWriter = new BufferedWriter(new FileWriter(file));
      bufWriter.write(content);
      bufWriter.close();
    } catch (IOException e) {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
  }

  /**
   * Eleminates // in a filename.
   * 
   * @param filename
   *          to convert
   * @return String
   */
  public static String eliminateDoubleSlashes(final String filename) {
    String result = "";
    if (filename.indexOf(":") == -1 && filename.startsWith("//")) {
      result = filename.substring(0, 2) + filename.substring(2).replaceAll("//", "/");
    } else {
      result = filename.replaceAll("//", "/");
    }
    return result;
  }

  /**
   * converts 2 backslashes into one slash.
   * 
   * @param path
   *          to convert
   * @return string
   */
  public static String convertSlashes(final String path) {
    return path.replaceAll("\\\\", "/");
  }

  /**
   * converts one slash into 2 backslashes.
   * 
   * @param path
   *          to convert
   * @return string
   */
  public static String convertBackSlashes(final String path) {
    return path.replaceAll("/", "\\\\");
  }

  /**
   * converts only if backslash is not proceeded by a blank.
   * 
   * @param aPath
   *          to convert
   * @return string
   */
  public static String convertBackSlashesNoParams(final String aPath) {
    String path = aPath;
    int slash = -1;
    while ((slash = path.indexOf("/", slash + 1)) > -1) {
      if (slash > -1 && path.charAt(slash - 1) != ' ') {
        String sLineStart = path.substring(0, slash);
        String sLineEnd = path.substring(slash + 1);
        path = sLineStart + "\\" + sLineEnd;
      }
    }
    return path;
  }

  /**
   * converting the given path with an File.separator at the end.
   * 
   * @param aPath
   *          the path to convert
   * @return the converted path
   */
  public static String normPath(final String aPath) {
    if (!aPath.endsWith("/") && !aPath.endsWith("\\")) {
      return aPath + File.separator;
    } else {
      return aPath;
    }
  }
}
