/*
 * MCS Media Computer Software Copyright (c) 2007 by MCS
 * -------------------------------------- Created on 15.01.2007 by w.klaas
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * 
 */
package de.mcs.utils;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Properties;

/**
 * This class extands the normal properties class with functions to directly get
 * simple types.
 * 
 * @author w.klaas
 * 
 */
public class ExtendedProperties extends Properties {

    /**
     * Constructor for directly loaing properties from file.
     * 
     * @param file
     *            the file to load properties from.
     * @throws IOException
     *             if something goes wrong.
     */
    public ExtendedProperties(final File file) throws IOException {
        super();
        load(new FileInputStream(file));
    }

    /**
     * Simple Constructor.
     */
    public ExtendedProperties() {
        super();
    }

    /**
     * directly getting boolean value.
     * 
     * @param key
     *            the property to get.
     * @return the desired boolean value.
     */
    public final boolean getBool(final String key) {
        return Boolean.parseBoolean(getProperty(key));
    }

    /**
     * directly getting boolean value.
     * 
     * @param key
     *            the property to get.
     * @param defaultValue
     *            the default value to get, if the property is not present.
     * @return the desired boolean value.
     */
    public final boolean getBool(final String key, final boolean defaultValue) {
        return Boolean.parseBoolean(getProperty(key, Boolean.toString(defaultValue)));
    }

    /**
     * directly getting integer value.
     * 
     * @param key
     *            the property to get.
     * @return the desired int value.
     */
    public final int getInt(final String key) {
        return NumberHelper.parseInt(getProperty(key));
    }

    /**
     * directly getting integer value.
     * 
     * @param key
     *            the property to get.
     * @param defaultValue
     *            the default value to get, if the property is not present.
     * @return the desired int value.
     */
    public final int getInt(final String key, final int defaultValue) {
        return NumberHelper.parseInt(getProperty(key, Integer.toString(defaultValue)));
    }

    /**
     * directly getting integer value.
     * 
     * @param key
     *            the property to get.
     * @param defaultValue
     *            the default value to get, if the property is not present.
     * @return the desired int value.
     */
    public final int getInt(final String key, final String defaultValue) {
        return NumberHelper.parseInt(getProperty(key, defaultValue));
    }

    /**
     * directly getting long value.
     * 
     * @param key
     *            the property to get.
     * @return the desired long value.
     */
    public final long getLong(final String key) {
        return Long.parseLong(getProperty(key));
    }

    /**
     * directly getting long value.
     * 
     * @param key
     *            the property to get.
     * @param defaultValue
     *            the default value to get, if the property is not present.
     * @return the desired long value.
     */
    public final long getLong(final String key, final long defaultValue) {
        return Long.parseLong(getProperty(key, Long.toString(defaultValue)));
    }

    /**
     * directly getting a file value.
     * 
     * @param key
     *            the property to get.
     * @return the desired file.
     */
    public final File getFile(final String key) {
        if (containsKey(key)) {
            return new File(getProperty(key));
        } else {
            return null;
        }
    }

    /**
     * directly getting a file value.
     * 
     * @param key
     *            the property to get.
     * @param defaultValue
     *            the default value to get, if the property is not present.
     * @return the desired file.
     */
    public final File getFile(final String key, final File defaultValue) {
        return new File(getProperty(key, defaultValue.getAbsolutePath()));
    }

    /**
     * directly getting a string array value.
     * 
     * @param key
     *            the property to get.
     * @return the desired string array.
     */
    public final String[] getStrings(final String key) {
        String value = getProperty(key);
        if (value != null) {
            return StringUtils.csvStringToArray(value, ',', '"');
        }
        return null;
    }

    /**
     * directly getting a string array value.
     * 
     * @param key
     *            the property to get.
     * @param defaultValue
     *            the default value to get, if the property is not present.
     * @return the desired string array.
     */
    public final String[] getStrings(final String key, final String[] defaultValue) {
        String value = getProperty(key);
        if (value != null) {
            return StringUtils.csvStringToArray(value, ',', '"');
        }
        return defaultValue;
    }

    /**
     * directly getting integer value.
     * 
     * @param key
     *            the property to get.
     * @param defaultValue
     *            the default value to get, if the property is not present.
     * @return the desired int value.
     */
    public final long getMSec(final String key, final String defaultValue) {
        return NumberHelper.parseTime(getProperty(key, defaultValue));
    }
}
